/*
 * Copyright (C) 2013 Martin Cerny
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package cz.cuni.amis.experiments.impl;

import cz.cuni.amis.experiments.ExperimentException;
import cz.cuni.amis.experiments.ILoggingHeaders;
import cz.cuni.amis.experiments.ILoggingOutput;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.List;
import java.util.Queue;

/**
 * 
 * @author Martin Cerny
 */
public abstract class AbstractLoggingOutput extends AbstractBareLoggingOutput implements ILoggingOutput{

    private final Object logQueueMutex = new Object();
    private final Object writeMutex = new Object();
    
    protected ILoggingHeaders headers;
       
    protected Queue<List<Object>> logQueue;
    
    private LoggingThread loggingThread;
    
    protected IOException iOException = null;
    
    @Override
    public void logData(List<Object> data) {        
        if(data.size() != headers.getColumnCount()){
            throw new ExperimentException("Tried to log " + data.size() + " columns, but the logger only has " + headers.getColumnCount() + " columns");                    
        }
        synchronized(logQueueMutex){
            if(iOException != null){
                throw new ExperimentException("Previous logging operation threw exception. The log is no longer usable.", iOException);
            }
            logQueue.add(data);
            logQueueMutex.notify();
        }
    }


    @Override
    public void init(ILoggingHeaders headers) throws IOException {
        this.headers = headers;
        logQueue = new ArrayDeque<List<Object>>();
        loggingThread = new LoggingThread();
        loggingThread.start();
    }

    @Override
    public final void close() throws IOException {
        loggingThread.cancel();
        synchronized(writeMutex){
            closeInternal();
        }
    }

    @Override
    public final void flush() throws IOException {
        synchronized(writeMutex){
            flushInternal();
        }
    }
    
    
    
    /**
     * Implementations override this to log a single row of data. Note that
     * this method may block and does not need to be thread-safe.
     * @param data 
     */
    protected abstract void logDataInternal(List<Object> data) throws IOException;

    /**
     * Implementations override this to close the logger. The implementation
     * does not need to be thread safe.
     */
    protected abstract void closeInternal() throws IOException;
    
    /**
     * Implementations override this to flush the logger. The implementation
     * does not need to be thread safe.
     */
    protected abstract void flushInternal() throws IOException;
    
    
    
    private class LoggingThread extends Thread {

        boolean cancelled = false;

        public LoggingThread() {
            super("Logging: " + AbstractLoggingOutput.this.toString());
        }
        
        
        @Override
        public void run() {
            while(!cancelled){
                synchronized(logQueueMutex){
                    while(logQueue.isEmpty()){
                        try {
                            logQueueMutex.wait();
                        } catch(InterruptedException ex){
                            if(cancelled){
                                return;
                            }
                        }
                    }
                    try {
                        synchronized(writeMutex){
                            logDataInternal(logQueue.poll());
                        }
                    } catch (IOException ex){
                        iOException = ex;
                        return;
                    }
                }
            }
        }
        
        
        public void cancel(){
            cancelled = true;
            interrupt();
        }
    }
    
}
