/*
 * Copyright (C) 2011 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.experiments.impl;

import cz.cuni.amis.experiments.*;
import java.util.Arrays;
import java.util.List;
import org.apache.log4j.Logger;

/**
 * Helper class for implementing IExperimentSuiteRunner interface
 * @author Martin Cerny
 */
public abstract class AbstractExperimentSuiteRunner<EXPERIMENT_TYPE extends IExperiment> implements IExperimentSuiteRunner<EXPERIMENT_TYPE> {
    
    private final Logger logger = Logger.getLogger(AbstractExperimentSuiteRunner.class);
    
    protected ILogCentral logCentral;

    public AbstractExperimentSuiteRunner(ILogCentral logCentral) {
        this.logCentral = logCentral;
    }   
    
    
    @Override
    public void runExperimentSuite(IExperimentSuite<EXPERIMENT_TYPE> suite, int startExperiment, int stopAfter, IExperimentFilter<EXPERIMENT_TYPE> filter) {

        for(IExperimentRunner<EXPERIMENT_TYPE> runner : getAllRunners()){
            runner.init();
        }
        runExperimentSuiteInternal(suite, startExperiment, stopAfter, filter);        

        for(IExperimentRunner<EXPERIMENT_TYPE> runner : getAllRunners()){
            runner.shutdown();
        }
    }

    protected abstract List<IExperimentRunner<EXPERIMENT_TYPE>> getAllRunners();
    
    protected abstract void runExperimentSuiteInternal(IExperimentSuite<EXPERIMENT_TYPE> suite, int startExperiment, int stopAfter, IExperimentFilter<EXPERIMENT_TYPE> filter);
    

    /**
     * Implementation should use this method to run a single experiment
     * @param experiment 
     */
    protected void runSingleExperiment(IExperimentRunner<EXPERIMENT_TYPE> runner, EXPERIMENT_TYPE experiment, ILogCentral logCentral) {        
        try {
            runner.runExperiment(experiment, logCentral);                        
        } catch(Exception ex){
            logger.error("Exception during experiment runtime", ex);                                
        } finally {
            logCentral.flush();
        }
    }
    
    
    
    @Override
    public final void runExperimentSuite(IExperimentSuite<EXPERIMENT_TYPE > suite) {
        runExperimentSuite(suite, 0);
    }

    @Override
    public final void runExperimentSuite(IExperimentSuite<EXPERIMENT_TYPE > suite, int startExperiment) {
        runExperimentSuite(suite, startExperiment, null);
    }

    @Override
    public final void runExperimentSuite(IExperimentSuite<EXPERIMENT_TYPE > suite, IExperimentFilter<EXPERIMENT_TYPE > filter) {
        runExperimentSuite(suite, 0, filter);
    }

    @Override
    public final void runExperimentSuite(IExperimentSuite<EXPERIMENT_TYPE > suite, int startExperiment, int stopAfter) {
        runExperimentSuite(suite, startExperiment, stopAfter, null);
    }

    @Override
    public final void runExperimentSuite(IExperimentSuite<EXPERIMENT_TYPE > suite, int startExperiment, IExperimentFilter<EXPERIMENT_TYPE> filter) {
        runExperimentSuite(suite, startExperiment, 0, filter);
    }
    
    
    
    
    
}
