package nl.tudelft.pogamut.ut2004.agent.module.shooting;

import java.util.HashMap;
import java.util.Map;

import nl.tudelft.pogamut.ut2004.agent.module.shooting.weapon.GenericWeaponShooting;
import cz.cuni.amis.pogamut.base.agent.module.SensorModule;
import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensomotoric.Weapon;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensomotoric.Weaponry;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.WeaponPref;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.WeaponPrefs;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.UT2004Navigation;
import cz.cuni.amis.pogamut.ut2004.bot.command.ImprovedShooting;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.communication.messages.ItemType;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.EndMessage;
import cz.cuni.amis.utils.SafeEquals;

/**
 * <p>
 * Module that handles shooting other players in an intelligent manner.
 * </p>
 * 
 * <p>
 * Will utilize {@link WeaponPrefs} to determine which weapon to use for the
 * target its {@link WeaponShooting} modules to effectively use the weapon.
 * </p>
 * 
 * <p>
 * The targets for shooting can be set on the logic thread, but the actual
 * shooting is evaluated by listening to the {@link EndMessage}.
 * </p>
 * 
 * <p>
 * This module obeys the restrictions set by
 * {@link ImprovedShooting#setChangeWeaponCooldown(long)} and uses
 * {@link WeaponPrefs#getWeaponPreference(ILocated)} to determine which weapon
 * to use.
 * </p>
 * 
 * <p>
 * The {@link UT2004Navigation#setFocus(ILocated)} is used to indicate where the
 * bot wants to look. It should be honored for best results.
 * </p>
 * 
 * @author mpkorstanje
 * 
 */
@SuppressWarnings("rawtypes")
public class WeaponryShooting extends SensorModule<UT2004Bot> {

	/**
	 * {@link EndMessage} listener.
	 */
	private class EndMessageListener implements IWorldEventListener<EndMessage> {
		/**
		 * Constructor. Registers itself on the given WorldView object.
		 * 
		 * @param worldView
		 *            WorldView object to listen to.
		 */
		public EndMessageListener(IWorldView worldView) {
			worldView.addEventListener(EndMessage.class, this);
		}

		@Override
		public void notify(EndMessage event) {
			shoot();
		}
	}

	/**
	 * Current target. Null if none.
	 */
	protected ILocated currentTarget = null;

	/**
	 * Current weapon pref to use.
	 */
	protected WeaponPref currentWeaponPref = null;
	/**
	 * Currently active shooting.
	 */
	protected WeaponShooting currentWeaponShooting = null;
	/** {@link EndMessage} listener */
	@SuppressWarnings("unused")
	private EndMessageListener endMessageListener;

	/**
	 * Focus provider, use to set where you want to look.
	 */
	protected FocusProvider focus = new FocusProvider();
	/**
	 * Info about the agent.
	 */
	protected AgentInfo info;

	/**
	 * Previous target.
	 */
	protected ILocated lastTarget = null;

	/**
	 * Weapon pref used in the previous evaluation of
	 * {@link WeaponryShooting#shoot()}.
	 */
	protected WeaponPref lastWeaponPref = null;
	/**
	 * Shooting active in the previous evaluation of
	 * {@link WeaponryShooting#shoot()}.
	 */
	protected WeaponShooting lastWeaponShooting = null;

	/**
	 * Reference to shooting module to shoot stuff.
	 */
	protected ImprovedShooting shoot;
	/**
	 * Reference to {@link WeaponPrefs}
	 */
	protected WeaponPrefs weaponPrefs;

	/**
	 * Reference to all the weapons we have.
	 */
	protected Weaponry weaponry;
	/**
	 * Map of weapon shooting and associated weapons.
	 */
	protected Map<ItemType, WeaponShooting> weaponShootings = new HashMap<ItemType, WeaponShooting>();

	/**
	 * Creates a new WeaponryShooting module.
	 * 
	 * @param bot
	 * @param info
	 * @param weaponry
	 * @param weaponPrefs
	 * @param shoot
	 */
	public WeaponryShooting(UT2004Bot bot, AgentInfo info, Weaponry weaponry, WeaponPrefs weaponPrefs,
			ImprovedShooting shoot) {
		super(bot);
		this.info = info;
		this.weaponry = weaponry;
		this.weaponPrefs = weaponPrefs;
		this.shoot = shoot;

		endMessageListener = new EndMessageListener(worldView);

	}

	/**
	 * Adds a new WeaponShooting. The module will be used as soon as the bot
	 * changes weapons to the weapon associated with the shooting.
	 * 
	 * @param weaponShooting
	 * @return the previous {@link WeaponShooting} for the associated weapon.
	 */
	public WeaponShooting addWeaponShooting(WeaponShooting weaponShooting) {
		return weaponShootings.put(weaponShooting.getWeaponType(), weaponShooting);
	}
	
	/**
	 * Removes the WeaponShooting for the given weapon.
	 * 
	 * @param weapon
	 *            the weapon for which the WeaponShooting should be removed.
	 * @return the removed WeaponShooting or null if none.
	 * 
	 */
	public WeaponShooting removeWeaponShooting(ItemType weapon) {
		return weaponShootings.remove(weapon);
	}

	/**
	 * @return location bot wants to look.
	 */

	public IFocus getFocus() {
		return focus;
	}

	/**
	 * @return the previous target.
	 */
	public ILocated getLastTarget() {
		return lastTarget;

	}

	/**
	 * @return the target
	 */
	public ILocated getTarget() {
		return currentTarget;
	}

	/**
	 * @return the current weaponPref
	 */
	public WeaponPref getWeaponPref() {
		return currentWeaponPref;
	}

	/**
	 * @return the currently active shooting.
	 */
	public WeaponShooting getWeaponShooting() {
		return currentWeaponShooting;
	}



	/**
	 * Selects a WeaponShooting apropriate for the type of the Weapon.
	 * 
	 * @param weapon
	 * @return
	 */
	protected WeaponShooting selectWeaponShooting(Weapon weapon) {
		if (weapon == null) {
			return null;
		}

		if (!weaponShootings.containsKey(weapon.getType())) {
			GenericWeaponShooting genericWeaponShooting = new GenericWeaponShooting(agent, info, shoot, weaponry,
					weapon.getType());
			weaponShootings.put(weapon.getType(), genericWeaponShooting);
			return genericWeaponShooting;
		}

		return weaponShootings.get(weapon.getType());
	}

	/**
	 * The actual shooting happens here. Called by {@link EndMessageListener}.
	 */
	protected void shoot() {
		lastWeaponShooting = currentWeaponShooting;

		WeaponPref suggested = weaponPrefs.getWeaponPreference(currentTarget);

		if (shoot.mayChangeWeapon()) {
			lastWeaponPref = currentWeaponPref;
			currentWeaponPref = suggested;
		}

		// The shooting module is selected based on the weapon preference.
		// Should none be available, the current weapon will be used to select
		// the shooting.
		if (currentWeaponPref != null) {
			currentWeaponShooting = selectWeaponShooting(weaponry.getWeapon(currentWeaponPref.getWeapon()));
		} else {
			currentWeaponShooting = selectWeaponShooting(weaponry.getCurrentWeapon());
			currentWeaponPref = new WeaponPref(weaponry.getCurrentWeapon().getType());
		}

		// New shooting, stop the old one.
		if (lastWeaponShooting != null && lastWeaponShooting != currentWeaponShooting) {
			lastWeaponShooting.stopShoot();
			focus.clearFocus();
		}
		// Unable to make selection.
		if (currentWeaponShooting == null) {
			return;
		}
		// New shooting, start new one.
		if (lastWeaponShooting != currentWeaponShooting) {
			currentWeaponShooting.shoot(currentWeaponPref, currentTarget);
			focus.setFocus(currentWeaponShooting.getFocus());
		}
		// Target or preference changed. Update.
		else if (!currentWeaponPref.equals(lastWeaponPref) || !SafeEquals.equals(lastTarget, currentTarget)) {
			currentWeaponShooting.shoot(currentWeaponPref, currentTarget);
		}

	}

	/**
	 * Sets the target to be shot at.
	 * 
	 * @param target
	 */
	public void shoot(ILocated target) {
		if (this.currentTarget != target) {
			this.lastTarget = this.currentTarget;
			this.currentTarget = target;
		}
	}

	/**
	 * Clears the target and stops the shooting.
	 * 
	 */
	public void stopShoot() {
		if (this.currentTarget != null) {
			this.lastTarget = this.currentTarget;
			this.currentTarget = null;
		}
		this.currentWeaponShooting.stopShoot();
	}
}
