package nl.tudelft.pogamut.ut2004.agent.module.shooting.weapon;

import nl.tudelft.pogamut.ut2004.agent.module.sensor.Projectiles;
import nl.tudelft.pogamut.ut2004.agent.module.shooting.AbstractWeaponShooting;
import nl.tudelft.pogamut.ut2004.agent.module.shooting.util.FacingUtil;

import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensomotoric.Weaponry;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.Senses;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.WeaponPref;
import cz.cuni.amis.pogamut.ut2004.bot.command.ImprovedShooting;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.communication.messages.ItemType;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.IncomingProjectile;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;

/**
 * <p>
 * Module to work efficiently with the shield gun.
 * </p>
 * 
 * <p>
 * When the primary mode is given to the weapon preferences the shield gun will
 * simply charge up. It's up to other component to run into someone.
 * </p>
 * <p>
 * When the secondary mode is requested by weapon preferences the shield gun
 * activate for 1 second when ever it detects that it is taking damage, sees an
 * incoming projectile or sees it's target shooting or sees it's target aiming a
 * hitscan weapon at it. While doing this the bot will aim in the direction of
 * its target, potentially bouncing shock cores and link gun projectiles to it's
 * target.
 * </p>
 * 
 * @author mpkorstanje
 * 
 */
public class ShieldGunShooting extends AbstractWeaponShooting {

	/**
	 * Distance at which incoming projectiles can be considered threatening.
	 */
	protected static final int INCOMMING_PROJECTILE_MIN_THREAT_DISTANCE = 750;

	/**
	 * Duration in second that the shield gun will fire.
	 */
	protected static final double SHIELD_GUN_SEC_BURST_DURATION = 1.0;

	/**
	 * Default is the secondary mode.
	 */
	protected static final WeaponPref DEFAULT_WEAPON_PREF = new WeaponPref(ItemType.SHIELD_GUN, false);


	/**
	 * {@link Projectiles} modules.
	 */
	protected Projectiles projectiles;
	/**
	 * Senses module. Note: shared with the main again. Careful with "Once"
	 * functions.
	 */
	protected Senses senses;

	public ShieldGunShooting(UT2004Bot<?, ?, ?> bot, AgentInfo info, ImprovedShooting shoot, Weaponry weaponry,
			Projectiles projectiles, Senses senses) {
		super(bot, info, shoot, weaponry);
		this.projectiles = projectiles;
		this.senses = senses;
	}

	@Override
	protected void shoot() {

		// Wrong weapon, wait up.
		if (!isWeaponReady()) {
			return;
		}

		// Reset focus
		focus.setFocus(target);

		// Primary works by running into people, just keep it charged.
		if (weaponPref.isPrimary()) {
			shootPrimary();
		}
		// Secondary should only be used when directly shot at.
		else {
			shootSecondary();
		}

	}

	protected void shootPrimary() {
		if (target == null) {
			shoot.shoot();
			return;
		}

		shoot.shootPrimary(target);

	}

	protected void shootSecondary() {

		// Incomming!
		IncomingProjectile projectile = projectiles.getNearestProjectile(agent,
				INCOMMING_PROJECTILE_MIN_THREAT_DISTANCE);
		// System.out.println(projectile);
		if (projectile != null) {
			ILocated safeTarget = (ILocated) (target == null ? projectile : target);
			shoot.shootSecondaryCharged(safeTarget, SHIELD_GUN_SEC_BURST_DURATION);
			focus.setFocus(safeTarget);
			return;
		}

		// Oh, no!
		if (senses.isShot()) {
			ILocated safeTarget = (ILocated) (target == null ? agent : target);
			shoot.shootSecondaryCharged(safeTarget, SHIELD_GUN_SEC_BURST_DURATION);
			return;
		}

		// No further reason to consider shooting.
		if (!hasTarget()) {
			return;
		}

		// Bit pointless.
		if (!(target instanceof Player)) {
			shoot.shootSecondaryCharged(target, SHIELD_GUN_SEC_BURST_DURATION);
			return;
		}

		// Dangerous people about
		Player player = (Player) target;
		boolean hitScanWeapon = player.getWeapon().contains("Shock") || player.getWeapon().contains("Lightning")
				|| player.getWeapon().contains("Sniper");
		boolean fireDangerous = player.getFiring() > 0 || hitScanWeapon;
		if (player.isVisible() && fireDangerous && FacingUtil.isFacing(player, agent, FACING_ANGLE)) {
			shoot.shootSecondaryCharged(target, SHIELD_GUN_SEC_BURST_DURATION);
			return;
		}

	}

	@Override
	protected WeaponPref getDefaultWeaponPref() {
		return DEFAULT_WEAPON_PREF;
	}

}
