package cz.cuni.pogamut.shed.presenter;

import cz.cuni.amis.pogamut.sposh.elements.ILapChainListener;
import cz.cuni.amis.pogamut.sposh.elements.INamedElement;
import cz.cuni.amis.pogamut.sposh.elements.LapChain;
import cz.cuni.amis.pogamut.sposh.elements.PoshElement;
import cz.cuni.amis.pogamut.sposh.elements.PoshElementListener;
import cz.cuni.amis.pogamut.sposh.elements.Result;
import cz.cuni.amis.pogamut.sposh.elements.Sense;
import cz.cuni.amis.pogamut.sposh.elements.TriggeredAction;
import cz.cuni.amis.pogamut.sposh.engine.VariableContext;
import cz.cuni.amis.pogamut.sposh.exceptions.UnexpectedElementException;
import cz.cuni.pogamut.shed.widget.ShedScene;
import cz.cuni.pogamut.shed.widget.ShedVariableWidget;
import java.util.LinkedList;
import java.util.List;

/**
 * Base class presenter for primitives, i.e. actions and senses. Since both
 * actions and senses presenters are quite similar, use this class as base class
 * for both of them.
 *
 * @param <PRIMITIVE_TYPE> Either {@link Sense} or {@link TriggeredAction}
 * depending on which primitive is used.
 * @author Honza
 */
public abstract class PrimitivePresenter<PRIMITIVE_TYPE extends PoshElement & INamedElement> extends AbstractPresenter implements PoshElementListener<PRIMITIVE_TYPE>, INameMapListener, ILapChainListener {

    /**
     * The primitive that is being presented by this presenter.
     */
    protected final PRIMITIVE_TYPE primitive;
    /**
     * Widget that represents the primitive in the scene.
     */
    protected final ShedVariableWidget primitiveWidget;
    /**
     * Chain of variables from the root to the primitive.
     */
    protected final LapChain primitiveChain;

    /**
     * Create new presenter for @primitive, detect and project all changes on
     * the @primitive and @primitiveChain to the @primitiveWidget.
     *
     * @param scene Scene that will be manipulated by this presenter
     * @param presenter The basic presenter
     * @param primitive The primitiver that is being presented.
     * @param primitiveWidget Widget that is representing the @primitive in the
     * @scene.
     * @param primitiveChain Chain of variables from the root to the @primitive
     */
    protected PrimitivePresenter(ShedScene scene, ShedPresenter presenter, PRIMITIVE_TYPE primitive, ShedVariableWidget primitiveWidget, LapChain primitiveChain) {
        super(scene, presenter);
        this.primitive = primitive;
        this.primitiveWidget = primitiveWidget;
        this.primitiveChain = primitiveChain;
    }

    @Override
    public void register() {
        primitiveWidget.setPresenter(this);
        primitive.addElementListener(this);

        presenter.addNameMapListener(this);

        primitiveChain.register();
        primitiveChain.addChainListener(this);

        updateWidget();
    }

    @Override
    public void unregister() {
        primitiveChain.removeChainListener(this);
        primitiveChain.unregister();

        presenter.removeNameMapListener(this);

        primitive.removeElementListener(this);
        primitiveWidget.setPresenter(null);
    }

    @Override
    public final void childElementAdded(PRIMITIVE_TYPE parent, PoshElement child) {
        throw UnexpectedElementException.create(child);
    }

    @Override
    public final void childElementMoved(PRIMITIVE_TYPE parent, PoshElement child, int oldIndex, int newIndex) {
        throw UnexpectedElementException.create(child);
    }

    @Override
    public final void childElementRemoved(PRIMITIVE_TYPE parent, PoshElement child, int removedChildIndex) {
        throw UnexpectedElementException.create(child);
    }

    @Override
    public final void nameMapChanged(String key, String oldName, String newName) {
        if (key.equals(primitive.getName())) {
            updateWidget();
        }
    }

    /**
     * When some link of the @primitiveChain is changed, this method is notified
     * and it will update the widget.
     */
    @Override
    public final void notifyLinkChanged() {
        updateWidget();
        scene.update();
    }

    /**
     * Method used by the {@link #updateWidget() } to 
     * @return 
     */
    protected abstract String getTitleText();

    /**
     * Update widget to reflect current state of the {@link #action}.
     */
    protected final void updateWidget() {
        String titleText = getTitleText();
        primitiveWidget.setDisplayName(titleText);

        String[] params = presenter.getPrimitiveParameters(primitive.getName());
        VariableContext ctx = primitiveChain.createContext();

        List<String> presentArgs = getPresentArgs(ctx, params);
        primitiveWidget.setPresent(presentArgs);

        List<String> missingArgs = getMissingArgs(ctx, params);
        primitiveWidget.setMissing(missingArgs);

        List<String> unusedArgs = getUnusedArgs(ctx, params);
        primitiveWidget.setUnused(unusedArgs);

        primitiveWidget.revalidate();
    }

    /**
     * Return all params that are used in the
     *
     * @param primitiveParams The parameters that are requested to exist within
     * the chain.
     * @return logical conjuction of @primitiveParams and chain variables.
     */
    private List<String> getPresentArgs(VariableContext ctx, String[] primitiveParams) {
        List<String> presentArgs = new LinkedList<String>();

        for (String primitiveParam : primitiveParams) {
            if (ctx.hasVariable(primitiveParam)) {
                Object argumentValue = ctx.getValue(primitiveParam);
                String argumentRepresentation = primitiveParam + '=' + Result.toLap(argumentValue);

                presentArgs.add(argumentRepresentation);
            }
        }
        return presentArgs;
    }

    private List<String> getMissingArgs(VariableContext ctx, String[] primitiveParams) {
        List<String> missingArgs = new LinkedList<String>();

        for (String primitiveParam : primitiveParams) {
            if (!ctx.hasVariable(primitiveParam)) {
                missingArgs.add(primitiveParam);
            }
        }
        return missingArgs;
    }

    private List<String> getUnusedArgs(VariableContext ctx, String[] primitiveArgs) {
        List<String> unusedArgs = new LinkedList<String>();
        String[] allKeys = ctx.getKeys();
        for (String key : allKeys) {
            boolean isVarNeeded = false;
            for (String presentArg : primitiveArgs) {
                if (presentArg.equals(key)) {
                    isVarNeeded = true;
                }
            }
            if (!isVarNeeded) {
                Object variableValue = ctx.getValue(key);
                String variableRepresentation = key + '=' + Result.toLap(variableValue);
                unusedArgs.add(variableRepresentation);
            }
        }
        return unusedArgs;
    }
}
