package cz.cuni.pogamut.shed.presenter;

import cz.cuni.amis.pogamut.sposh.elements.ActionPattern;
import cz.cuni.amis.pogamut.sposh.elements.Arguments;
import cz.cuni.amis.pogamut.sposh.elements.Arguments.Argument;
import cz.cuni.amis.pogamut.sposh.elements.FormalParameters;
import cz.cuni.amis.pogamut.sposh.elements.FormalParameters.Parameter;
import cz.cuni.amis.pogamut.sposh.elements.ILapChainListener;
import cz.cuni.amis.pogamut.sposh.elements.LapChain;
import cz.cuni.amis.pogamut.sposh.elements.PoshElement;
import cz.cuni.amis.pogamut.sposh.elements.PoshElementListener;
import cz.cuni.amis.pogamut.sposh.elements.Result;
import cz.cuni.amis.pogamut.sposh.elements.TriggeredAction;
import cz.cuni.amis.pogamut.sposh.engine.VariableContext;
import cz.cuni.pogamut.shed.widget.ShedScene;
import cz.cuni.pogamut.shed.widget.ShedVariableWidget;
import java.beans.PropertyChangeEvent;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import javax.swing.Action;
import org.netbeans.api.visual.action.ActionFactory;
import org.netbeans.api.visual.action.WidgetAction;

/**
 * {@link IPresenter} for widget representing the {@link ActionPattern}. This
 * presenter updates only properties, no children ect.
 *
 * @see CompetencePresenter
 * @see DrivePresenter
 * @author Honza H.
 */
final class ActionPatternPresenter extends AbstractPresenter implements IPresenter, PoshElementListener<ActionPattern>, ILapChainListener {

    private final TriggeredAction referencingAction;
    /**
     * The pattern this presenter is for.
     */
    private final ActionPattern actionPattern;
    private final ShedVariableWidget actionPatternWidget;
    private final LapChain chain;

    /**
     * Create presenter for the AP widget, only the widget, not its children
     * ect.
     *
     * @param scene Scene of the widget.
     * @param presenter Main presenter
     * @param referencingAction The action that references this AP expansion.
     * @param actionPattern The action pattern.
     * @param actionPatternWidget The widget that will represent the AP.
     * @param chain 
     */
    ActionPatternPresenter(ShedScene scene, ShedPresenter presenter, TriggeredAction referencingAction, ActionPattern actionPattern, ShedVariableWidget actionPatternWidget, LapChain chain) {
        super(scene, presenter);

        assert referencingAction.getName().equals(actionPattern.getName());

        this.referencingAction = referencingAction;
        this.actionPattern = actionPattern;
        this.actionPatternWidget = actionPatternWidget;
        this.chain = chain;

        updateWidget();
    }

    @Override
    public void register() {
        actionPatternWidget.setPresenter(this);
        actionPattern.addElementListener(this);
        referencingAction.addElementListener(this);

        chain.register();
        chain.addChainListener(this);
    }

    @Override
    public void unregister() {
        chain.removeChainListener(this);
        chain.unregister();

        referencingAction.removeElementListener(this);
        actionPattern.removeElementListener(this);
        actionPatternWidget.setPresenter(null);
    }

    @Override
    public Action[] getMenuActions() {
        PoshElement parentStructure = referencingAction.getParent();
        if (isActionPattern(parentStructure)) {
            ActionPattern parentAP = (ActionPattern) parentStructure;
            int thisAPPosition = getPosition(parentAP.getActions(), referencingAction);
            return new Action[]{
                        ShedMenuActionFactory.appendAction(actionPattern, 0),
                        ShedMenuActionFactory.changeActionPatternParameters(actionPattern),
                        ShedMenuActionFactory.deleteAction(parentAP, thisAPPosition)
                    };
        } else {
            return new Action[]{
                        ShedMenuActionFactory.appendAction(actionPattern, 0),
                        ShedMenuActionFactory.changeActionPatternParameters(actionPattern)
                    };
        }
    }

    @Override
    public void childElementAdded(ActionPattern parent, PoshElement child) {
        // Not your job
    }

    @Override
    public void childElementMoved(ActionPattern parent, PoshElement child, int oldIndex, int newIndex) {
        // Not your job
    }

    @Override
    public void childElementRemoved(ActionPattern parent, PoshElement child, int removedChildIndex) {
        // Not your job
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        String propertyName = evt.getPropertyName();

        if (propertyName.equals(TriggeredAction.taArgs)
                || propertyName.equals(TriggeredAction.taName)
                || propertyName.equals(ActionPattern.apName)
                || propertyName.equals(ActionPattern.apParams)) {
            updateWidget();
        } else {
            throw new IllegalArgumentException(propertyName);
        }
    }

    @Override
    public AbstractAcceptAction[] getAcceptProviders() {
        return new AbstractAcceptAction[]{
                    AcceptActionFactory.createCompetence2Action(referencingAction),
                    AcceptActionFactory.createActionPatternAction(referencingAction),
                    AcceptActionFactory.createAction2Action(referencingAction)
                };
    }

    /**
     * Update widget to reflect current state of the {@link #action}.
     */
    private void updateWidget() {
        // Careful, while renaming the AP, referencing action and C can have different names.

        StringBuilder displayText = new StringBuilder();
        displayText.append(referencingAction.getName());

        Arguments args = referencingAction.getArguments();
        FormalParameters params = actionPattern.getParameters();

        VariableContext ctx = chain.createContext();

        List<String> variablesRepresentation = getDefinedVariablesRepresentation(ctx, params, args);
        actionPatternWidget.setPresent(variablesRepresentation);

        actionPatternWidget.setMissing(Collections.<String>emptyList());

        List<String> argsRepresentation = getOtherVariables(ctx, params, args);
        actionPatternWidget.setUnused(argsRepresentation);

        actionPatternWidget.setDisplayName(referencingAction.getName());
        actionPatternWidget.revalidate();
    }

    @Override
    public void notifyLinkChanged() {
        updateWidget();
    }

    // XXX: Put it somewhere else(probably VariableContext), this way it is referenced from APPresenter and CPresenter.
    /**
     * Get representation (variableName = valueRepresentationFromContext) of 
     * each parameter and argument.
     * 
     * @param ctx Context is used to retrieve the values of params and args
     * @param params Parameters we are interested int
     * @param args Arguments we are interested in.
     * @return List of representations, first params, then args. No EOL at the end.
     */
    static List<String> getDefinedVariablesRepresentation(VariableContext ctx, FormalParameters params, Arguments args) {
        List<String> variablesRepresentation = new LinkedList<String>();

        for (Parameter param : params) {
            String paramName = param.getName();
            Object paramValue = ctx.getValue(paramName);
            String paramRepresentation = paramName + '=' + Result.toLap(paramValue);

            variablesRepresentation.add(paramRepresentation);
        }
        
        for (Argument arg : args) {
            String argName = arg.getParameterName();
            if (!params.containsVariable(argName)) {
                Object argValue = ctx.getValue(argName);
                String argRepresentation = argName + '=' + Result.toLap(argValue);
                variablesRepresentation.add(argRepresentation);
            }
        }
        return variablesRepresentation;
    }

    // XXX: Put it somewhere else(probably VariableContext), this way it is referenced from APPresenter and CPresenter.
    static List<String> getOtherVariables(VariableContext ctx, FormalParameters params, Arguments args) {
        List<String> otherVariables = new LinkedList<String>();
        String[] allVariables = ctx.getKeys();
        
        for (String variableName : allVariables) {
            boolean nameIsUsed = false;
            for (Argument arg : args) {
                if (arg.getParameterName().equals(variableName)) {
                    nameIsUsed = true;
                }
            }
            
            if (params.containsVariable(variableName)) {
                nameIsUsed = true;
            }
            
            if (!nameIsUsed) {
                Object variableValue = ctx.getValue(variableName);
                String variableRepresentation = variableName + '=' + Result.toLap(variableValue);
                otherVariables.add(variableRepresentation);
            }
        }
        return otherVariables;
    }

    @Override
    public WidgetAction getEditAction() {
        return ActionFactory.createInplaceEditorAction(ShedInplaceEditorFactory.createActionPatternEditor(actionPattern, referencingAction));
    }

}
