package cz.cuni.pogamut.posh.explorer;

import cz.cuni.amis.pogamut.sposh.elements.ActionPattern;
import cz.cuni.amis.pogamut.sposh.elements.Competence;
import cz.cuni.amis.pogamut.sposh.elements.PoshPlan;
import javax.swing.JComponent;
import org.netbeans.api.project.Project;

/**
 * Factory for creating the crawler explorers. Explorers are GUI elements that 
 * display crawled items.
 *
 * @author Honza
 */
public final class ExplorerFactory {

    /**
     * Create explorer for actions (classes implementing IAction) on classpath
     * of the project. The crawler for actions is automatically started, no need
     * to start it.
     *
     * @param project Project whose classpath is searched for actions.
     * @param listeners Listeners to be registered with crawler, they are
     * automatically removed, once crawling is finished.
     * @return Created explorer.
     */
    public static JComponent createActionsExplorer(Project project, CrawlerListener<PrimitiveData>... listeners) {
        ClassCrawler crawler = new IActionCrawler(project);
        Explorer<PrimitiveData> explorer = new ActionExplorer(crawler);

        crawler.addListener(explorer);
        for (CrawlerListener<PrimitiveData> listener : listeners) {
            crawler.addListener(listener);
        }
        crawler.crawl();
        // Note: once crawling is finished, explorer will remove itself as listener of crawler.
        return explorer;
    }

    /**
     * Create explorer for sense (classes implementing ISense) on classpath
     * of the project. The crawler for sense is automatically started, no need
     * to start it.
     *
     * @param project Project whose classpath is searched for sense.
     * @param listeners Listeners to be registered with crawler, they are
     * automatically removed, once crawling is finished.
     * @return Created explorer.
     */
    public static JComponent createSensesExplorer(Project project, CrawlerListener<PrimitiveData>... listeners) {
        ClassCrawler crawler = new ISenseCrawler(project);
        Explorer<PrimitiveData> explorer = new SenseExplorer(crawler);

        crawler.addListener(explorer);
        for (CrawlerListener<PrimitiveData> listener : listeners) {
            crawler.addListener(listener);
        }
        crawler.crawl();
        // Note: once crawling is finished, explorer will remove itself as listener of crawler.
        return explorer;
    }

    /**
     * Create explorer for {@link Competence competences} in the @plan.
     *
     * @param plan Plan in which explorer looks for competences.
     * @return Created explorer
     */
    public static Explorer<Competence> createCompetenceExplorer(PoshPlan plan) {
        Crawler<Competence> crawler = CrawlerFactory.createCompetenceCrawler(plan);
        Explorer<Competence> explorer = new CompetenceExplorer(crawler, plan);

        crawler.addListener(explorer);
        crawler.crawl();
        return explorer;
    }

    /**
     * Create explorer for {@link ActionPattern action patterns} in the @plan.
     *
     * @param plan Plan in which explorer looks for action patterns.
     * @return Created explorer
     */
    public static Explorer<ActionPattern> createAPExplorer(PoshPlan lapTree) {
        Crawler<ActionPattern> crawler = CrawlerFactory.createAPCrawler(lapTree);
        Explorer<ActionPattern> explorer = new APExplorer(crawler, lapTree);

        crawler.addListener(explorer);
        crawler.crawl();
        return explorer;
    }
}
