package cz.cuni.pogamut.posh.explorer;

import cz.cuni.amis.pogamut.sposh.elements.*;
import cz.cuni.amis.pogamut.sposh.exceptions.DuplicateNameException;
import cz.cuni.amis.pogamut.sposh.exceptions.FubarException;
import cz.cuni.amis.pogamut.sposh.executor.IAction;
import cz.cuni.amis.pogamut.sposh.executor.ISense;
import cz.cuni.pogamut.posh.widget.accept.DataNodeExTransferable;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import org.netbeans.api.java.classpath.GlobalPathRegistry;
import org.openide.cookies.EditorCookie;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.util.Exceptions;
import org.openide.util.datatransfer.ExTransferable;

/**
 * {@link Explorer} for primitives, i.e. actions and senses.
 *
 * @author Honza
 */
abstract class Explorers extends Explorer<PrimitiveData> {

    protected Explorers(Crawler<PrimitiveData> crawler) {
        super(crawler);
    }

    @Override
    public boolean filter(String query, boolean caseSensitive, PrimitiveData item) {
        // TODO: Better query analysis.
        if (item.classFQN.contains(query)) {
            return false;
        }
        if (item.name != null && item.name.contains(query)) {
            return false;
        }
        for (String tag : item.tags) {
            if (tag.contains(query)) {
                return false;
            }
        }
        return true;
    }

    @Override
    protected String getRenderedLabel(PrimitiveData item) {
        return (item.name != null ? item.name : item.classFQN.replaceFirst("^.*\\.", "")) + "(" + item.classFQN + ")";
    }

    @Override
    protected String getItemDescription(PrimitiveData item) {
        StringBuilder info = new StringBuilder();
        info.append("<html>Class: ");
        info.append(item.classFQN);
        if (item.name != null) {
            info.append("<br/>Name: ");
            info.append(item.name);
        }
        if (item.description != null) {
            info.append("<br/>Description: ");
            info.append(item.description);
        }
        if (item.tags.length > 0) {
            info.append("<br/>Tags: ");
            for (int tagIndex = 0; tagIndex < item.tags.length; ++tagIndex) {
                info.append(item.tags[tagIndex]);
                if (tagIndex != item.tags.length - 1) {
                    info.append(',');
                    info.append(' ');
                }
            }
        }
        info.append("</html>");
        return info.toString();
    }

    @Override
    protected void displayItem(PrimitiveData item) {
        String javaFilePath = item.classFQN.replace('.', '/') + ".java";
        for (FileObject curRoot : GlobalPathRegistry.getDefault().getSourceRoots()) {
            FileObject fileObject = curRoot.getFileObject(javaFilePath);
            if (fileObject != null) {
                // do something, e.g. openEditor(fileObject, lineNumber);
                DataObject dobj = null;
                try {
                    dobj = DataObject.find(fileObject);
                } catch (DataObjectNotFoundException ex) {
                    Exceptions.printStackTrace(ex);
                }
                if (dobj != null) {
                    EditorCookie ec = (EditorCookie) dobj.getCookie(EditorCookie.class);
                    if (ec != null) {
                        ec.open();
                    }
                }
            }
        }
    }
    
    @Override
    protected void deleteItem(PrimitiveData item) {
        PGSupport.message("Primitives can't be deleted this way, you have to delete the class in the project and press refresh.");
    }
}


/**
 * Explorer class for actions. This class is a panel that shows actions (public
 * non-abstract classes implementing {@link IAction} interface).
 *
 * @author Honza
 */
class ActionExplorer extends Explorers {

    ActionExplorer(Crawler<PrimitiveData> crawler) {
        super(crawler);
    }

    @Override
    protected String getNewItemLabel() {
        return "New action (drag and drop)";
    }

    @Override
    protected Transferable createItemTransferable(PrimitiveData data) {
        if (data == null) {
            return null;
        }
        String actionName = data.classFQN;
        return new DataNodeExTransferable(LapElementsFactory.createAction(actionName));
    }

    @Override
    protected Transferable createNewItemTransferable() {
        return new ExTransferable.Single(TriggeredAction.dataFlavor) {

            @Override
            protected Object getData() {
                String actionName = PGSupport.getIdentifierFromDialog("Name of new action");
                if (actionName == null) {
                    return null;
                }
                return LapElementsFactory.createAction(actionName);
            }
        };
    }
}

/**
 * Explorer class for senses. This class is a panel that shows actions (public
 * non-abstract classes implementing {@link ISense} interface).
 *
 * @author Honza
 */
class SenseExplorer extends Explorers {

    SenseExplorer(Crawler<PrimitiveData> crawler) {
        super(crawler);
    }

    @Override
    protected String getNewItemLabel() {
        return "New sense (drag and drop)";
    }

    @Override
    protected Transferable createItemTransferable(PrimitiveData data) {
        if (data == null) {
            return null;
        }
        String senseName = data.classFQN;
        return new DataNodeExTransferable(new Sense(senseName));
    }

    @Override
    protected Transferable createNewItemTransferable() {
        return new ExTransferable.Single(Sense.dataFlavor) {

            @Override
            protected Object getData() {
                String id = PGSupport.getIdentifierFromDialog("Name of sense");
                if (id == null) {
                    return null;
                }
                return new Sense(id);
            }
        };
    }
}

/**
 * Explorer for {@link PoshElement elementes of posh} in the editor. Provides
 * search function, label of the element and transferable for DnD.
 *
 * @author Honza
 * @param <T> Type of element.
 */
abstract class ElementExplorer<T extends PoshElement & INamedElement> extends Explorer<T> {
    
    final PoshPlan plan;
    
    protected ElementExplorer(Crawler<T> crawler, PoshPlan plan) {
        super(crawler);
        this.plan = plan;
    }

    @Override
    protected boolean filter(String query, boolean caseSensitive, T item) {
        if (caseSensitive) {
            return !item.toString().contains(query);
        }
        return !item.toString().toLowerCase().contains(query.toLowerCase());
    }

    @Override
    protected String getRenderedLabel(T item) {
        return item.getName();
    }

    @Override
    protected Transferable createItemTransferable(T data) {
        return new DataNodeExTransferable(data);
    }

    @Override
    protected void displayItem(T item) {
        /**
         * Do nothing
         */
    }
}


class CompetenceExplorer extends ElementExplorer<Competence> {

    CompetenceExplorer(Crawler<Competence> crawler, PoshPlan plan) {
        super(crawler, plan);
    }

    @Override
    protected String getNewItemLabel() {
        return "New competence (drag and drop)";
    }

    @Override
    protected Transferable createNewItemTransferable() {
        return new ExTransferable.Single(Competence.dataFlavor) {

            @Override
            protected Object getData() throws IOException, UnsupportedFlavorException {
                String competenceName = PGSupport.getIdentifierFromDialog("Name of competence");
                if (competenceName == null) {
                    return null;
                }

                String elementName = PGSupport.getIdentifierFromDialog("Name of competence atom");
                if (elementName == null) {
                    return null;
                }
                try {
                    return LapElementsFactory.createCompetence(competenceName, elementName);
                } catch (DuplicateNameException ex) {
                    throw new FubarException("Creating new competence with only one name, what duplicate?", ex);
                }
            }
        };
    }

    @Override
    protected void deleteItem(Competence competence) {
        boolean confirmed = PGSupport.confirm("Are you sure you want to delete competence " + competence.getName());
        if (confirmed) {
            plan.removeCompetence(competence);
        }
    }

    @Override
    protected String getItemDescription(Competence competence) {
        return "<html>Competence: " + competence.getName() + "<br/><pre>" + competence.toString()+ "</pre></html>";
    }
}

class APExplorer extends ElementExplorer<ActionPattern> {

    public APExplorer(Crawler<ActionPattern> crawler, PoshPlan plan) {
        super(crawler, plan);
    }

    @Override
    protected String getNewItemLabel() {
        return "New action pattern (drag and drop)";
    }

    @Override
    protected Transferable createNewItemTransferable() {
        return new ExTransferable.Single(ActionPattern.dataFlavor) {

            @Override
            protected Object getData() throws IOException, UnsupportedFlavorException {
                String name = PGSupport.getIdentifierFromDialog("Name of new AP");
                if (name == null) {
                    return null;
                }

                return LapElementsFactory.createActionPattern(name);
            }
        };
    }

    @Override
    protected void deleteItem(ActionPattern actionPattern) {
        boolean confirmed = PGSupport.confirm("Are you sure you want to delete action pattern " + actionPattern.getName());
        if (confirmed) {
            plan.removeActionPattern(actionPattern);
        }
    }

    @Override
    protected String getItemDescription(ActionPattern ap) {
        return "<html>Action pattern: " + ap.getName() + "<br/><pre>" + ap.toString()+ "</pre></html>";
    }
}
