package cz.cuni.pogamut.shed.widget.editor;

import cz.cuni.amis.pogamut.sposh.elements.ActionPattern;
import cz.cuni.amis.pogamut.sposh.elements.Arguments;
import cz.cuni.amis.pogamut.sposh.elements.FormalParameters;
import cz.cuni.amis.pogamut.sposh.elements.TriggeredAction;
import cz.cuni.amis.pogamut.sposh.exceptions.CycleException;
import cz.cuni.amis.pogamut.sposh.exceptions.DuplicateNameException;
import cz.cuni.amis.pogamut.sposh.exceptions.InvalidNameException;
import cz.cuni.amis.pogamut.sposh.exceptions.MissingParameterException;
import java.awt.Rectangle;
import java.awt.event.KeyListener;
import java.text.MessageFormat;
import java.util.EnumSet;
import java.util.List;
import org.netbeans.api.visual.action.InplaceEditorProvider;
import org.netbeans.api.visual.widget.Widget;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.util.Exceptions;

/**
 * This class is responsible for handling the editor component for {@link ActionPattern}
 * and for taking the data from the user visble editor component and updating
 * correct structures.
 */
public final class APEditorProvider implements InplaceEditorProvider<APEditor> {

    private final ActionPattern actionPattern;
    private final TriggeredAction referencingAction;
    private KeyListener keyListener;

    public APEditorProvider(ActionPattern actionPattern, TriggeredAction referencingAction) {
        this.actionPattern = actionPattern;
        this.referencingAction = referencingAction;
    }

    @Override
    public void notifyOpened(InplaceEditorProvider.EditorController controller, Widget widget, APEditor editor) {
    }

    private void notify(String message) {
        NotifyDescriptor.Message infoMessage = new NotifyDescriptor.Message(message, NotifyDescriptor.INFORMATION_MESSAGE);
        DialogDisplayer.getDefault().notify(infoMessage);
    }

    @Override
    public void notifyClosing(InplaceEditorProvider.EditorController editorController, Widget widget, APEditor editor, boolean commit) {

        if (commit) {
            String oldAPName = actionPattern.getName();
            String newAPName = editor.getElementName();
            if (!oldAPName.equals(newAPName)) {
                try {
                    actionPattern.rename(newAPName);
                } catch (InvalidNameException ex) {
                    String errorMessage = MessageFormat.format("Name \"{0}\" is not valid", newAPName);
                    notify(errorMessage);
                } catch (CycleException ex) {
                    String errorMessage = MessageFormat.format("Changing name to \"{0}\" would cause a cycle.", newAPName);
                    notify(errorMessage);
                } catch (DuplicateNameException ex) {
                    String errorMessage = MessageFormat.format("Name \"{0}\" is already used.", newAPName);
                    notify(errorMessage);
                }
            }
            List<TableParameter> editorParams = editor.getParameters();
            FormalParameters newParams = new FormalParameters();
            for (TableParameter editorParam : editorParams) {
                String paramName = editorParam.getName();
                Object paramDefaultValue = editorParam.getDefaultValue();
                newParams.add(new FormalParameters.Parameter(paramName, paramDefaultValue));
            }
            try {
                actionPattern.setParameters(newParams);
            } catch (MissingParameterException ex) {
                // TODO: remake, so that we don't check for presence of variables
                // directly here. Variable can bubble through from the top..
                Exceptions.printStackTrace(ex);
            }

            List<TableArgument> editorArgs = editor.getArguments();
            Arguments newArgs = new Arguments();
            for (TableArgument editorArg : editorArgs) {
                newArgs.add(editorArg.createArgument());
            }

            for (TableParameter editorParam : editorParams) {
                if (editorParam.isOverriden()) {
                    newArgs.add(editorParam.createOverrideArgument());
                }
            }
            referencingAction.setArguments(newArgs);
        }
    }

    @Override
    public APEditor createEditorComponent(InplaceEditorProvider.EditorController controller, Widget widget) {
        APEditor editor = new APEditor(actionPattern, referencingAction, controller);
        return editor;
    }

    @Override
    public Rectangle getInitialEditorComponentBounds(InplaceEditorProvider.EditorController controller, Widget widget, APEditor editor, Rectangle viewBounds) {
        return new Rectangle(600, 300);
    }

    @Override
    public EnumSet<InplaceEditorProvider.ExpansionDirection> getExpansionDirections(InplaceEditorProvider.EditorController ec, Widget widget, APEditor c) {
        return EnumSet.of(InplaceEditorProvider.ExpansionDirection.BOTTOM, InplaceEditorProvider.ExpansionDirection.RIGHT);
    }
}
