/*
 * Copyright (C) 2014 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.ut2004.agent.navigation.navmesh.pathfollowing;

import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObject;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.unreal.communication.messages.UnrealId;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.IUT2004PathRunner;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.navmesh.NavMesh;
import cz.cuni.amis.pogamut.ut2004.bot.command.AdvancedLocomotion;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.Move;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPointNeighbourLink;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.WallCollision;
import cz.cuni.amis.pogamut.ut2004.utils.LinkFlag;
import cz.cuni.amis.pogamut.ut2004.utils.UnrealUtils;
import cz.cuni.amis.utils.NullCheck;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Runner for navigation with navigation mesh. Evolved from {@link KefikRunner}.
 *
 * We assume that we work with path generated by NavMesh path planner. It should
 * contain point on the navigation mesh, which should be navigable by simple
 * running, and off-mesh connections, which we will handle as original runner.
 *
 * Jump computing is working with exact locations, no reserves. Needed edge
 * should be achieved by delay in executing the jump.
 *
 * @author Bogo
 */
public class NavMeshRunner implements IUT2004PathRunner {

    private UT2004Bot bot;
    private AgentInfo memory;
    private AdvancedLocomotion body;
    private Logger log;

    private JumpBoundaries jumpBoundaries;

    /**
     * Our custom listener for WallCollision messages.
     */
    IWorldEventListener<WallCollision> myCollisionsListener = new IWorldEventListener<WallCollision>() {
        @Override
        public void notify(WallCollision event) {
            lastCollidingEvent = event;
        }

    };

    private JumpModule jumpModule;

    private CollisionDetector collisionDetector;
    private boolean inCollision = false;

    // MAINTAINED CONTEXT
    /**
     * Number of steps we have taken.
     */
    private int runnerStep = 0;

    /**
     * Jumping sequence of a single-jumps.
     */
    private int jumpStep = 0;

    /**
     * Collision counter.
     */
    private int collisionNum = 0;

    /**
     * Collision location.
     */
    private Location collisionSpot = null;

    // COMPUTED CONTEXT OF THE runToLocation
    /**
     * Current distance to the target, recalculated every
     * {@link NavMeshRunner#runToLocation(Location, Location, ILocated, NavPointNeighbourLink, boolean)}
     * invocation.
     */
    private double distance;

    /**
     * Current 2D distance (only in x,y) to the target, recalculated every
     * {@link NavMeshRunner#runToLocation(Location, Location, ILocated, NavPointNeighbourLink, boolean)}
     * invocation.
     */
    private double distance2D;

    /**
     * Current Z distance to the target (positive => target is higher than us,
     * negative => target is lower than us), recalculated every
     * {@link NavMeshRunner#runToLocation(Location, Location, ILocated, NavPointNeighbourLink, boolean)}
     * invocation.
     */
    private double distanceZ;

    /**
     * Current velocity of the bot, recalculated every
     * {@link NavMeshRunner#runToLocation(Location, Location, ILocated, NavPointNeighbourLink, boolean)}
     * invocation.
     */
    private double velocity;

    /**
     * Current velocity in Z-coord (positive, we're going up / negative, we're
     * going down), recalculated every
     * {@link NavMeshRunner#runToLocation(Location, Location, ILocated, NavPointNeighbourLink, boolean)}
     * invocation.
     */
    private double velocityZ;

    /**
     * Whether the jump is required somewhere along the link, recalculated every
     * {@link NavMeshRunner#runToLocation(Location, Location, ILocated, NavPointNeighbourLink, boolean)}
     * invocation.
     */
    private boolean jumpRequired;

    // CONTEXT PASSED INTO runToLocation
    /**
     * Current context of the
     * {@link NavMeshRunner#runToLocation(Location, Location, Location, ILocated, NavPointNeighbourLink, boolean)}.
     */
    private Location runningFrom;

    /**
     * Current context of the
     * {@link NavMeshRunner#runToLocation(Location, Location, Location, ILocated, NavPointNeighbourLink, boolean)}.
     */
    private Location firstLocation;

    /**
     * Current context of the
     * {@link NavMeshRunner#runToLocation(Location, Location, Location, ILocated, NavPointNeighbourLink, boolean)}.
     */
    private Location secondLocation;

    /**
     * Current context of the
     * {@link NavMeshRunner#runToLocation(Location, Location, Location, ILocated, NavPointNeighbourLink, boolean)}.
     */
    private ILocated focus;

    /**
     * Current context of the
     * {@link NavMeshRunner#runToLocation(Location, Location, Location, ILocated, NavPointNeighbourLink, boolean)}.
     */
    private NavPointNeighbourLink link;

    /**
     * Current context of the
     * {@link NavMeshRunner#runToLocation(Location, Location, Location, ILocated, NavPointNeighbourLink, boolean)}.
     */
    private boolean reachable;

    /**
     * Current angle of the bot movement to the ideal direction
     */
    private double angle;

    /**
     * If the bot is accelerating
     */
    private boolean accelerating = false;

    /**
     * Last received wall colliding event
     */
    protected WallCollision lastCollidingEvent = null;
    /**
     * If we have collided in last second we will signal it
     */
    private static final double WALL_COLLISION_THRESHOLD = 1;

    @Override
    public void reset() {
        // reset working info
        runnerStep = 0;
        jumpStep = 0;
        collisionNum = 0;
        collisionSpot = null;
        lastCollidingEvent = null;
        distance = 0;
        distance2D = 0;
        distanceZ = 0;
        velocity = 0;
        velocityZ = 0;
        angle = 0;
        jumpRequired = false;
        jumpBoundaries = null;
        accelerating = false;
    }

    @Override
    public boolean runToLocation(Location runningFrom, Location firstLocation, Location secondLocation, ILocated focus, NavPointNeighbourLink navPointsLink, boolean reachable) {
        // take another step
        ++runnerStep;

        // save context
        this.runningFrom = runningFrom;
        this.firstLocation = firstLocation;
        this.secondLocation = secondLocation;
        this.focus = focus;
        this.link = navPointsLink;
        this.reachable = reachable;

        // compute additional context
        distance = memory.getLocation().getDistance(firstLocation);
        distance2D = memory.getLocation().getDistance2D(firstLocation);
        distanceZ = firstLocation.getDistanceZ(memory.getLocation());

        double newVelocity = memory.getVelocity().size();
        accelerating = isAccelerating(newVelocity, velocity);

        velocity = newVelocity;
        velocityZ = memory.getVelocity().z;
        jumpRequired = !reachable || jumpModule.needsJump(link);

        if (jumpBoundaries == null || jumpBoundaries.getLink() != link) {
            jumpBoundaries = jumpModule.computeJumpBoundaries(link);
            debug("Computed jump boundaries. Jumpable: " + jumpBoundaries.isJumpable() + ", Start: " + jumpBoundaries.getTakeOffMin() + ", Ene: " + jumpBoundaries.getTakeOffMax());
        }

        Location direction = Location.sub(firstLocation, memory.getLocation()).setZ(0);
        direction = direction.getNormalized();
        Location velocityDir = new Location(memory.getVelocity().asVector3d()).setZ(0);
        velocityDir = velocityDir.getNormalized();
        angle = direction.dot(velocityDir);

        logDebugData(firstLocation, secondLocation, focus, reachable);

        // DELIBERATION
        if (runnerStep <= 1) {
            debug("FIRST STEP - start running towards new location");
            move(firstLocation, secondLocation, focus);
        }

        //internal collision detector
        if (collisionDetector.isColliding(memory.getLocation(), velocity, distance)) {
            inCollision = true;
            debug("Internal collision detector signalling collision, solving by force JUMPING!");
            return initJump(true);
        }

        // are we jumping already?
        if (jumpStep > 0) {
            debug("we're already jumping");
            return iterateJumpSequence();
        }

        // collision experienced?
        if (isColliding()) {
            debug("sensing collision");
            // try to resolve it
            return resolveCollision();
        } else {
            if (collisionSpot != null || collisionNum != 0) {
                debug("no collision, clearing collision data");
                collisionNum = 0;
                collisionSpot = null;
            }
        }

        //Get bot some time to start running - there is delay and we use ACC navigation, so the steps increase fast
        log.log(Level.FINER, "TRACE: RunnerStep: {0}", runnerStep);
        if (velocity < 5 && runnerStep > 5) {
            debug("velocity is zero and we're in the middle of running");
            if (link != null && (link.getFromNavPoint().isLiftCenter() || link.getFromNavPoint().isLiftExit())) {
                if (link.getFromNavPoint().isLiftCenter()) {
                    debug("we're standing on the lift center, ok");
                } else {
                    debug("we're standing on the lift exit, ok");
                }
            } else {
                debug("and we're not standing on the lift center");
                return initJump(true);
            }
        }

        // check jump
        if (jumpRequired) {
            debug("jump is required");
            return resolveJump();
        }

        // just continue with ordinary run
        debug("keeping running to the target");
        move(firstLocation, secondLocation, focus);

        return true;
    }

    private boolean isAccelerating(double newVelocity, double oldVelocity) {
        return velocity > 0 && (isMaxVelocity(newVelocity) || newVelocity > oldVelocity);
    }

    private void logDebugData(Location firstLocation, Location secondLocation, ILocated focus, boolean reachable) {
        // DEBUG LOG
        if (log != null && log.isLoggable(Level.FINER)) {
            debug("NavMeshRunner!");
            debug("running to    = " + firstLocation + " and than to " + secondLocation + " and focusing to " + focus);
            debug("bot position  = " + memory.getLocation());
            debug("distance      = " + distance);
            debug("distance2D    = " + distance2D);
            debug("distanceZ     = " + distanceZ);
            debug("velocity      = " + velocity);
            debug("velocityZ     = " + velocityZ);
            debug("angle         = " + Math.acos(angle) * (180 / Math.PI));
            debug("jumpRequired  = " + jumpRequired
                    + (!reachable ? " NOT_REACHABLE" : "")
                    + (link == null
                    ? ""
                    : ((link.getFlags() & LinkFlag.JUMP.get()) != 0 ? " JUMP_FLAG" : "") + (link.isForceDoubleJump() ? " DOUBLE_JUMP_FORCED" : "") + (link.getNeededJump() != null ? " AT[" + link.getNeededJump() + "]" : ""))
            );
            debug("reachable     = " + reachable);
            if (link != null) {
                debug("link          = " + link);
            } else {
                debug("LINK NOT PRESENT");
            }
            debug("collisionNum  = " + collisionNum);
            debug("collisionSpot = " + collisionSpot);
            debug("jumpStep      = " + jumpStep);
            debug("runnerStep    = " + runnerStep);
        }
    }

    /**
     * Constructor.
     *
     * @param bot Agent's bot.
     * @param agentInfo
     * @param locomotion
     * @param log
     * @param navMesh
     */
    public NavMeshRunner(UT2004Bot bot, AgentInfo agentInfo, AdvancedLocomotion locomotion, Logger log, NavMesh navMesh) {
        // setup reference to agent
        NullCheck.check(bot, "bot");
        this.bot = bot;
        NullCheck.check(agentInfo, "agentInfo");
        this.memory = agentInfo;
        NullCheck.check(locomotion, "locomotion");
        this.body = locomotion;

        //registering listener for wall collisions
        bot.getWorldView().addEventListener(WallCollision.class, myCollisionsListener);

        this.log = log;
        if (this.log == null) {
            this.log = bot.getLogger().getCategory(this.getClass().getSimpleName());
        }

        this.jumpModule = new JumpModule(navMesh, this.log);
        this.collisionDetector = new CollisionDetector();
    }

    private void debug(String message) {
        if (log.isLoggable(Level.FINER)) {
            log.log(Level.FINER, "Runner: {0}", message);
        }
    }

    private void move(ILocated firstLocation, ILocated secondLocation, ILocated focus) {
        Move move = new Move();
        if (firstLocation != null) {

            move.setFirstLocation(firstLocation.getLocation());
            if (secondLocation == null || secondLocation.equals(firstLocation)) {
                //We want to reach end of the path, we won't extend the second location.
                move.setSecondLocation(firstLocation.getLocation());
            } else {
                //Extend the second location so the bot doesn't slow down, when it's near the original target.
                double dist = firstLocation.getLocation().getDistance(secondLocation.getLocation());
                double quantifier = 1 + (200 / dist);

                Location extendedSecondLocation = firstLocation.getLocation().interpolate(secondLocation.getLocation(), quantifier);
                move.setSecondLocation(extendedSecondLocation);
            }
        } else if (secondLocation != null) {
            //First location was not set
            move.setSecondLocation(secondLocation.getLocation());
        }

        if (focus != null) {
            if (focus instanceof Player) {
                move.setFocusTarget((UnrealId) ((IWorldObject) focus).getId());
            } else {
                move.setFocusLocation(focus.getLocation());
            }
        }

        debug("MOVING: " + move);
        bot.getAct().act(move);
    }

    private boolean resolveCollision() {
        // are we colliding at a new spot?
        if ( // no collision yet
                (collisionSpot == null)
                // or the last collision is far away
                || (memory.getLocation().getDistance2D(collisionSpot) > 120)) {
            // setup new collision spot info
            if (log != null && log.isLoggable(Level.FINER)) {
                log.finer("Runner.resolveCollision(): collision");
            }
            collisionSpot = memory.getLocation();
            collisionNum = 1;
            // meanwhile: keep running to the location..
            move(firstLocation, secondLocation, focus);
            return true;
        } // so, we were already colliding here before..
        // try to solve the problem according to how long we're here..
        else {
            //Wait a while if the collision hasn't resolved by itself.
            if (collisionNum > 8) {
                if (log != null && log.isLoggable(Level.FINER)) {
                    log.finer("Runner.resolveCollision(): Solving collision by FORCE JUMPING");
                }
                inCollision = true;
                return initJump(true);
            } else {
                ++collisionNum;
                // meanwhile: keep running to the location..
                move(firstLocation, secondLocation, focus);
                return true;
            }
        }
    }

    private boolean isColliding() {
        if (lastCollidingEvent == null) {
            return false;
        }
        debug("isColliding():" + "(memory.getTime():" + memory.getTime() + " - (lastCollidingEvent.getSimTime() / 1000):" + (lastCollidingEvent.getSimTime() / 1000) + " <= WALL_COLLISION_THRESHOLD:" + WALL_COLLISION_THRESHOLD + " )");
        if (memory.getTime() - (lastCollidingEvent.getSimTime() / 1000) <= WALL_COLLISION_THRESHOLD) {
            debug("isColliding():return true;");
            return true;
        }

        return false;
    }

    private boolean resolveJump() {
        debug("resolveJump(): called");

        // cut the jumping distance2D of the next jump, this is to allow to
        // jump more than once per one runner request, while ensuring that
        // the last jump will always land exactly on the destination..
        //TODO: Changed - not exactly sure what it was for, hoping for 1 jump per link
        int jumpDistance2D = (int) distance2D; // ((int) distance2D) % 1000;

        debug("resolveJump(): jumpDistance2D = " + jumpDistance2D);

        // follow the deliberation about the situation we're currently in
        boolean jumpIndicated = false;      // whether we should jump now
        boolean mustJumpIfIndicated = false; // whether we MUST jump NOW

        // deliberation, whether we may jump
        if (jumpModule.needsJump(link)) {
            debug("resolveJump(): deliberation - jumping condition present");
            jumpIndicated = true;
        }

        //We need near perfect conditions for the jump, so we evaluate       
        if (!jumpBoundaries.isJumpable() && ((runnerStep > 1 && velocity > UnrealUtils.MAX_VELOCITY - 50 && angle < 20 && jumpDistance2D < 500) || jumpDistance2D < 250)) {
            debug("resolveJump(): we're facing forced jump and are near - forcing jump");
            mustJumpIfIndicated = true;
        }

        debug("resolveJump(): jumpIndicated       = " + jumpIndicated);
        debug("resolveJump(): mustJumpIfIndicated = " + mustJumpIfIndicated);

        if (jumpIndicated && mustJumpIfIndicated) {
            if (distanceZ > 0) {
                debug("resolveJump(): we MUST jump!");
                return prepareJump(true); // true == forced jump
            } else {
                debug("resolveJump(): we MUST fall down with a jump!");
                return prepareJump(true); // true == forced jump
            }
        } else if (jumpIndicated) {
            debug("resolveJump(): we should jump");
            return prepareJump(false); // false == we're not forcing to jump immediately         	
        } else {
            debug("resolveJump(): we do not need to jump, waiting to reach the right spot to jump from");
            // otherwise, wait for the right double-jump distance2D
            // meanwhile: keep running to the location..
            move(firstLocation, secondLocation, focus);
            return true;
        }
    }

    /*========================================================================*/
    /**
     * This method is called from {@link NavMeshRunner#resolveJump()} that has
     * decided that the jump is necessary to reach the the target (it is already
     * known that distanceZ > 0).
     * <p>
     * <p>
     * jumpForced == true ... we will try to run no matter what
     * <p>
     * <p>
     * jumpForced == false ... we will check whether the time is right for
     * jumping assessing the {@link NavMeshRunner#distanceZ}.
     *
     * @return whether we should reach the target
     */
    private boolean prepareJump(boolean jumpForced) {
        debug("prepareJump(): called");

        Double jumpAngleDeviation = Math.acos(jumpModule.getCorrectedAngle(angle, runnerStep <= 1));
        jumpForced |= runnerStep > 1 && jumpBoundaries.isPastBoundaries(memory.getLocation());
        Double jumpVelocity = jumpModule.getCorrectedVelocity(velocity, accelerating);

        boolean angleSuitable = !jumpAngleDeviation.isNaN() && jumpAngleDeviation < (Math.PI / 9);

        debug("prepareJump(): jumpAngleDeviation = " + jumpAngleDeviation);
        debug("prepareJump(): angleSuitable      = " + angleSuitable);

        if (jumpAngleDeviation > Math.PI / 3) {
            debug("prepareJump(): Impossible jump angle, postponing jump..");
            move(firstLocation, secondLocation, focus);
            return true;
        }

        if (jumpForced) {
            debug("prepareJump(): jump is forced, bypassing jump checks!");
        } else {
            debug("prepareJump(): jump is not forced, checking jump conditions");

            if (!jumpModule.isJumpable(memory.getLocation(), jumpBoundaries.getLandingTarget(), jumpVelocity)) {
                debug("prepareJump(): not jumpable! Start: " + memory.getLocation() + " Target: " + jumpBoundaries.getLandingTarget() + " Velocity: " + velocity + " Jump Velocity: " + jumpVelocity);
                debug("prepareJump(): proceeding with the straight movement to gain speed");
                move(firstLocation, secondLocation, focus);
                return true;
            }

            if (!angleSuitable) {
                debug("prepareJump(): angle is not suitable for jumping (angle > 20 degrees)");
                debug("prepareJump(): proceeding with the straight movement to gain speed");
                move(firstLocation, secondLocation, focus);
                return true;
            }

            //Waiting for ideal JUMP conditions
            if (jumpBoundaries.isJumpable() && jumpBoundaries.getTakeOffMax().getDistance2D(memory.getLocation()) > IDEAL_JUMP_RESERVE) {
                boolean angleIdeal = !jumpAngleDeviation.isNaN() && jumpAngleDeviation < (Math.PI / 90);
                if (!angleIdeal) {
                    debug("prepareJump(): proceeding with the straight movement - waiting for ideal jump ANGLE");
                    move(firstLocation, secondLocation, focus);
                    return true;
                }

                if (velocity < UnrealUtils.MAX_VELOCITY - 50) {
                    debug("prepareJump(): proceeding with the straight movement - waiting for ideal speed");
                    move(firstLocation, secondLocation, focus);
                    return true;
                }
            } else {
                debug("prepareJump(): passed ideal reserve spot, lower requirments on speed and angle");
            }

            debug("prepareJump(): velocity & angle is OK!");
        }

        debug("prepareJump(): JUMP");
        return initJump(jumpForced);

    }
    private static final int IDEAL_JUMP_RESERVE = 80;

    /**
     * We have to jump up (distanceZ > 0) if there is a possibility that we get
     * there by jumping (i.e., params for jump exists that should get us there)
     * or 'jumpForced is true'.
     * <p>
     * <p>
     * Decides whether we need single / double jump and computes the best args
     * for jump command according to current velocity.
     *
     * @param jumpForced
     */
    private boolean initJump(boolean jumpForced) {
        debug("initJump(): called");

        boolean doubleJump = true;
        Double jumpForce = Double.NaN;
        Double jumpVelocity = jumpModule.getCorrectedVelocity(velocity, accelerating);
        Double jumpAngleCos = jumpModule.getCorrectedAngle(angle, runnerStep <= 1);

        if (!jumpBoundaries.isJumpable()) {
            debug("initJump(): jump could not be made (distanceZ = " + distanceZ + " > 130)");
            if (jumpForced) {
                debug("initJump(): but jump is being forced!");
            } else {
                debug("initJump(): jump is not forced ... we will wait till the bot reach the right jumping spot");
                move(firstLocation, secondLocation, focus);
                jumpStep = 0; // we have not performed the JUMP
                return true;
            }
        }

        if (!jumpBoundaries.isJumpable()) {
            debug("Jump boundaries not present! We shouldn't be trying to JUMP!");
            jumpForce = Double.NaN;
        } else if (!jumpBoundaries.isInBoundaries(memory.getLocation())) {

            if (runnerStep > 1 && jumpBoundaries.isPastBoundaries(memory.getLocation())) {
                debug("Already passed max take-off point, forcing jump!");
                jumpForced = true;
                jumpForce = jumpModule.computeJump(memory.getLocation(), jumpBoundaries, jumpVelocity, jumpAngleCos);
            } else {
                debug("Not within jump boundaries! We should'n t JUMP");
                jumpForce = Double.NaN;
            }
        } else {

            jumpForce = jumpModule.computeJump(memory.getLocation(), jumpBoundaries, jumpVelocity, jumpAngleCos);
            if (jumpForce < UnrealUtils.FULL_JUMP_FORCE) {
                doubleJump = false;
            }

        }

        if (jumpForced && inCollision) {
            inCollision = false;
            jumpStep = 1; // we have performed the JUMP
            debug("initJump(): In collision, forcing jump - MAX!");
            return jump(true, UnrealUtils.FULL_DOUBLEJUMP_DELAY, UnrealUtils.FULL_DOUBLEJUMP_FORCE);

        }

        if (jumpForce.isNaN()) {
            if (jumpForced) {
                if (!accelerating) {
                    debug("initJump(): Forcing jump but NOT accelerating, postpone!");
                    move(firstLocation, secondLocation, focus);
                    return true;
                }

                jumpStep = 1; // we have performed the JUMP
                debug("initJump(): Forcing jump - MAX!");
                return jump(true, UnrealUtils.FULL_DOUBLEJUMP_DELAY, UnrealUtils.FULL_DOUBLEJUMP_FORCE);
            }

            //We cannot jump
            debug("initJump(): Jump failed to compute, continuing with move! Computed force: " + jumpForce);
            move(firstLocation, secondLocation, focus);
            return true;
        } else if (jumpForce < 0) {
            //We don't need to jump, so we will set
            debug("initJump(): We don't need to jump, continuing with move! Computed force: " + jumpForce);
            if (jumpBoundaries.isJumpable() && (this.jumpBoundaries.getTakeoffEdgeDirection() != null)) {
                //TODO: Jump down

                Location movementDirection = jumpBoundaries.getLandingTarget().sub(jumpBoundaries.getTakeOffMax()).setZ(0).getNormalized();
                Location meshDirection = jumpBoundaries.getTakeoffEdgeDirection();

                double fallAngleCos = meshDirection.setZ(0).getNormalized().dot(movementDirection);
                double takeOffDistance = jumpBoundaries.getLandingTarget().getDistance2D(jumpBoundaries.getTakeOffMax());
                if (Math.abs(fallAngleCos) > Math.cos(Math.PI / 2.5)) {
                    //Not direct approach, we should propably jump a little.
                    debug("initJump(): Not direct approach to fall, we should jump a little. Angle: " + Math.acos(fallAngleCos) * (180 / Math.PI));
                    //TODO: Fixed here
                    jumpBoundaries.setLandingTarget(jumpBoundaries.getTakeOffMax().interpolate(jumpBoundaries.getLandingTarget(), (IDEAL_JUMP_RESERVE / takeOffDistance)).setZ(jumpBoundaries.getTakeOffMax().z));
                    return true;
                } else {
                    debug("initJump(): Fall solved by not jumping, as angle is suitable. AngleCos: " + fallAngleCos);
                    jumpStep = 1;
                    return true;
                }
            } else {
                debug("initJump(): Fall solved by not jumping, as angle is suitable. Boundaries not jumpable.");
                jumpStep = 1;
                return true;
            }
        } else {
            jumpStep = 1; // we have performed the JUMP
            return jump(doubleJump, UnrealUtils.FULL_DOUBLEJUMP_DELAY, jumpForce);
        }

    }

    /*========================================================================*/
    /**
     * Perform jump right here and now with provided args.
     */
    private boolean jump(boolean doubleJump, double delay, double force) {
        if (doubleJump) {
            debug("DOUBLE JUMPING (delay = " + delay + ", force = " + force + ")");
        } else {
            debug("JUMPING (delay = " + delay + ", force = " + force + ")");
        }
        body.jump(doubleJump, delay, force);

        return true;
    }

    /*========================================================================*/
    /**
     * Follows single-jump sequence steps.
     *
     * @return True, if no problem occured.
     */
    private boolean iterateJumpSequence() {
        debug("iterateJumpSequence(): called");
        // what phase of the jump sequence?
        switch (jumpStep) {
            // the first phase: wait for the jump
            case 1:
                // did the agent started the jump already?
                if (velocityZ > 100) {
                    debug("iterateJumpSequence(): jumping in progress (velocityZ > 100), increasing jumpStep");
                    jumpStep++;
                }
                // meanwhile: just wait for the jump to start
                debug("iterateJumpSequence(): issuing move command to the target (just to be sure)");
                move(firstLocation, secondLocation, focus);
                return true;

            //  the last phase: finish the jump
            default:
                // did the agent started to fall already
                jumpStep++;
                if (velocityZ <= 0.01) {

                    if (velocityZ > lastVelocityZ) {
                        debug("iterateJumpSequence(): jump has ended");
                        lastVelocityZ = 0.02;
                        //jumpStep = 0;
                    } else {
                        lastVelocityZ = velocityZ;
                    }

                }
                debug("iterateJumpSequence(): continuing movement to the target");
                move(firstLocation, secondLocation, focus);
                return true;
        }

    }

    private double lastVelocityZ = 0.02;

    private boolean isMaxVelocity(double newVelocity) {
        return Math.abs(newVelocity - UnrealUtils.MAX_VELOCITY) < 5;
    }

}
