package cz.cuni.amis.pogamut.ut2004.agent.navigation;

import cz.cuni.amis.pogamut.base.agent.navigation.IPathExecutorState;
import cz.cuni.amis.pogamut.base.agent.navigation.IPathFuture;
import cz.cuni.amis.pogamut.base.agent.navigation.IStuckDetector;
import cz.cuni.amis.pogamut.base.agent.navigation.impl.PrecomputedPathFuture;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.floydwarshall.FloydWarshallMap;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.loquenavigator.KefikRunner;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.loquenavigator.LoqueNavigator;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.stuckdetector.UT2004PositionHistoryStuckDetector;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.stuckdetector.UT2004TimeStuckDetector;
import cz.cuni.amis.pogamut.ut2004.bot.command.AdvancedLocomotion;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.utils.flag.FlagListener;
import cz.cuni.amis.utils.future.FutureStatus;
import cz.cuni.amis.utils.future.FutureWithListeners;
import cz.cuni.amis.utils.future.IFutureListener;
import java.util.ArrayList;
import java.util.List;

/**
 * Facade for navigation in UT2004. Method navigate() can be called both synchronously and asynchronously.
 * Uses UT2004PathExecutor and FloydWarshallMap for navigating and planning.
 * @author knight
 */
public class UT2004Navigation {

    /** UT2004PathExecutor that is used for the navigation. */
    private UT2004PathExecutor ut2004PathExecutor;
    /** FloydWarshallMap that is used for path planning. */
    private FloydWarshallMap fwMap;
    /** Last location target. */
    private ILocated lastLocationTarget;
    /** Current location target. */
    private ILocated currentLocationTarget = null;
    /** Current path stored in IPathFuture object. */
    private IPathFuture currentFuturePath;
    /** UT2004Bot reference. */
    private UT2004Bot bot;
    /** Location threshold for requesting of a new path or switching a path. */
    private static final int NEW_PATH_DISTANCE_THRESHOLD = 20;
    /** Location threshold for checking whether we have arrived on target. For XY - 2D plane distance */
    private static final int ARRIVED_AT_LOCATION_XY_THRESHOLD = 50;
    /** Location threshold for checking whether we have arrived on target. For Z distance. */
    private static final int ARRIVED_AT_LOCATION_Z_THRESHOLD = 100;
    /** Set to true if user supported his own user path. */
    private boolean bUsingUserPath = false;
    /**
     * Listener for UT2004PathExecutor state. Not used right now.
     */
    FlagListener myUT2004PathExecutorStateListener = new FlagListener<IPathExecutorState>() {

        @Override
        public void flagChanged(IPathExecutorState changedValue) {
            switch (changedValue.getState()) {
                case PATH_COMPUTATION_FAILED:
                    // if path computation fails to whatever reason, just try another navpoint
                case TARGET_REACHED:
                    // most of the time the execution will go this way
                    break;
                case STUCK:
                    // the bot has stuck! ... target nav point is unavailable currently
                    break;
            }
        }
    };
    /**
     * This listener servers two purposes: 1) It stores path obtained through IPathFuture to normal
     * list and 2) it checks whether the last path element location is in reasonable delta from
     * the last location and if not, then we will add our target location as the last path element.
     */
    IFutureListener<List<ILocated>> myIPathFutureListener = new IFutureListener<List<ILocated>>() {

        @Override
        public void futureEvent(FutureWithListeners<List<ILocated>> source, FutureStatus oldStatus, FutureStatus newStatus) {
            //System.out.println("In:UT2004Navigation:myIPathFutureListener: " + oldStatus + ";new" + newStatus);
            if (newStatus == FutureStatus.FUTURE_IS_READY) {
                if (!source.get().isEmpty()) {
                    ILocated lastPathElement = source.get().get(source.get().size() - 1);
                    if (lastPathElement.getLocation().getDistance(currentLocationTarget.getLocation()) > NEW_PATH_DISTANCE_THRESHOLD) {
                        currentFuturePath.get().add(currentLocationTarget);
                        ut2004PathExecutor.followPath(currentFuturePath);
                        //System.out.println("In:UT2004Navigation:myIPathFutureListener:BEEN HERE2");
                    }
                } else {
                    currentFuturePath.get().add(currentLocationTarget);
                    ut2004PathExecutor.followPath(currentFuturePath);
                    //System.out.println("In:UT2004Navigation:myIPathFutureListener:BEEN HERE3");
                }
            }
        }
    };

    /**
     * Checks if last path element is in close distance from our desired target and if not, we
     * will add our desired target as the last path element.
     * @param futurePath
     */
    private void processIPathFuture(IPathFuture futurePath) {
        //System.out.println("In:UT2004Navigation:processIPathFuture():BEEN HERE1");
        List<ILocated> pathList = futurePath.get();
        if (!pathList.isEmpty()) {
            ILocated lastPathElement = pathList.get(pathList.size() - 1);
            if (lastPathElement.getLocation().getDistance(currentLocationTarget.getLocation()) > NEW_PATH_DISTANCE_THRESHOLD) {
                currentFuturePath.get().add(currentLocationTarget);
                //System.out.println("In:UT2004Navigation:processIPathFuture():BEEN HERE2");
            }
        } else {
            currentFuturePath.get().add(currentLocationTarget);
            //System.out.println("In:UT2004Navigation:processIPathFuture():BEEN HERE3");
        }
    }

    /**
     * True if UT2004PathExecutor is currently trying to get somewhere.
     * @return
     */
    public boolean isNavigating() {
        return ut2004PathExecutor.isExecuting();
    }

    /**
     * This method can be called periodically or asynchronously. Will move bot to input location.
     * Uses UT2004PathExecutor and FloydWarshallMap.
     * The bot will stop on bad input (location null).
     * @param location target location
     */
    public void navigate(ILocated location) {
        //we store this information for comparing our previous and current location targets
        lastLocationTarget = currentLocationTarget;

        currentLocationTarget = location;

        //we will stop on bad input!
        if (currentLocationTarget == null) {
            if (ut2004PathExecutor.isExecuting()) {
                ut2004PathExecutor.stop();
            }
            bUsingUserPath = false;
            currentFuturePath = null;
            return;
        }

        /*
        if (bot.getLocation().getDistance2D(currentLocationTarget.getLocation()) < ARRIVED_AT_LOCATION_XY_THRESHOLD
        && bot.getLocation().getDistanceZ(currentLocationTarget.getLocation()) < ARRIVED_AT_LOCATION_Z_THRESHOLD) {
        if (!ut2004PathExecutor.isTargetReached()) {
        currentPath.clear();
        currentFuturePath = fwMap.computePath(getNearestNavPoint(bot.getLocation()), getNearestNavPoint(currentLocationTarget));
        currentFuturePath.addFutureListener(new IFutureListener<List<ILocated>>() {

        @Override
        public void futureEvent(FutureWithListeners<List<ILocated>> source, FutureStatus oldStatus, FutureStatus newStatus) {
        currentPath.addAll(source.get());
        }
        });
        ut2004PathExecutor.followPath(currentFuturePath);
        }
        return;
        }*/

        if (!ut2004PathExecutor.isExecuting()) {
            //if we are not moving, we will start to move
            currentFuturePath = fwMap.computePath(getNearestNavPoint(bot.getLocation()), getNearestNavPoint(currentLocationTarget));
            bUsingUserPath = false;
            if (currentFuturePath.isDone()) {
                processIPathFuture(currentFuturePath);
            } else {
                currentFuturePath.addFutureListener(myIPathFutureListener);
            }
            ut2004PathExecutor.followPath(currentFuturePath);
        } else {
            //check if last target differs from current target in the means of location distance or
            //if we were running on user specified path if any condition apply we will get new path
            if (( bUsingUserPath
                   || (lastLocationTarget != currentLocationTarget)
                    && lastLocationTarget.getLocation().getDistance(currentLocationTarget.getLocation()) > NEW_PATH_DISTANCE_THRESHOLD)) {
                currentFuturePath = fwMap.computePath(getNearestNavPoint(bot.getLocation()), getNearestNavPoint(currentLocationTarget));
                bUsingUserPath = false;
                if (currentFuturePath.isDone()) {
                    processIPathFuture(currentFuturePath);
                } else {
                    currentFuturePath.addFutureListener(myIPathFutureListener);
                }
                ut2004PathExecutor.followPath(currentFuturePath);
            } else {
                //if someone else used this path executor from his custom code, we will check here
                //whether pathExecutors target matches our target through checking last path elements of
                //path executors path and our stored current path
                List<ILocated> pathExecutorPath = ut2004PathExecutor.getPath();
                if (!pathExecutorPath.isEmpty()) {
                    ILocated lastPathExecutorPathElement = pathExecutorPath.get(pathExecutorPath.size() - 1);
                    if (!currentFuturePath.get().isEmpty()) {
                        ILocated lastCurrentPathElement = (ILocated) currentFuturePath.get().get(currentFuturePath.get().size() - 1);
                        if (lastPathExecutorPathElement.getLocation().getDistance(lastCurrentPathElement.getLocation()) > NEW_PATH_DISTANCE_THRESHOLD) {
                            currentFuturePath = fwMap.computePath(getNearestNavPoint(bot.getLocation()), getNearestNavPoint(currentLocationTarget));
                            bUsingUserPath = false;
                            if (currentFuturePath.isDone()) {
                                processIPathFuture(currentFuturePath);
                            } else {
                                currentFuturePath.addFutureListener(myIPathFutureListener);
                            }
                            ut2004PathExecutor.followPath(currentFuturePath);
                        }
                    }
                }
            }
        }
    }

    /**
     * This method can be called synchronously or asynchronously, will move bot along input
     * path by using UT2004PathExecutor.
     * The bot will stop on bad input (empty or null path).
     * Do NOT support circular paths (paths that begin and end on the same spot) - this
     * will NOT work - UT2004PathExecutor does not handle well those paths.
     *
     * @param path custom path
     */
    public void navigate(List<ILocated> path) {
        //we will stop on bad input!
        if (path == null || path.isEmpty()) {
            if (ut2004PathExecutor.isExecuting()) {
                ut2004PathExecutor.stop();
            }
            currentLocationTarget = null;
            currentFuturePath = null;
            bUsingUserPath = false;
            return;
        }

        //System.out.println("In:UT2004Navigation:navigate(List<ILocated> path):BEEN1");

        if (!ut2004PathExecutor.isExecuting()) {
            //.out.println("In:UT2004Navigation:navigate(List<ILocated> path):BEEN2");
            //if we are not moving, we will start to move
            currentFuturePath = new PrecomputedPathFuture<ILocated> (path.get(0), path.get(path.size() - 1), path);
            bUsingUserPath = true;
            ut2004PathExecutor.followPath(currentFuturePath);
        } else if (currentFuturePath == null) {
            //System.out.println("In:UT2004Navigation:navigate(List<ILocated> path):BEEN3");
            //this should never happen
            bot.getLog().warning("In:UT2004Navigation:navigate(List<ILocated> path): weird state - pathExecutor running and currentFuturePath is null");
            currentFuturePath = new PrecomputedPathFuture<ILocated> (path.get(0), path.get(path.size() - 1), path);
            bUsingUserPath = true;
            ut2004PathExecutor.followPath(currentFuturePath);
        } else {
            //System.out.println("In:UT2004Navigation:navigate(List<ILocated> path):BEEN4");
            List<ILocated> oldPath = currentFuturePath.get();
            //check if the two paths differ, if yes, use new path.
            if (!bUsingUserPath || oldPath == null || oldPath.isEmpty() || oldPath.size() != path.size()) {
                System.out.println("In:UT2004Navigation:navigate(List<ILocated> path):BEEN5");
                currentFuturePath = new PrecomputedPathFuture<ILocated> (path.get(0), path.get(path.size() - 1), path);
                bUsingUserPath = true;
                ut2004PathExecutor.followPath(currentFuturePath);
            } else if (oldPath != path) { //first we test if those are different objects - if they are the same we need to do nothing
                //System.out.println("In:UT2004Navigation:navigate(List<ILocated> path):BEEN6");
                int i = 0;
                for (ILocated iLoc : path) {
                    if (oldPath.get(i) == null || oldPath.get(i).getLocation().getDistance(iLoc.getLocation()) > NEW_PATH_DISTANCE_THRESHOLD) {
                        //System.out.println("In:UT2004Navigation:navigate(List<ILocated> path):BEEN7");
                        currentFuturePath = new PrecomputedPathFuture<ILocated> (path.get(0), path.get(path.size() - 1), path);
                        bUsingUserPath = true;
                        ut2004PathExecutor.followPath(currentFuturePath);
                        break;
                    }
                    i++;
                }
            }
        }
    }

    /**
     * Returns COPY of current path in list. May take some time to fill up. Returns
     * empty list if path not computed.
     * @return
     */
    public List<ILocated> getCurrentPathCopy() {
        List<ILocated> result = new ArrayList();
        if (currentFuturePath != null) {
            result.addAll(currentFuturePath.get());
        }
        return result;
    }

    /**
     * Returns current path as in IPathFuture object that is used by ut2004pathExecutor
     * to navigate. Can be altered. May return null if path not computed!
     * Be carefull when altering this during UT2004PathExecutor run - it may cause
     * undesirable behavior.
     * @return
     */
    public List<ILocated> getCurrentPathDirect() {

        if (currentFuturePath != null) {
            return currentFuturePath.get();
        }

        return null;
    }

    /**
     * Returns nearest navigation point to input location. FloydWarshallMap works only
     * on NavPoints.
     * @param location
     * @return
     */
    private NavPoint getNearestNavPoint(ILocated location) {
        NavPoint result = null;
        for (NavPoint nav : bot.getWorldView().getAll(NavPoint.class).values()) {
            if (result == null || location.getLocation().getDistance(result.getLocation()) > location.getLocation().getDistance(nav.getLocation())) {
                result = nav;
            }
        }

        return result;
    }

    /**
     * Use this to register listener to various states of path execution - stuck, target reached, etc.
     * @param listener
     */
    public void addStrongNavigationListener(FlagListener<IPathExecutorState> listener) {
        ut2004PathExecutor.getState().addStrongListener(listener);
    }

    /**
     * Removes input listener.
     * @param listener
     */
    public void removeStrongNavigationListener(FlagListener<IPathExecutorState> listener) {
        ut2004PathExecutor.getState().removeListener(listener);
    }

    /**
     * Sets focus of the bot when navigating (when using this object to run to some location target)!
     * To reset focus call this method with null parameter.
     * @param located
     */
    public void setFocus(ILocated located) {
        ut2004PathExecutor.setFocus(located);
    }

    /**
     * Stops navigation and resets currentPath.
     * Does NOT reset focus!
     */
    public void stopNavigation() {
        ut2004PathExecutor.stop();
        //reseting
        currentLocationTarget = null;
        lastLocationTarget = null;
        currentFuturePath = null;
        bUsingUserPath = false;
    }

    /**
     * Adds stuck detector for UT2004PathExecutor.
     * @param stuckDetector
     */
    public void addStuckDetector(IStuckDetector stuckDetector) {
        ut2004PathExecutor.addStuckDetector(stuckDetector);
    }

    /**
     * Removes stuck detector for UT2004PathExecutor.
     * @param stuckDetector
     */
    public void removeStuckDetector(IStuckDetector stuckDetector) {
        ut2004PathExecutor.removeStuckDetector(stuckDetector);
    }

    /**
     * Returns list of all stuck detectors.
     * @return
     */
    public List<IStuckDetector> getStuckDetectors() {
        return ut2004PathExecutor.getStuckDetectors();
    }

    /**
     * Enables to set custom UT2004PathExecutor.
     *
     * @param bot
     * @param ut2004PathExecutor
     * @param fwMap
     */
    public UT2004Navigation(UT2004Bot bot, UT2004PathExecutor ut2004PathExecutor, FloydWarshallMap fwMap) {
        this.bot = bot;
        this.ut2004PathExecutor = ut2004PathExecutor;
        this.fwMap = fwMap;

        ut2004PathExecutor.getState().addListener(myUT2004PathExecutorStateListener);
    }

    /**
     * Enables to use already created instance of FloydWarshallMap. Inits UT2004PathExecutor
     * with LoqueNavigator and KefikRunner.
     * @param bot
     * @param fwMap
     */
    public UT2004Navigation(UT2004Bot bot, FloydWarshallMap fwMap) {
        this.bot = bot;
        this.fwMap = fwMap;

        KefikRunner kRunner = new KefikRunner(bot, new AgentInfo(bot), new AdvancedLocomotion(bot, bot.getLog()), bot.getLog());
        LoqueNavigator lNav = new LoqueNavigator(bot, kRunner, bot.getLog());
        ut2004PathExecutor = new UT2004PathExecutor(bot, lNav);
        ut2004PathExecutor.addStuckDetector(new UT2004TimeStuckDetector(bot));       // if the bot does not move for 3 seconds, considered that it is stuck
        ut2004PathExecutor.addStuckDetector(new UT2004PositionHistoryStuckDetector(bot));

        ut2004PathExecutor.getState().addListener(myUT2004PathExecutorStateListener);
    }

    /**
     * Basic constructor. Inits FloydWarshallMap and UT2004PathExecutor
     * with LoqueNavigator and KefikRunner. Has to be called in prepareBot method in
     * order to work properly (FloydWarshallMap requires this).
     * @param bot
     */
    public UT2004Navigation(UT2004Bot bot) {
        this.bot = bot;
        fwMap = new FloydWarshallMap(bot);
        KefikRunner kRunner = new KefikRunner(bot, new AgentInfo(bot), new AdvancedLocomotion(bot, bot.getLog()), bot.getLog());
        LoqueNavigator lNav = new LoqueNavigator(bot, kRunner, bot.getLog());
        ut2004PathExecutor = new UT2004PathExecutor(bot, lNav);
        ut2004PathExecutor.addStuckDetector(new UT2004TimeStuckDetector(bot));       // if the bot does not move for 3 seconds, considered that it is stuck
        ut2004PathExecutor.addStuckDetector(new UT2004PositionHistoryStuckDetector(bot));

        ut2004PathExecutor.getState().addListener(myUT2004PathExecutorStateListener);
    }

    /**
     * Basic constructor. Inits FloydWarshallMap and UT2004PathExecutor
     * with LoqueNavigator and KefikRunner. Has to be called in prepareBot method in
     * order to work properly (FloydWarshallMap requires this). Enables to specify
     * user AgentInfo and AdvancedLocomotion modules so UT2004Navigation will not need
     * to duplicate them.
     * @param bot
     * @param info
     * @param locomotion
     */
    public UT2004Navigation(UT2004Bot bot, AgentInfo info, AdvancedLocomotion locomotion) {
        this.bot = bot;
        fwMap = new FloydWarshallMap(bot);
        KefikRunner kRunner = new KefikRunner(bot, info, locomotion, bot.getLog());
        LoqueNavigator lNav = new LoqueNavigator(bot, kRunner, bot.getLog());
        ut2004PathExecutor = new UT2004PathExecutor(bot, lNav);
        ut2004PathExecutor.addStuckDetector(new UT2004TimeStuckDetector(bot));       // if the bot does not move for 3 seconds, considered that it is stuck
        ut2004PathExecutor.addStuckDetector(new UT2004PositionHistoryStuckDetector(bot));

        ut2004PathExecutor.getState().addListener(myUT2004PathExecutorStateListener);
    }
}
