package cz.cuni.amis.pogamut.udk.server;

import java.util.Random;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import junit.framework.Assert;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import cz.cuni.amis.pogamut.base.agent.IAgentId;
import cz.cuni.amis.pogamut.base.agent.impl.AbstractAgent;
import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.agent.params.impl.RemoteAgentParameters;
import cz.cuni.amis.pogamut.base.agent.utils.AgentKeepAlive;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.utils.Pogamut;
import cz.cuni.amis.pogamut.base.utils.PogamutProperty;
import cz.cuni.amis.pogamut.udk.factory.guice.remoteagent.UDKServerFactory;
import cz.cuni.amis.pogamut.udk.factory.guice.remoteagent.UDKServerModule;
import cz.cuni.amis.pogamut.udk.server.IUDKServer;
import cz.cuni.amis.pogamut.udk.server.exception.UCCStartException;
import cz.cuni.amis.pogamut.udk.server.impl.UDKServer;
import cz.cuni.amis.pogamut.udk.test.BaseUDKTest;
import cz.cuni.amis.pogamut.udk.utils.PogamutUDKProperty;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper.UCCWrapperConf;

/**
 * Testing the {@link UDKServer#setGameMap(String)} by changing maps on the UCCWrapper + server is managed by {@link AgentKeepAlive}.
 */
public class UDKTest09_UDKServer_SetGameMap_AgentKeepAlive2 extends BaseUDKTest {
	
	private static Random random = new Random(System.currentTimeMillis());

	private static final Level LOG_LEVEL = Level.INFO;

	protected static boolean useInternalUcc = !Pogamut.getPlatform().getProperty(PogamutUDKProperty.POGAMUT_UNREAL_TEST_EXT_SERVER.getKey()).equals("true");
	
	private UCCWrapper uccWrapper;
	
    IUDKServer server;
    UDKServerFactory factory;
   
    @Override
    @Before
    public void startTest() throws UCCStartException {
    	if (!useInternalUcc) {
    		//Assert.fail("Must run on internal ucc to allow UCC restarts.");
    	} else {
    		System.out.println("Starting UCC...");
    		uccWrapper = new UCCWrapper(new UCCWrapperConf().setStartOnUnusedPort(false));
    	}
    }

    @Override
    @After
    public void endTest() {
    	if (server != null) {
    		server.kill();
    	}
    	if (useInternalUcc) {
	    	System.out.println("Stopping UCC...");
	    	uccWrapper.stop();
    	}
    	System.out.println("Closing PogamutPlatform...");
    	Pogamut.getPlatform().close();
    }
    
    private void changeMap(IUDKServer server, String map) {
    	if (awaitAgentUp((AbstractAgent)server)) {
    		System.out.println("Changing map to '" + map + "'...");
    		if (useInternalUcc) {
    			System.out.println("UCC Host - " + uccWrapper.getHost() + ", Port - " + uccWrapper.getControlPort());
    		}
    		Future<Boolean> future = server.setGameMap(map);
    		try {
    			System.out.println("Waiting for the GBUDK to change the map (60sec timeout).");
    			Boolean result = future.get(60000, TimeUnit.MILLISECONDS);
        		if (result == null || !result) {
        			Assert.fail("Failed to change map to '" + map + "'.");
        		}
    		} catch (Exception e) {
    			Assert.fail("Failed to change map to '" + map + "'.");
    		}
    	} else {
    		Assert.fail("Failed to connect to GBUDK...");
    	}
	}

    @Test
    public void test01_SetGameMap() {
    	String host = "localhost";
    	int port = 3001;
    	
    	if (useInternalUcc) {
    		host = uccWrapper.getHost();
    		port = uccWrapper.getControlPort();
    	}
        
        factory = new UDKServerFactory(new UDKServerModule());
        
        IAgentId agentId = new AgentId("Test-ChangeMap");
        server = (IUDKServer) factory.newAgent(new RemoteAgentParameters().setAgentId(agentId).setWorldAddress(new SocketConnectionAddress(host, port)));
        AbstractAgent serverAgent = (AbstractAgent)server;
        serverAgent.getLogger().setLevel(LOG_LEVEL);
        serverAgent.getLogger().addDefaultConsoleHandler();
        
        AgentKeepAlive keepAlive = new AgentKeepAlive(serverAgent, 1000);
        keepAlive.start();
        
        if (!awaitAgentUp(serverAgent)) {
        	Assert.fail("Failed to connect to GBUDK.");
        } else {
        	System.out.println("Connected...");
        }
		
        String[] maps = new String[] {
        		"DM-Deck",
        		"DM-Sanctuary"
        };
        
        try {
        	for (int i = 0; i < 5; ++i) {
        		System.out.println("ChangeMap + AgentKeepAlive test: " + (i+1) + " / 5...");
		        for (int j = 0; j < 5; ++j) {
		        	System.out.println("Change map " + (j+1) + " / 5...");
		        	changeMap(server, maps[random.nextInt(maps.length)]);
		        }		        
		        
		        if (useInternalUcc) {		        
		        	System.out.println("Stopping UCC...");		        
		        	uccWrapper.stop();		        	
		        }
	        	
	        	try {
	        		awaitAgentDown(serverAgent);
	        	} catch (Exception e) {
	        		System.out.println("???");
	        	}
	        	
	        	if (i == 4) break;
	        	
	        	if (useInternalUcc) {
	        		System.out.println("Starting UCC...");
	        		uccWrapper = new UCCWrapper(new UCCWrapperConf());
	        		server.setAddress(uccWrapper.getHost(), uccWrapper.getControlPort());
	        	}
	        	
	        	try {
	        		awaitAgentUp(serverAgent);
	        	} catch (Exception e) {
	        		System.out.println("???");
	        	}
        	}
        } finally {
        	try {
        		keepAlive.stop();
        	} finally {
        		server.kill();
        	}
        }
		
        System.out.println("---/// TEST OK ///---");        
    }

	public static void main(String[] args) {
        UDKTest09_UDKServer_SetGameMap_AgentKeepAlive2 test = new UDKTest09_UDKServer_SetGameMap_AgentKeepAlive2();
        test.startTest();
        test.test01_SetGameMap();
        test.endTest();
    }
}
