package cz.cuni.amis.pogamut.udk.communication.messages;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Set;

import cz.cuni.amis.utils.maps.HashMapSet;

/**
 * Type of the item.
 * 
 * <p>
 * Note: Items of the same type might have different names in UT engine.
 * <b>Always use {@link #equals(Object)} to safely compare two ItemTypes.</b>
 * 
 * <p>
 * Use {@link #getCategory()} to obtain basic categorization of items.
 * 
 * <p>
 * Use {@link #getGroup()} to obtain detailed group info of items.
 * 
 * @author Juraj 'Loque' Simlovic
 * @author Jimmy
 * @author Radek 'Black_Hand' Pibil
 */
public class ItemType implements Serializable {
	
	/**
	 * Contains item types that belongs to their categories.
	 */
	public static final HashMapSet<Category, ItemType> CATEGORIES = new HashMapSet<Category, ItemType>();
	
	/**
	 * List of all item categories. Categories divide items into several basic
	 * categories, based on what the item does and what is it intended to do.
	 */
	public enum Category {
		/** Weapons of all sorts. */
		WEAPON("Weapon"),
		/** Ammunition for weapons of all sorts. */
		AMMO("Ammo"),
		/** Health packs and other health restorers. */
		HEALTH("Health"),
		/** Armor packs and other armor restorers. */
		ARMOR("Armor"),
		/** UDamage, Keys + user defined items */
		OTHER("Other");

		/* =================================================================== */

		/** Human-readable name of the category. */
		public String name;

		/* =================================================================== */

		/**
		 * Constructor.
		 * 
		 * @param name
		 *            Human-readable name of the category.
		 */
		Category(String name) {
			this.name = name;
		}
		
		/**
		 * Return all item types of a certain category.
		 * @return
		 */
		public Set<ItemType> getTypes() {
			return CATEGORIES.get(this);
		}
		
	}

	/* ======================================================================== */


	/**
	 * Contains item types that belongs to their groups.
	 */
	public static final HashMapSet<Group, ItemType> GROUPS = new HashMapSet<Group, ItemType>();
	
	/**
	 * List of all item groups. Groups fine down the categories into specific
	 * groups, based on what the item belongs to. Also, groups join items from
	 * different categories together, if they belong together (e.g. weapon with
	 * its ammo).
	 */
	public enum Group {
		/** LinkGun weapon and accessory. */
		LINK_GUN("LinkGun"),
		/** ShockRifle weapon and accessory. */
		SHOCK_RIFLE("ShockRifle"),		
		/** RocketLauncher weapon and accessory. */
		ROCKET_LAUNCHER("RocketLauncher"),
		/** Physics gun */
		PHYSICS_GUN("PhysicsGun"),

		/** Classic health pack. */
		HEALTH("HealthKit"),
		/** Mini health vial. */
		MINI_HEALTH("HealthVial"),

		/** Thigh pad. */
		THIGH_PAD("Thighpad"),
		/** Base armor. */
		BASE_ARMOR("BaseArmor"),
		/** ShieldBelt. */
		SHIELD_BELT("ShieldBelt"),

		/** Jump boots. */
		JUMP_BOOTS("JumpBoots"),
		/** UDamage bonus items. */
		UDAMAGE("UDamage"),
		/** Other items with user-defined group. */
		OTHER("Unknown"),
		/** No group, used for the prototype None */
		NONE("None");

		/* =================================================================== */

		/** Human-readable name of the group. */
		public String name;

		/* =================================================================== */

		/**
		 * Constructor.
		 * 
		 * @param name
		 *            Human-readable name of the group.
		 */
		Group(String name) {
			this.name = name;
		}
		
		public Set<ItemType> getTypes() {
			return GROUPS.get(this);
		}
	}

	/* ======================================================================== */

	/**
	 * Map of all registered ItemType prototypes.
	 */
	private static HashMap<String, ItemType> protos = new HashMap<String, ItemType>();

	/* ======================================================================== */

	/** ShockRifle weapon. */
	public static final ItemType SHOCK_RIFLE = MakePrototype(Category.WEAPON,
			Group.SHOCK_RIFLE, new String[] { "ShockRifle.WeaponPickup" });
	/** ShockRifle ammo. */
	public static final ItemType SHOCK_RIFLE_AMMO = MakePrototype(
			Category.AMMO, Group.SHOCK_RIFLE,
			new String[] { "ShockRifleAmmo.AmmoPickup" });

	/** LinkGun weapon. */
	public static final ItemType LINK_GUN = MakePrototype(Category.WEAPON,
			Group.LINK_GUN, new String[] { "LinkGun.WeaponPickup" });
	/** LinkGun ammo. */
	public static final ItemType LINK_GUN_AMMO = MakePrototype(Category.AMMO,
			Group.LINK_GUN, new String[] { "LinkGunAmmo.AmmoPickup" });

	/** RocketLauncher weapon. */
	public static final ItemType ROCKET_LAUNCHER = MakePrototype(
			Category.WEAPON, Group.ROCKET_LAUNCHER, new String[] {
					"RocketLauncher.WeaponPickup",
					"RocketLauncher_Content.WeaponPickup" });
	/** RocketLauncher ammo. */
	public static final ItemType ROCKET_LAUNCHER_AMMO = MakePrototype(
			Category.AMMO, Group.ROCKET_LAUNCHER,
			new String[] { "RocketLauncherAmmo.AmmoPickup",
					"RocketLauncher_ContentAmmo.AmmoPickup" });

	/** Health pack. */
	public static final ItemType HEALTH_PACK = MakePrototype(Category.HEALTH,
			Group.HEALTH, new String[] { "HealthPack.HealthPickup" });
	/** Health vial. */
	public static final ItemType HEALTH_VIAL = MakePrototype(
			Category.HEALTH, Group.MINI_HEALTH,
			new String[] { "HealthVial.HealthPickup" });

	/** Thigh pad. */
	public static final ItemType THIGH_PAD = MakePrototype(Category.ARMOR,
			Group.THIGH_PAD, new String[] { "Thighpad.ArmorPickup" });
	/** Base armor. */
	public static final ItemType BASE_ARMOR = MakePrototype(
			Category.ARMOR, Group.BASE_ARMOR,
			new String[] { "BaseArmor.ArmorPickup" });
	/** Shield belt. */
	public static final ItemType SHIELD_BELT = MakePrototype(
			Category.ARMOR, Group.SHIELD_BELT,
			new String[] { "ShieldBelt.ArmorPickup" });	

	/** UDamage bonus (damage multiplier). */
	public static final ItemType UDAMAGE = MakePrototype(
			Category.OTHER, Group.UDAMAGE, new String[] {
					"UDamage.Pickup" });
	/** UDamage bonus (damage multiplier). */
	public static final ItemType JUMP_BOOTS = MakePrototype(
			Category.OTHER, Group.JUMP_BOOTS, new String[] {
					"JumpBoots.Pickup" });
	
	/** Weapons locker */
	public static final ItemType WEAPON_LOCKER = MakePrototype(
			Category.OTHER, Group.OTHER, new String[] {
					"WeaponLocker.LockerPickup" });
	
	/** No ItemType */
	public static final ItemType NONE = MakePrototype(Category.OTHER, Group.NONE,
			new String[] { "None", "NONE", "none" });
	
	/* ======================================================================== */

	/**
	 * Name of the item in UT engine.
	 * 
	 * <p>
	 * Note: Items of the same type might have different names in UT engine. Use
	 * {@link #equals(Object)} to safely compare two ItemTypes. This name is
	 * informative only.
	 */
	private String name;
	
	public String toString() {
		return "ItemType[name = " + name + ", category = " + category + ", group = " + group + "]";
	}

	/* ======================================================================== */

	/**
	 * Category of the type.
	 */
	private Category category;

	/**
	 * Group of the type.
	 */
	private Group group;

	/**
	 * Retreives category of the item type.
	 * 
	 * @return Category of the item type.
	 */
	public Category getCategory() {
		// do we have a category already?
		return (category == null) ? (category = getProto().category) : category;
	}

	/**
	 * Retreives group of the item type.
	 * 
	 * @return Group of the item type.
	 */
	public Group getGroup() {
		// do we have a group already?
		return (group == null) ? (group = getProto().group) : group;
	}

	/* ======================================================================== */

	/**
	 * Prototype reference.
	 */
	private ItemType proto;

	/**
	 * Retreives (and caches) ItemType prototype.
	 * 
	 * @return ItemType prototype.
	 */
	private ItemType getProto() {
		// do we have a prototype already?
		if (proto != null) return proto;
		synchronized(protos) {
			return proto = protos.get(name);			
		}
	}

	/* ======================================================================== */

	/**
	 * Indicates whether some other ItemType is "equal to" this one.
	 * 
	 * @param obj
	 *            Object to be compared with.
	 * @return True, if the objects are equal.
	 */
	@Override
	public boolean equals(Object obj) {
		// the same object?
		if (this == obj)
			return true;

		// the same type?
		if (obj instanceof ItemType) {
			// the same prototype?
			if (getProto() == ((ItemType) obj).getProto())
				return true;
		}

		return false;
	}

	/**
	 * Returns a hash code value for the object.
	 * 
	 * @return A hash code value for this object.
	 */
	@Override
	public int hashCode() {
		// provide hash of the string name
		return getProto().name.hashCode();
	}

	/* ======================================================================== */

	/**
	 * Public constructor - creates ItemType of the EXTRA category and Group OTHER.
	 * 
	 * @param name
	 *            Type name from GB engine.
	 */
	public ItemType(String name) {
		this.name = name;
		this.category = Category.OTHER;
		this.group = Group.OTHER;
		this.proto = this;
	}

	/**
	 * Prototypes constructor.
	 */
	private ItemType(String name, Category category, Group group) {
		this.name = name;
		this.category = category;
		this.group = group;
		this.proto = this;
	}

	/* ======================================================================== */

	/**
	 * Proto-constructor.
	 * 
	 * @param category
	 *            Category of the item.
	 * @param group
	 *            Group of the item.
	 * @param utNames
	 *            Names of the item in UT engine.
	 * @return Prototype of known ItemType.
	 */
	public static ItemType MakePrototype(Category category,
			Group group, String[] utNames) {
		ItemType type;
		synchronized(protos) {
			// create new itemtype prototype
			type = new ItemType(utNames[0], category, group);
			// register the itemtype prototype
			for (String utName : utNames)
				protos.put(utName, type);
			// C'est la vie..
			if (category != null) {
				CATEGORIES.get(category).add(type);
			}
			if (group != null) {
				GROUPS.get(group).add(type);
			}
		}
		return type;
	}

	/**
	 * Retrieves an ItemType for the specified item type name.
	 * @param utName e.g. Item.getType()
	 * @return
	 */
	public static ItemType getItemType(String utName) {
		ItemType type;
		synchronized(protos) {
			type = protos.get(utName);
			if (type != null) return type;
			
			type = new ItemType(utName);
			protos.put(utName, type);
		}
		return type;
	}

	public String getName() {
		return name;
	}

}