package cz.cuni.amis.pogamut.udk.server;

import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import junit.framework.Assert;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import cz.cuni.amis.pogamut.base.agent.IAgentId;
import cz.cuni.amis.pogamut.base.agent.impl.AbstractAgent;
import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.agent.params.impl.RemoteAgentParameters;
import cz.cuni.amis.pogamut.base.agent.state.WaitForAgentStateChange;
import cz.cuni.amis.pogamut.base.agent.state.level0.IAgentState;
import cz.cuni.amis.pogamut.base.agent.state.level2.IAgentStateRunning;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.utils.Pogamut;
import cz.cuni.amis.pogamut.base.utils.PogamutProperty;
import cz.cuni.amis.pogamut.udk.factory.guice.remoteagent.UDKServerFactory;
import cz.cuni.amis.pogamut.udk.factory.guice.remoteagent.UDKServerModule;
import cz.cuni.amis.pogamut.udk.server.IUDKServer;
import cz.cuni.amis.pogamut.udk.server.exception.UCCStartException;
import cz.cuni.amis.pogamut.udk.server.impl.UDKServer;
import cz.cuni.amis.pogamut.udk.utils.PogamutUDKProperty;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper.UCCWrapperConf;

/**
 * Testing the {@link UDKServer#setGameMap(String)} by changing maps on the UCCWrapper.
 */
public class UDKTest07_UDKServer_SetGameMap {

	private static final Level LOG_LEVEL = Level.INFO;

	protected static boolean useInternalUcc = !Pogamut.getPlatform().getProperty(PogamutUDKProperty.POGAMUT_UNREAL_TEST_EXT_SERVER.getKey()).equals("true");
	
	private UCCWrapper uccWrapper;
	
    IUDKServer server;
    UDKServerFactory factory;
   
    @Before
    public void startUp() throws UCCStartException {
    	if (useInternalUcc) {
    		System.out.println("Starting UCC...");
    		uccWrapper = new UCCWrapper(new UCCWrapperConf().setStartOnUnusedPort(false));
    	}
    }

    @After
    public void tearDown() {
    	if (server != null) {
    		server.kill();
    	}
    	if (useInternalUcc) {
	    	System.out.println("Stopping UCC...");
	    	uccWrapper.stop();
    	}
    	System.out.println("Closing PogamutPlatform...");
    	Pogamut.getPlatform().close();
    }
    
    private boolean awaitAgentUp(AbstractAgent agent) {
    	System.out.println("Awaiting server up (timeout 60s)...");
    	IAgentState state = new WaitForAgentStateChange(agent.getState(), IAgentStateRunning.class).await(60000, TimeUnit.MILLISECONDS);
    	return state instanceof IAgentStateRunning;
    }
    
    private void changeMap(IUDKServer server, String map) {
    	if (awaitAgentUp((AbstractAgent)server)) {
    		System.out.println("Changing map to '" + map + "'...");
    		Future<Boolean> future = server.setGameMap(map);
    		try {
    			System.out.println("Waiting for the GBUDK to change the map (60sec timeout).");
    			Boolean result = future.get(99990000, TimeUnit.MILLISECONDS);
        		if (result == null || !result) {
        			Assert.fail("Failed to change map to '" + map + "'.");
        		}
    		} catch (Exception e) {
    			Assert.fail("Failed to change map to '" + map + "'.");
    		}
    	} else {
    		Assert.fail("Failed to connect to GBUDK...");
    	}
	}

    @Test
    public void test01_SetGameMap() {
    	String host = "localhost";
    	int port = 3001;
    	
    	if (useInternalUcc) {
    		host = uccWrapper.getHost();
    		port = uccWrapper.getControlPort();
    	}
        
        factory = new UDKServerFactory(new UDKServerModule());
        
        IAgentId agentId = new AgentId("Test-ChangeMap");
        server = (IUDKServer) factory.newAgent(new RemoteAgentParameters().setAgentId(agentId).setWorldAddress(new SocketConnectionAddress(host, port)));
        AbstractAgent serverAgent = (AbstractAgent)server;
        serverAgent.getLogger().setLevel(LOG_LEVEL);
        serverAgent.getLogger().addDefaultConsoleHandler();
        
        serverAgent.start();
        
        if (!awaitAgentUp(serverAgent)) {
        	Assert.fail("Failed to connect to GBUDK.");
        } else {
        	System.out.println("Connected...");
        }
		
        String[] maps = new String[] {
        		"DM-Deck",
        		"DM-Sanctuary"
        };
        
        try {
	        for (int i = 0; i < maps.length; ++i) {
	        	System.out.println("Change map test, map " + (i+1) + " / " + maps.length + "...");
	        	changeMap(server, maps[i]);
	        }
        } finally {
	        server.kill();
        }
		
        System.out.println("---/// TEST OK ///---");        
    }

	public static void main(String[] args) {
        UDKTest07_UDKServer_SetGameMap test = new UDKTest07_UDKServer_SetGameMap();
        test.startUp();
        test.test01_SetGameMap();
        test.tearDown();
    }
}
