/*
 * Copyright (C) 2012 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.udk.experiments.impl;

import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.udk.experiments.IExperiment;
import cz.cuni.amis.pogamut.udk.experiments.IExperimentRunResult;
import cz.cuni.amis.pogamut.udk.experiments.IExperimentRunner;
import cz.cuni.amis.pogamut.udk.factory.direct.remoteagent.UDKServerFactory;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper.UCCWrapperConf;
import cz.cuni.amis.utils.ExceptionToString;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author Martin Cerny
 */
public abstract class AbstractExperimentRunner<RESULT, PARAMETERS> implements IExperimentRunner<RESULT, PARAMETERS> {
    protected LogCategory log;
    UCCWrapperConf uccConfiguration;
    
    public static long DEFAULT_CLEANUP_TIMEOUT = 3000;
    
    /**
     * Timeout for the experiment in ms
     */
    protected long timeout;
    protected long serverStartupTimeout;
    protected long cleanupTimeout = DEFAULT_CLEANUP_TIMEOUT;
    protected long udkExeSpawnTimeoutWindows = UCCWrapper.DEFAULT_UDK_EXE_SPAWN_TIMEOUT_WINDOWS;
    protected long udkExeSpawnTimeoutUnix = UCCWrapper.DEFAULT_UDK_EXE_SPAWN_TIMEOUT_UNIX;
    protected UDKServerFactory serverFactory;
    protected boolean aggressiveKilling = true; //default for now, seems better for experiments...
    

    public AbstractExperimentRunner(LogCategory log, UCCWrapperConf uccConfiguration, long timeout, long serverStartupTimeout) {
        this.log = log;
        this.uccConfiguration = uccConfiguration;
        this.timeout = timeout;
        this.serverStartupTimeout = serverStartupTimeout;        
    }

    public void setCleanupTimeout(long cleanupTimeout) {
        this.cleanupTimeout = cleanupTimeout;
    }

    public void setServerStartupTimeout(long serverStartupTimeout) {
        this.serverStartupTimeout = serverStartupTimeout;
    }

    public void setTimeout(long timeout) {
        this.timeout = timeout;
    }

    public void setUdkExeSpawnTimeoutUnix(long udkExeSpawnTimeoutUnix) {
        this.udkExeSpawnTimeoutUnix = udkExeSpawnTimeoutUnix;
    }

    public void setUdkExeSpawnTimeoutWindows(long udkExeSpawnTimeoutWindows) {
        this.udkExeSpawnTimeoutWindows = udkExeSpawnTimeoutWindows;
    }

    public void setAggressiveKilling(boolean aggressiveKilling) {
        this.aggressiveKilling = aggressiveKilling;
    }    

    

    protected UCCWrapper configureAndStartUCCWrapper(UCCWrapperConf configuration){
        UCCWrapper uccWrapper = new UCCWrapper(configuration, serverFactory, false);
        uccWrapper.setStartingTimeout(serverStartupTimeout);
        uccWrapper.setUdkExeSpawnTimeoutUnix(udkExeSpawnTimeoutUnix);
        uccWrapper.setUdkExeSpawnTimeoutWindows(udkExeSpawnTimeoutWindows);
        uccWrapper.setAggressiveKilling(aggressiveKilling);
        uccWrapper.start();
        return uccWrapper;
    }

    @Override
    public void cleanup() {
        serverFactory = null;
    }

    @Override
    public void prepare() {
        serverFactory = new UDKServerFactory();
    }

    protected abstract void cleanupServerAfterExperiment();

    protected abstract UCCWrapper getUCCWrapper();

    protected abstract void prepareServerForExperiment(IExperiment<RESULT, PARAMETERS> experiment);

    @Override
    public IExperimentRunResult<RESULT> runExperiment(IExperiment<RESULT, PARAMETERS> experiment) {
        long startTime = System.currentTimeMillis();
        experiment.setLog(log);
        String experimentDescription = experiment.getDescription();
        try {
            log.info("ExperimentRunner: Running experiment " + experimentDescription);
            prepareServerForExperiment(experiment);
            experiment.setServer(getUCCWrapper());
            experiment.startExperiment();
            experiment.getFinished().waitFor(timeout, Boolean.TRUE);
            long runningTime = System.currentTimeMillis() - startTime;
            if (!experiment.getFinished().getFlag()) {
                log.info("ExperimentRunner:  Experiment timed out");
                return ExperimentRunResult.timeout(runningTime, startTime);
            } else {
                if (experiment.isSuccess()) {
                    log.info("ExperimentRunner:  Experiment success");
                    return ExperimentRunResult.succes(experiment.getResult(), runningTime, startTime);
                } else {
                    log.info("ExperimentRunner:  Experiment failed");
                    return ExperimentRunResult.failure("Failed.", runningTime, startTime);
                }
            }
        } catch (Exception ex) {
            long runningTime = System.currentTimeMillis() - startTime;
            log.severe("ExperimentRunner:  Experiment exception: " + ex);
            log.severe(ExceptionToString.process(ex));
            return ExperimentRunResult.exception(ex, runningTime, startTime);
        } finally {
            CleanupThread cleanupThread = new CleanupThread(experiment);
            cleanupThread.start();
            try {
                if(!cleanupThread.cleanupLatch.await(cleanupTimeout, TimeUnit.MILLISECONDS)){
                    log.severe("ExperimentRunner: Waiting for experiment cleanup timed out.");
                    cleanupThread.interrupt();
                }
            } catch (InterruptedException ex) {
                log.severe("ExperimentRunner: Waiting for experiment cleanup interrupted.", ex);                
            }
            cleanupServerAfterExperiment();
        }
    }
    
    private class CleanupThread extends Thread {
        protected CountDownLatch cleanupLatch;
        IExperiment experiment;

        public CleanupThread(IExperiment experiment) {
            super("ExperimentCleanup");
            this.experiment = experiment;
            cleanupLatch = new CountDownLatch(1);            
        }

        @Override
        public void run() {
            experiment.cleanup();
            cleanupLatch.countDown();
        }
        
        
        
        
    }
}
