/*
 * Copyright (C) 2012 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.udk.experiments.impl;

import cz.cuni.amis.pogamut.udk.experiments.ITimeConsumedMeasuring;
import java.util.HashMap;
import java.util.Map;

/**
 * Simple implementation of {@link ITimeConsumedMeasuring}. Uses HashMaps to keep
 * track of running operations. If an operation recurses time is counted from the first
 * operation start to last operation finish uninterrupted.
 * @author Martin Cerny
 */
public class TimeConsumedMeasuring<IDENTIFIER> implements ITimeConsumedMeasuring<IDENTIFIER>{

    Map<IDENTIFIER, Integer> taskRecursionDepth = new HashMap<IDENTIFIER, Integer>();
    Map<IDENTIFIER, Long> taskStartTime = new HashMap<IDENTIFIER, Long>();
    Map<IDENTIFIER, Long> taskConsumedTime = new HashMap<IDENTIFIER, Long>();
    
    protected void addConsumedTime(IDENTIFIER identifier, long milis){
        Long previousTime = taskConsumedTime.get(identifier);
        if(previousTime == null){
            previousTime = 0L;
        }
        taskConsumedTime.put(identifier, previousTime + milis);
    }
    
    @Override
    public synchronized  void taskConsumedTime(IDENTIFIER identifier, long milis) {
        addConsumedTime(identifier, milis);
    }

    @Override
    public synchronized void taskFinished(IDENTIFIER identifier) {
        Integer recursionDepth = taskRecursionDepth.get(identifier);
        if(recursionDepth == null || recursionDepth <= 0){
            throw new IllegalStateException("Cannot finish a task that has not started. Task:" + identifier);
        }
        int newRecursionDepth = recursionDepth - 1;
        if(newRecursionDepth == 0){
            long timeSpent = System.currentTimeMillis() - taskStartTime.get(identifier);
            addConsumedTime(identifier, timeSpent);
        }
        taskRecursionDepth.put(identifier,newRecursionDepth);
    }

    @Override
    public synchronized void taskStarted(IDENTIFIER identifier) {
        Integer recursionDepth = taskRecursionDepth.get(identifier);
        if(recursionDepth == null || recursionDepth <= 0){
            recursionDepth = 0;
            taskStartTime.put(identifier, System.currentTimeMillis());
        }
        
        taskRecursionDepth.put(identifier, recursionDepth + 1);
        
    }

    @Override
    public long getConsumedTime(IDENTIFIER identifier) {
        Long time = taskConsumedTime.get(identifier);
        if(time == null){
            return 0;
        } else {
            return time;
        }
    }

    @Override
    public synchronized void finishAllTasks() {
        long finishTime = System.currentTimeMillis();
        for(Map.Entry<IDENTIFIER, Integer> recDepth : taskRecursionDepth.entrySet()){
            if(recDepth.getValue() > 0){
                recDepth.setValue(0);
                long timeSpent = finishTime - taskStartTime.get(recDepth.getKey());
                addConsumedTime(recDepth.getKey(), timeSpent);
            }
        }
    }
    
    
    
}
