/*
 * Copyright (C) 2011 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.udk.experiments.impl;

import cz.cuni.amis.pogamut.base.agent.state.WaitForAgentStateChange;
import cz.cuni.amis.pogamut.base.agent.state.level1.IAgentStateDown;
import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.udk.bot.IUDKBot;
import cz.cuni.amis.pogamut.udk.communication.messages.gbcommands.ClearBotsDirty;
import cz.cuni.amis.pogamut.udk.experiments.IExperiment;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper.UCCWrapperConf;
import cz.cuni.amis.utils.ExceptionToString;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * A simple experiment runner that reuses the same UDK server for all experiments, it only resets map for each experiment.
 * @author Martin Cerny
 */
public class ReuseServerExperimentRunner<RESULT, PARAMETERS> extends AbstractExperimentRunner<RESULT, PARAMETERS> {

    public static long DEFAULT_MAP_CHANGE_TIMEOUT = 100000;

    private long mapChangeTimeout;

    /**
     * Timeout waiting for bots to die
     */
    private long botDieTimeout = 1000;
    
    private UCCWrapper uccWrapper;

    public ReuseServerExperimentRunner(LogCategory log, long timeout, UCCWrapperConf configuration) {
        this(log, timeout, configuration, UCCWrapper.DEFAULT_START_TIMEOUT);
    }

    public ReuseServerExperimentRunner(LogCategory log, long timeout, UCCWrapperConf configuration, long serverStartTimeout) {
        this(log, timeout, configuration, serverStartTimeout, DEFAULT_MAP_CHANGE_TIMEOUT);
    }

    public ReuseServerExperimentRunner(LogCategory log, long timeout, UCCWrapperConf configuration, long serverStartTimeout, long mapChangeTimeout) {
        super(log, configuration, timeout, serverStartTimeout);
        this.mapChangeTimeout = mapChangeTimeout;
    }

    @Override
    public void prepare() {
        super.prepare();
        uccWrapper = new UCCWrapper(uccConfiguration, serverStartupTimeout, serverFactory);
    }

    @Override
    public void cleanup() {
        if (uccWrapper != null) {
            try {
                uccWrapper.stop();
            } catch (Exception ex) {
                log.severe("Failed to stop UCC.");
                log.severe(ExceptionToString.process(ex));
            }
        }
        super.cleanup();
    }

    @Override
    protected UCCWrapper getUCCWrapper() {
        return uccWrapper;
    }

    @Override
    protected void prepareServerForExperiment(IExperiment<RESULT, PARAMETERS> experiment) {
        String mapName = experiment.getMapName();

        uccWrapper.changeMap(mapName,false, mapChangeTimeout);
    }

    @Override
    protected void cleanupServerAfterExperiment() {
        //take a snapshot of active bots
        List<IUDKBot> activeBots = new ArrayList<IUDKBot>(uccWrapper.getUTServer().getAgents());
        for(IUDKBot bot : activeBots){
            if(bot.getState().getFlag().isNotState(IAgentStateDown.class)){
                bot.kill();
            }
        }
        
        for(IUDKBot bot : activeBots){
            if(bot.getState().getFlag().isNotState(IAgentStateDown.class)){
                new WaitForAgentStateChange(bot.getState(), IAgentStateDown.class).await(botDieTimeout, TimeUnit.MILLISECONDS);
            }            
        }
        
        
        uccWrapper.getUTServer().getAct().act(new ClearBotsDirty());
    }
}
