/*
 * Copyright (C) 2012 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.udk.experiments.impl;

import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.udk.experiments.IExperiment;
import cz.cuni.amis.pogamut.udk.experiments.IExperimentRunResult;
import cz.cuni.amis.pogamut.udk.experiments.IExperimentRunner;
import cz.cuni.amis.pogamut.udk.factory.direct.remoteagent.UDKServerFactory;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper.UCCWrapperConf;
import cz.cuni.amis.utils.ExceptionToString;

/**
 *
 * @author Martin Cerny
 */
public abstract class AbstractExperimentRunner<RESULT, PARAMETERS> implements IExperimentRunner<RESULT, PARAMETERS> {
    protected LogCategory log;
    UCCWrapperConf uccConfiguration;
    
    /**
     * Timeout for the experiment in ms
     */
    protected long timeout;
    protected long serverStartupTimeout;
    protected UDKServerFactory serverFactory;

    public AbstractExperimentRunner(LogCategory log, UCCWrapperConf uccConfiguration, long timeout, long serverStartupTimeout) {
        this.log = log;
        this.uccConfiguration = uccConfiguration;
        this.timeout = timeout;
        this.serverStartupTimeout = serverStartupTimeout;        
    }



    @Override
    public void cleanup() {
        serverFactory = null;
    }

    @Override
    public void prepare() {
        serverFactory = new UDKServerFactory();
    }

    protected abstract void cleanupServerAfterExperiment();

    protected abstract UCCWrapper getUCCWrapper();

    protected abstract void prepareServerForExperiment(IExperiment<RESULT, PARAMETERS> experiment);

    @Override
    public IExperimentRunResult<RESULT> runExperiment(IExperiment<RESULT, PARAMETERS> experiment) {
        long startTime = System.currentTimeMillis();
        experiment.setLog(log);
        String experimentDescription = experiment.getDescription();
        try {
            log.info("ExperimentRunner: Running experiment " + experimentDescription);
            prepareServerForExperiment(experiment);
            experiment.setServer(getUCCWrapper());
            experiment.startExperiment();
            experiment.getFinished().waitFor(timeout, Boolean.TRUE);
            long runningTime = System.currentTimeMillis() - startTime;
            if (!experiment.getFinished().getFlag()) {
                log.info("ExperimentRunner:  Experiment timed out");
                return ExperimentRunResult.timeout(runningTime, startTime);
            } else {
                if (experiment.isSuccess()) {
                    log.info("ExperimentRunner:  Experiment success");
                    return ExperimentRunResult.succes(experiment.getResult(), runningTime, startTime);
                } else {
                    log.info("ExperimentRunner:  Experiment failed");
                    return ExperimentRunResult.failure("Failed.", runningTime, startTime);
                }
            }
        } catch (Exception ex) {
            long runningTime = System.currentTimeMillis() - startTime;
            log.severe("ExperimentRunner:  Experiment exception: " + ex);
            log.severe(ExceptionToString.process(ex));
            return ExperimentRunResult.exception(ex, runningTime, startTime);
        } finally {
            experiment.cleanup();
            cleanupServerAfterExperiment();
        }
    }
    
}
