package cz.cuni.amis.pogamut.release;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import cz.cuni.amis.utils.NullCheck;
import cz.cuni.amis.utils.StopWatch;
import cz.cuni.amis.utils.process.ProcessExecution;
import cz.cuni.amis.utils.process.ProcessExecutionConfig;
import cz.cuni.amis.utils.rewrite.RewriteFiles;
import cz.cuni.amis.utils.rewrite.RewriteFilesConfig;
import cz.cuni.amis.utils.simple_logging.SimpleLogging;

public class PogamutRelease implements Runnable {
	
	private PogamutReleaseConfig config;
	
	private Logger log;

	public PogamutRelease() {
		this.config = new PogamutReleaseConfig();
	}
	
	public PogamutRelease(PogamutReleaseConfig config) {
		NullCheck.check(config, "config");
		this.config = config;
	}
	
	public PogamutRelease(File xmlFile) {
		if (xmlFile == null) throw new IllegalArgumentException("'xmlFile' can't be null!");
		this.config = PogamutReleaseConfig.loadXML(xmlFile);		
	}

	public PogamutReleaseConfig getConfig() {
		return config;
	}

	public void setConfig(PogamutReleaseConfig config) {
		this.config = config;
	}
	
	public Logger getLog() {
		return log;
	}

	public void setLog(Logger log) {
		this.log = log;
	}

	public void run() {
		logInfo("=========================================");
		logInfo("------- EXECUTING POGAMUT RELEASE -------");
		logInfo("=========================================");
		
		if (config.getSteps() == null || config.getSteps().size() == 0) {
			logWarning("No steps defined in the configuration! ABORTING!");
			logWarning("FAILURE!!!");
			return;
		}
		
		List<Double>    times = new ArrayList<Double>(config.getSteps().size());
		List<Boolean> success = new ArrayList<Boolean>(config.getSteps().size());
		
		StopWatch allWatch = new StopWatch();
		StopWatch stepWatch = new StopWatch();
		boolean allSuccess  = true;
		boolean stepSuccess = true;
		boolean itemSuccess = true;
		
		try {
			int i = 1;
			for (PogamutReleaseStep step : config.getSteps()) {
				stepWatch.start();
				stepSuccess = true;
				try {
					if (step.getRewriteFiles() != null && step.getRewriteFiles().size() > 0) {
						int j = 1;
						for (RewriteFilesConfig rewriteConfig : step.getRewriteFiles()) {
							StopWatch itemWatch = new StopWatch();
							itemSuccess = true;
							try {
								logInfo("--- STEP (" + i + " / " + config.getSteps().size() + "): " + step.getId());
								logInfo("------ REWRITE FILES " + j + " / " + step.getRewriteFiles().size());
								RewriteFiles rewrite = new RewriteFiles(rewriteConfig);
								rewrite.setLog(log);
								if (!rewrite.rewrite()) {
									logSevere("FAILURE!!!");
									itemSuccess = false;
									stepSuccess = false;
									allSuccess = false;
									if (step.isStopOnFail()) {
										logWarning("Step.failStop == true, TERMINATING!");
										return;
									}
								} else {
									itemSuccess = true;
								}
								++j;
							} finally {
								times.add(itemWatch.stop());
								success.add(itemSuccess);
							}
						}
					}
					if (step.getProcessExecution() != null && step.getProcessExecution().size() > 0) {
						int j = 1;
						for (ProcessExecutionConfig processConfig : step.getProcessExecution()) {
							StopWatch itemWatch = new StopWatch();
							itemSuccess = true;
							try {
								logInfo("--- STEP (" + i + " / " + config.getSteps().size() + "): " + step.getId());
								logInfo("------ PROCESS EXECUTION " + j + " / " + step.getProcessExecution().size());
								if (processConfig.getTimeout() == null) {
									processConfig.setTimeout((long)(60 * 60 * 1000));
									logInfo("Timeout for process not specified, setting: 1 hour");
								}
								ProcessExecution process = new ProcessExecution(processConfig, log);					
								process.start();
								process.getRunning().waitFor(false);
								
								if (process.isTimeout() || process.getExitValue() == null || process.getExitValue() != 0) {
									// FAILURE
									if (process.isTimeout()) {
										logWarning("Process timed out!");
									} else
									if (process.getExitValue() == null) {
										logWarning("Process failed to return exit value!");
									} else {
										logWarning("Process exit value is " + process.getExitValue() + " != 0!");
									}
									logWarning("FAILURE!!!");
									itemSuccess = false;
									stepSuccess = false;
									allSuccess = false;
									if (step.isStopOnFail()) {
										logWarning("Step.failStop == true, TERMINATING!");
										return;
									}
								} else {
									itemSuccess = true;
								}
								
								++j;
							} finally {
								times.add(itemWatch.stop());
								success.add(itemSuccess);
							}
						}
					}
				} finally {
					double time = stepWatch.stop();
					times.add(time);
					success.add(stepSuccess);
					if (stepSuccess) {
						logInfo("--- STEP (" + i + " / " + config.getSteps().size() + "): " + step.getId() + " [SUCCESS, " + formatTime(time) + "]");
					} else {
						logInfo("--- STEP (" + i + " / " + config.getSteps().size() + "): " + step.getId() + " [FAILURE, " + formatTime(time) + "]");
					}
				}
				
				++i;
			}
		} finally {
			allWatch.stop();
			logInfo("--------------------------");
			logInfo("STATISTICS:");
			if (allSuccess) {
				logInfo("  Result:                   SUCCESS [ " + formatTime(allWatch.time()) + " ]");
			} else {
				logInfo("  Result:                   FAILURE [ " + formatTime(allWatch.time()) + " ]");
			}
			int index = 0;
			int stepIndex = 1;
			for (PogamutReleaseStep step : config.getSteps()) {
				int rewriteCount = step.getRewriteFiles() == null ? 0 : step.getRewriteFiles().size();
				int processCount = step.getProcessExecution() == null ? 0 : step.getProcessExecution().size();
				double stepTime = 0;
				boolean stepSucc = false;
				if (index + rewriteCount + processCount >= success.size()) {
					// PREMATURE TERMINATION -> FAILURE
					stepTime = times.get(times.size()-1);
					stepSucc = success.get(success.size()-1);
				} else {
					stepTime = times.get(index + rewriteCount + processCount);
					stepSucc = success.get(index + rewriteCount + processCount);
				}
				logInfo(" -- Step: " + step.getId());
				if (stepSucc) {
					logInfo("                            SUCCESS [ " + formatTime(stepTime) + " ]");
				} else {
					logInfo("                            FAILURE [ " + formatTime(stepTime) + " ]");
				}
				for (RewriteFilesConfig rewriteConfig : step.getRewriteFiles()) {
					if (index+1 >= times.size()) break;
					if (success.get(index)) {
						logInfo(" ---- Rewrite:              SUCCESS [ " + formatTime(times.get(index)) + " ]");
					} else {
						logInfo(" ---- Rewrite:              FAILURE [ " + formatTime(times.get(index)) + " ]");
					}
					++index;
				}
				for (ProcessExecutionConfig processConfig : step.getProcessExecution()) {
					if (index+1 >= times.size()) break;
					if (success.get(index)) {
						logInfo(" ---- Process:              SUCCESS [ " + formatTime(times.get(index)) + " ]");
					} else {
						logInfo(" ---- Process:              FAILURE [ " + formatTime(times.get(index)) + " ]");
					}
					++index;
				}
				
				++stepIndex;
				index += 1;
			}
			logWarning("--------------------------");
			if (allSuccess) {
				logWarning("PogamutRelease:          SUCCESS [ " + formatTime(allWatch.time()) + " ]");
			} else {
				logWarning("PogamutRelease:          FAILURE [ " + formatTime(allWatch.time()) + " ]");
			}
		}
	}
	
	protected void logInfo(String msg) {
		if (log != null && log.isLoggable(Level.INFO)) {
			log.info(msg);
		}
	}
	
	protected void logWarning(String msg) {
		if (log != null && log.isLoggable(Level.WARNING)) {
			log.warning(msg);
		}
	}
	
	protected void logSevere(String msg) {
		if (log != null && log.isLoggable(Level.SEVERE)) {
			log.severe(msg);
		}
	}
	
	public static String formatTime(double millis) {
		double origMillis = millis;
		
		boolean time = false;
		
		StringBuffer sb = new StringBuffer();
		if (millis > 60 * 60 * 1000) {
			int hours = (int)Math.floor(millis / (60 * 60 * 1000));
			millis -= hours * 60 * 60 * 1000;
			
			sb.append(hours);
			sb.append("h ");
			
			time = true;
		}
		
		if (time || millis > 60 * 1000) {
			int mins = (int)Math.floor(millis / (60 * 1000));
			millis -= mins * 60 * 1000;
			
			String strMins = String.valueOf(mins);
			while (strMins.length() < 2) strMins = "0" + strMins; 
			sb.append(mins);
			sb.append("m ");
			
			time = true;
		}
		
		int secs = (int)Math.floor(millis / (1000));
		millis -= secs * 1000;
		int ms = (int)Math.floor(millis);
		millis -= ms;
		
		String strSecs = String.valueOf(secs);
		while (strSecs.length() < 2) strSecs = "0" + strSecs;
		
		String strMs = String.valueOf(ms);
		while (strMs.length() < 3) strMs = "0" + strMs;
		
		sb.append(strSecs);
		sb.append(".");
		sb.append(strMs);
		
		sb.append("s");
		
		return sb.toString();
	}
	
	// ..........................................
	// ------------------------------------------
	// ==========================================
	// ------------------------------------------
	// ..........................................
	
	public static void main(String[] args) {
		String definition = "PogamutRelease.xml";
		if (args.length > 0) {
			definition = args[0];
			if (definition == null) definition = "PogamutRelease.xml";
		}
		
		SimpleLogging.initLogging();
		SimpleLogging.addFileLogging("PogamutRelease.log");
		
		Logger log = Logger.getAnonymousLogger();
		log.info("---[[ POGAMUT RELEASE ]]---");
		log.info("Loading definition from xml file: " + definition + " --> " + new File(definition).getAbsoluteFile());
		File file = new File(definition);
		if (!file.exists() || !file.isFile()) {
			log.severe("FAILED! Definition file not found at: " + file.getAbsolutePath());
			log.severe("Usage: java -jar PogamutRelease.jar [path-to-definition-xml-file]");
			log.info("---[[ END ]]---");
			System.exit(1);
			return;
		}
		PogamutRelease release;
		try {
			release = new PogamutRelease(file);
		} catch (Exception e) {
			e.printStackTrace();
			log.severe("Usage: java -jar PogamutRelease.jar [path-to-definition-xml-file]");
			log.info("---[[ END ]]---");
			return;
		}
		release.setLog(log);
		log.info("Definition file loaded.");
		release.run();
		
		log.info("---[[ END ]]---");
	}

}
