package cz.cuni.amis.pogamut.defcon.agent.impl;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.LinkedList;
import java.util.Random;

import cz.cuni.amis.pogamut.base.agent.module.LogicModule;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.AnnotationListenerRegistrator;
import cz.cuni.amis.pogamut.base.utils.guice.AgentScoped;
import cz.cuni.amis.pogamut.defcon.agent.DefConAgent;
import cz.cuni.amis.pogamut.defcon.agent.IDefConAgentLogicController;
import cz.cuni.amis.pogamut.defcon.agent.module.logic.DefConAgentLogic;
import cz.cuni.amis.pogamut.defcon.agent.module.sensor.GameInfo;
import cz.cuni.amis.pogamut.defcon.communication.mailbox.IMailBox;
import cz.cuni.amis.pogamut.defcon.communication.mailbox.MailBox;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.GameRunningChanged;
import cz.cuni.amis.pogamut.defcon.communication.worldview.DefConWorldView;
import cz.cuni.amis.pogamut.defcon.communication.worldview.modules.grid.flags.IFlagChecker;
import cz.cuni.amis.pogamut.defcon.communication.worldview.modules.managers.buildings.BuildingsManager;
import cz.cuni.amis.pogamut.defcon.communication.worldview.modules.managers.fleets.FleetsManager;

/**
 * Implements infrastructure for a logic of the bot and simplifies distinction between different stages
 * of the game.
 * @author Radek 'Black_Hand' Pibil
 *
 * @param <AGENT> Controlled agent's type.
 */
@AgentScoped
public class DefConAgentLogicController<AGENT extends DefConAgent>
		extends DefConAgentController<AGENT> implements
		IDefConAgentLogicController<AGENT, LogicModule<AGENT>> {

	protected DefConAgentLogic<AGENT> logicModule;
	
	private AnnotationListenerRegistrator listenerRegistrator;
	
	private Method logicMethod;
	private Method gameLogic;
	private Method preGameLogic;
	private Method firstGameLogic;

	protected DefConWorldView worldview;
	protected GameInfo gameInfo;
	
	private final Random random = new Random();
	
	private final IMailBox mailBox = new MailBox();

	private final LinkedList<ILogicUpdateListener> logicUpdateListeners = new LinkedList<ILogicUpdateListener>();

	protected FleetsManager fleetsManager;

	protected BuildingsManager buildingsManager;

	protected IFlagChecker flagChecker;

	/**
	 * Assigns logicMethod, which gets executed in logic().
	 * This could be either gameLogic() or preGameLogic(). 
	 */
	public DefConAgentLogicController() {
		try {
			preGameLogic = this.getClass().getMethod(
					"preGameLogic",
					new Class[] {});
			logicMethod = preGameLogic;
			gameLogic = this.getClass().getMethod(
					"gameLogicWorker",
					new Class[] {});
			firstGameLogic = DefConAgentLogicController.class
				.getDeclaredMethod("firstGameLogicWorker", new Class[]{});
		} catch (SecurityException e) {
			e.printStackTrace();
			agent.stop();
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
			agent.stop();
		}
	}

	/**
	 * Annotation-based event listener registrator cannot handle inheritance, so we have to use
	 * this old styled method.
	 */
	public IWorldEventListener<GameRunningChanged> gameStartedListener =
		new IWorldEventListener<GameRunningChanged>() {
			
			@Override
			public void notify(GameRunningChanged event) {
				if (event.getRunning()) {
					try {						
						logicMethod = firstGameLogic;
					} catch (SecurityException e) {
						e.printStackTrace();
						agent.stop();
					}
				}
			}
		};

	/**
	 * Registers anonymous listeners and creates logic module.
	 */
	@Override
	public void initializeController(AGENT agent) {
		super.initializeController(agent);
		logicModule = new DefConAgentLogic<AGENT>(agent, this);
		this.worldview = agent.getWorldView();
		this.gameInfo = this.worldview.getGameInfo();
		listenerRegistrator = new AnnotationListenerRegistrator(this,
				worldview, getLog());
		worldview.addEventListener(
				GameRunningChanged.class,
				gameStartedListener);		
		listenerRegistrator.addListeners();
	}

	@Override
	public long getLogicInitializeTime() {
		return 120000;
	}

	@Override
	public long getLogicShutdownTime() {
		return 120000;
	}	

	@Override
	public void beforeFirstLogic() {
	}
	
	/**
	 * Gets executed periodically. Here we call a method that is currently
	 * stored in {@link #logicMethod} field.
	 */
	@Override
	public void logic() {
		try {
			worldview.updateProducer();
			logicMethod.invoke(this, new Object[] {});
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
			agent.stop();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
			agent.stop();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
			agent.stop();
		}
	}	
	
	/**
	 * Executed instead of {@link #gameLogic} while still in lobby.
	 * This is what you could use to talk while the game is not running yet.
	 * Defcon API is not entirely clear if this is OK, so if you run into any problems with it,
	 * we would be glad to hear about them.
	 */
	public void preGameLogic() {}
	
	/**
	 * Executed the first tick instead of {@link #gameLogic}.
	 */		
	public void firstGameLogic() {
		gameLogic();
	}
	
	private final void firstGameLogicWorker() {
		logicMethod = gameLogic;
		firstGameLogic();
	}

	/**
	 * Executed periodically while still in game.
	 * This is THE method for your AI. Override it to make your bot do something else than (possibly)
	 * a reaction to incoming events.
	 */	
	public void gameLogic() {}

	/**
	 * Updates listeners, then calls gameLogic();
	 */
	public final void gameLogicWorker() {
		for (ILogicUpdateListener listener : logicUpdateListeners) {
			listener.update();
		}
		gameLogic();
	}

	@Override
	public void logicInitialize(LogicModule<AGENT> logicModule) {
		this.logicModule = (DefConAgentLogic<AGENT>) logicModule;
	}

	@Override
	public void logicShutdown() {
	}

	/**
	 * Registers a listener from updates.
	 * 
	 * @param updateListener
	 */
	public void addGameLogicListener(ILogicUpdateListener updateListener) {
		if (updateListener != null)
			logicUpdateListeners.add(updateListener);
	}

	/**
	 * Unregisters a listener from updates.
	 * 
	 * @param updateListener
	 */
	public void unregisterUpdates(ILogicUpdateListener updateListener) {
		if (logicUpdateListeners.contains(updateListener))
			logicUpdateListeners.remove(updateListener);
	}

	/**
	 * Returns a random number generator.
	 * 
	 * @return
	 */
	public final Random getRandom() {
		return random;
	}

	/**
	 * Returns the mailbox.
	 * 
	 * @return
	 */
	public final IMailBox getMailBox() {
		return mailBox;
	}

	/**
	 * Returns flag checker used for map analysis.
	 * 
	 * @return
	 */
	public IFlagChecker getFlagChecker() {
		return flagChecker;
	}

	public final FleetsManager getFleetsManager() {
		return fleetsManager;
	}

	public final BuildingsManager getBuildingsManager() {
		return buildingsManager;
	}

}
