package cz.cuni.amis.pogamut.defcon.communication.worldview;

import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.util.BitSet;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.defcon.agent.module.sensor.GameInfo;
import cz.cuni.amis.pogamut.defcon.communication.worldview.modules.grid.flags.BasicFlag;
import cz.cuni.amis.pogamut.defcon.consts.UnitType;
import cz.cuni.amis.pogamut.defcon.utils.Pair;

/**
 * Uses ingame representation of to provide info about it.
 * 
 * @author Radek 'Black_Hand' Pibil
 * 
 */
public class NativeMapSource extends AbstractMapSource {

	private final SortedMap<Integer, Pair<BitSet, BitSet>> unifiedTerritories = new TreeMap<Integer, Pair<BitSet, BitSet>>();
	private final BitSet sailable;
	private final Pair<BitSet, BitSet> ownTerritories =
			new Pair<BitSet, BitSet>();
	private final SortedMap<Integer, Pair<BitSet, BitSet>> enemyTerritories =
			new TreeMap<Integer, Pair<BitSet, BitSet>>();
	private final Logger log;

	private final double X_SPAN = 180d;
	private final double Y_SPAN = 100d;

	private GameInfo gameInfo;

	public NativeMapSource(GameInfo gameInfo) {
		this(gameInfo, null);
	}

	public NativeMapSource(GameInfo gameInfo, Logger log) {
		this.gameInfo = gameInfo;

		this.log = log;

		this.sailable = cache(BasicFlag.SEA);

		testOutput("sailable", sailable);


		int[] teamIds = gameInfo.getTeamIds();

		for (int teamId : teamIds) {
			if (teamId == gameInfo.getOwnTeamId()) {
				ownTerritories.first = cache(BasicFlag.OWN_PLACEABLE_SEA);
				ownTerritories.second = cache(BasicFlag.OWN_PLACEABLE_LAND);

				testOutput("own_sea", ownTerritories.first);
				testOutput("own_land", ownTerritories.second);

				unifiedTerritories.put(teamId, ownTerritories);
			}
			else {
				Pair<BitSet, BitSet> pair = new Pair<BitSet, BitSet>();
				pair.first = cacheEnemy(BasicFlag.ENEMY_PLACEABLE_SEA, teamId);
				pair.second = cacheEnemy(BasicFlag.ENEMY_PLACEABLE_LAND, teamId);

				unifiedTerritories.put(teamId, pair);
				enemyTerritories.put(teamId, pair);

				testOutput("enemy_sea", pair.first);
				testOutput("enemy_land", pair.second);
			}
		}
	}

	private void testOutput(String filename, BitSet flags) {
		try {
			BufferedWriter ostream = new BufferedWriter(new FileWriter(
					filename + ".def"));

			for (double i = -X_SPAN; i < X_SPAN; ++i) {
				for (double j = -Y_SPAN; j < Y_SPAN; ++j) {
					int index = convert2dIndicesTo1d(i, j);
					ostream.write((flags.get(index) ? "X" : " "));
				}
				ostream.write("\n");
			}
			ostream.close();
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	/**
	 * Caches the map for a given flag.
	 * 
	 * @param flag
	 * @return
	 */
	private BitSet cache(BasicFlag flag) {
		BitSet bit_set = new BitSet(360 * 200);
		for (double i = -X_SPAN; i < X_SPAN; ++i) {
			for (double j = -Y_SPAN; j < Y_SPAN; ++j) {
				int index = convert2dIndicesTo1d(i, j);
				boolean isPresent = hasFlagWorker(
						i,
						j,
						flag);
				bit_set.set(index, isPresent);
			}
		}
		return bit_set;
	}

	/**
	 * Caches the map for a given flag and given teamId.
	 * 
	 * @param flag
	 * @param teamId
	 * @return
	 */
	private BitSet cacheEnemy(BasicFlag flag, int teamId) {
		
		boolean sea;
		
		switch (flag) {
			case ENEMY_PLACEABLE_LAND:
				sea = false;
				break;
			case ENEMY_PLACEABLE_SEA:
				sea = true; 
				break;
			default:
				return null;
		}
		
		BitSet bit_set = new BitSet(360 * 200);
		for (double i = -X_SPAN; i < X_SPAN; ++i) {
			for (double j = -Y_SPAN; j < Y_SPAN; ++j) {
				int index = convert2dIndicesTo1d(i, j);
				boolean isPresent = hasEnemyTerritoryFlagWorker(
						i,
						j,
						teamId,
						sea);
				bit_set.set(index, isPresent);
			}
		}
		return bit_set;
	}

	/**
	 * Converts 2d indices to an index into linearized 2D array.
	 * 
	 * @param x
	 * @param y
	 * @return
	 */
	private final int convert2dIndicesTo1d(double x, double y) {
		return (int) ((x + X_SPAN) * X_SPAN) + (int) (y + Y_SPAN);
	}

	/**
	 * Used during caching. Returns true if the flag is present on the given
	 * location.
	 * 
	 * @param x
	 * @param y
	 * @param flag
	 * @return
	 */
	private boolean hasFlagWorker(double x, double y, BasicFlag flag) {
		if (x < -X_SPAN || x > X_SPAN || y < -Y_SPAN || y > Y_SPAN)
			return false;

		switch (flag) {
			case SEA:
				return gameInfo.isValidTerritory(-1, x, y, true);
			case LAND:
				return gameInfo.isValidTerritory(-1, x, y, false);
			case OWN_TERRITORY: {
				int id = gameInfo.getOwnTeamId();
				return gameInfo.isValidTerritory(id, x, y, false)
						|| gameInfo.isValidTerritory(id, x, y, true);
			}
			case ENEMY_TERRITORY: {
				Boolean ok = false;
				for (int id : gameInfo.getEnemyTeamIds()) {
					if (ok = hasEnemyTerritoryFlagWorker(x, y, id))
						break;
				}
				return ok;
			}
			case OWN_PLACEABLE_LAND: {
				return gameInfo.isValidPlacementLocation(x, y, UnitType.RADAR);
			}
			case OWN_PLACEABLE_SEA:
				return gameInfo.isValidPlacementLocation(
						x, y, UnitType.BATTLE_SHIP);
			case ENEMY_PLACEABLE_LAND: {
				for (int id : gameInfo.getEnemyTeamIds()) {
					if (hasEnemyTerritoryFlagWorker(x, y, id, false))
						return true;
				}
				return false;
			}
			case ENEMY_PLACEABLE_SEA: {
				for (int id : gameInfo.getEnemyTeamIds()) {
					if (hasEnemyTerritoryFlagWorker(x, y, id, true))
						return true;
				}
				return false;

			}
		}
		return false;
	}

	/**
	 * Used during caching. Returns true if the given location is a valid
	 * territory for the given teamId.
	 * 
	 * @param x
	 * @param y
	 * @param enemyId
	 * @return
	 */
	private boolean hasEnemyTerritoryFlagWorker(double x, double y, int enemyId) {
		return gameInfo.isValidTerritory(enemyId, x, y, false) ||
					gameInfo.isValidTerritory(enemyId, x, y, true);
	}

	/**
	 * Used during caching. Returns true if the given location is a valid
	 * territory for the given teamId.
	 * 
	 * @param x
	 * @param y
	 * @param enemyId
	 * @param seaArea
	 *            if true then checks, whether [x, y] is a sea territory
	 * @return
	 */
	private boolean hasEnemyTerritoryFlagWorker(double x, double y,
			int enemyId,
			boolean seaArea) {
		return gameInfo.isValidTerritory(enemyId, x, y, seaArea);
	}

	@Override
	public boolean hasFlag(Location location, BasicFlag flag) {
		return hasFlag(location.getX(), location.getY(), flag);
	}

	@Override
	public boolean hasFlag(double x, double y, BasicFlag flag) {
		if (x < -X_SPAN || x > X_SPAN || y < -Y_SPAN || y > Y_SPAN)
			return false;

		int index = convert2dIndicesTo1d(x, y);
		switch (flag) {
			case SEA: {
				return sailable.get(index);
			}
			case LAND: {
				return !sailable.get(index);
			}
			case OWN_TERRITORY: {
				return ownTerritories.first.get(index)
						|| ownTerritories.second.get(index);
			}
			case OWN_PLACEABLE_SEA: {
				return ownTerritories.first.get(index);
			}
			case OWN_PLACEABLE_LAND: {
				return ownTerritories.second.get(index);
			}
			case ENEMY_TERRITORY: {
				for (Pair<BitSet, BitSet> enemyArea : enemyTerritories.values()) {
					if (enemyArea.first.get(index)
							|| enemyArea.second.get(index))
						return true;
				}
				return false;
			}
			case ENEMY_PLACEABLE_SEA: {
				for (Pair<BitSet, BitSet> enemyArea : enemyTerritories.values()) {
					if (enemyArea.first.get(index))
						return true;
				}
				return false;
			}
			case ENEMY_PLACEABLE_LAND: {
				for (Pair<BitSet, BitSet> enemyArea : enemyTerritories.values()) {
					if (enemyArea.second.get(index))
						return true;
				}
				return false;
			}

		}
		return false;
	}

	@Override
	public boolean hasEnemyTerritoryFlag(Location location, int enemyId) {
		return hasEnemyTerritoryFlag(location.getX(), location.getY(), enemyId);
	}

	@Override
	public boolean hasEnemyTerritoryFlag(double x, double y, int enemyId) {
		int index = convert2dIndicesTo1d(x, y);
		if (enemyTerritories.get(enemyId).first.get(index)
				|| enemyTerritories.get(enemyId).second.get(index))
			return true;
		else
			return false;
	}

	@Override
	public boolean hasEnemyTerritoryFlag(Location location, int enemyId,
			boolean seaArea) {
		return hasEnemyTerritoryFlag(
				location.getX(),
				location.getY(),
				enemyId,
				seaArea);
	}

	@Override
	public boolean hasEnemyTerritoryFlag(double x, double y, int enemyId,
			boolean seaArea) {
		int index = convert2dIndicesTo1d(x, y);

		if (seaArea) {
			return enemyTerritories.get(enemyId).first.get(index);
		} else {
			return enemyTerritories.get(enemyId).second.get(index);
		}
	}
}
