package cz.cuni.amis.pogamut.defcon.utils.closestpoints;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.defcon.agent.module.sensor.GameInfo;
import cz.cuni.amis.pogamut.defcon.base3d.worldview.object.DefConLocation;
import cz.cuni.amis.pogamut.defcon.utils.Pair;
import cz.cuni.amis.pogamut.defcon.utils.quadtree.QuadTree;
import cz.cuni.amis.pogamut.defcon.utils.quadtree.QuadTreeBFSIterator;
import cz.cuni.amis.pogamut.defcon.utils.quadtree.QuadTreeNode;
import cz.cuni.amis.pogamut.defcon.utils.quadtree.WidthLimitedQuadTreePostorderIterator;

/**
 * Collects the reasonably good points for fleet spawn.
 * 
 * @author Radek 'Black_Hand' Pibil
 * 
 */
public class ClosestPointsLookUp {

	private GameInfo gameInfo;
	private Logger log;
	private SortedMap<Integer, Pair<List<QuadTree>, List<QuadTree>>> enemyQuadTrees;
	private SortedMap<Integer, Pair<List<QuadTree>, List<QuadTree>>> ownQuadTrees;

	private boolean isFinished = false;

	private Iterator<Entry<Integer, Pair<List<QuadTree>, List<QuadTree>>>> enemyTerritoryIterator;
	private Iterator<QuadTree> enemyQuadTreeIterator;
	private Iterator<Entry<Integer, Pair<List<QuadTree>, List<QuadTree>>>> ownTerritoriesIterator;

	private Entry<Integer, Pair<List<QuadTree>, List<QuadTree>>> enemyTerritory;
	private QuadTree enemyQuadTree;
	private int enemyId;

	private final SortedMap<Integer, SortedMap<Integer, List<ClosestPoints>>> closestPointsToEnemyTerritories =
			new TreeMap<Integer, SortedMap<Integer, List<ClosestPoints>>>();
	private SortedMap<Integer, List<ClosestPoints>> closestPointsToAnEnemyTerritory;
	private int pointsLimit;

	public static final SortedMap<Integer, Pair<List<QuadTree>, List<QuadTree>>> prepareEnemyQuadTrees(
			SortedMap<Integer, SortedMap<Integer, Pair<List<QuadTree>, List<QuadTree>>>> enemyQuadTrees) {

		SortedMap<Integer, Pair<List<QuadTree>, List<QuadTree>>> output = new TreeMap<Integer, Pair<List<QuadTree>, List<QuadTree>>>();

		for (SortedMap<Integer, Pair<List<QuadTree>, List<QuadTree>>> territory : enemyQuadTrees
				.values()) {
			output.putAll(territory);
		}

		return output;
	}

	public ClosestPointsLookUp(
			GameInfo gameInfo,
			Logger log,
			SortedMap<Integer, Pair<List<QuadTree>, List<QuadTree>>> enemyQuadTrees,
			SortedMap<Integer, Pair<List<QuadTree>, List<QuadTree>>> ownQuadTrees) {

		this(gameInfo, log, enemyQuadTrees, ownQuadTrees, 20);
	}

	public ClosestPointsLookUp(
			GameInfo gameInfo,
			Logger log,
			SortedMap<Integer, Pair<List<QuadTree>, List<QuadTree>>> enemyQuadTrees,
			SortedMap<Integer, Pair<List<QuadTree>, List<QuadTree>>> ownQuadTrees,
			int pointsLimit) {

		this.gameInfo = gameInfo;
		this.log = log;
		this.enemyQuadTrees = enemyQuadTrees;
		this.ownQuadTrees = ownQuadTrees;
		this.enemyTerritoryIterator = enemyQuadTrees.entrySet().iterator();
		this.pointsLimit = pointsLimit;
	}

	/**
	 * Finds or ticks looking for {@link #pointsLimit} for pairs of closest
	 * points to a given pair of own and enemy territories
	 * 
	 * @return SortedMap of enemies, their territories, my territories (in this
	 *         order) of closest points.
	 */
	public ClosestPointsManager getTickedClosestPoints() {
		tickGetClosestPointsWorker();
		return (isFinished) ? new ClosestPointsManager(
				closestPointsToEnemyTerritories, log)
				: null;
	}

	/**
	 * Finds or ticks looking for {@link #pointsLimit} for pairs of closest
	 * points to a given pair of own and enemy territories
	 * 
	 * @return SortedMap of enemies, their territories, my territories (in this
	 *         order) of closest points.
	 */
	public ClosestPointsManager getClosestPoints() {

		for (Entry<Integer, Pair<List<QuadTree>, List<QuadTree>>> enemy_territory : enemyQuadTrees
				.entrySet()) {

			enemyId = gameInfo.getTerritoryOwner(enemy_territory.getKey());

			// log.info("EnemyTerritory " + enemy_territory.getKey() + " of "
			// + enemyId);

			closestPointsToAnEnemyTerritory = new TreeMap<Integer, List<ClosestPoints>>();

			closestPointsToEnemyTerritories.put(
					enemy_territory.getKey(),
					closestPointsToAnEnemyTerritory);


			for (QuadTree enemy_sea : enemy_territory.getValue().first) {
				// log.info("EnemyQTree " + enemy_sea.getRoot().getCenter());

				for (Entry<Integer, Pair<List<QuadTree>, List<QuadTree>>> own_territory : ownQuadTrees
						.entrySet()) {


					List<ClosestPoints> points;

					if (closestPointsToAnEnemyTerritory
							.containsKey(own_territory.getKey())) {
						points = closestPointsToAnEnemyTerritory
								.get(own_territory.getKey());
					} else {
						points = new ArrayList<ClosestPoints>();
						closestPointsToAnEnemyTerritory.put(
								own_territory.getKey(),
								points);
					}

					


					// log.info("MyTerritory: " + own_territory.getKey());

					for (QuadTree own_sea : own_territory.getValue().first) {
						// log.info("MyQTree");

						ClosestPoints result = findClosestPointsBetweenTwoTrees(
								enemy_sea,
								own_sea);

						if (result != null) {
							boolean found = false;
							for (ClosestPoints point : points) {
								if (point.getTarget()
										.equals(result.getTarget())) {
									found = true;
									point.addOrigins(result
											.getCompleteOrigins());
									break;
								}
							}

							if (!found) {
								points.add(result);
							}
						}
					}
				}
			}
		}

		return new ClosestPointsManager(
				closestPointsToEnemyTerritories, log);
	}

	public class ClosestPoints {
		private DefConLocation target;

		private List<DistanceOrigin> completeOrigins = new ArrayList<DistanceOrigin>(
				pointsLimit);

		private List<DefConLocation> origins = new ArrayList<DefConLocation>(
				pointsLimit);

		public ClosestPoints(DefConLocation target) {
			this.target = target;
		}

		public final DefConLocation getTarget() {
			return target;
		}

		public final List<DistanceOrigin> getCompleteOrigins() {
			return completeOrigins;
		}

		public final List<DefConLocation> getOrigins() {
			return origins;
		}

		/**
		 * Only for first add.
		 * 
		 * @param origins
		 */
		public void initOrigins(Queue<DistanceOrigin> origins) {

			if (origins == null)
				throw new IllegalArgumentException("Origins cannot be null.");

			while (this.completeOrigins.size() < pointsLimit
					&& !origins.isEmpty()) {

				DistanceOrigin origin = origins.poll();
				this.completeOrigins.add(origin);
				this.origins.add(origin.getOrigin());
			}
		}

		/**
		 * THE origins LIST MUST BE ORDERED!
		 * 
		 * @param origins
		 */
		public void addOrigins(List<DistanceOrigin> origins) {

			if (origins == null)
				throw new IllegalArgumentException("Origins cannot be null.");

			ArrayList<DistanceOrigin> tmp = new ArrayList<DistanceOrigin>(10);

			this.origins.clear();

			int completeOriginsIndex = 0;
			int originsIndex = 0;
			int counter = 0;
			while (counter < pointsLimit && originsIndex < origins.size()
					&& completeOriginsIndex < completeOrigins.size()) {

				DistanceOrigin origin;

				if (origins.get(originsIndex).compareTo(
						completeOrigins.get(completeOriginsIndex)) < 0) {
					origin = origins.get(originsIndex);
					++originsIndex;
				} else {
					origin = completeOrigins.get(completeOriginsIndex);
					++completeOriginsIndex;
				}

				tmp.add(origin);
				this.origins.add(origin.getOrigin());
				++counter;
			}

			this.completeOrigins = tmp;
		}

	}

	public class DistanceOrigin implements Comparable<DistanceOrigin> {
		public double distance;
		public DefConLocation origin;

		public DistanceOrigin(double distance, DefConLocation origin) {
			this.distance = distance;
			this.origin = origin;
		}

		public final double getDistance() {
			return distance;
		}

		public final DefConLocation getOrigin() {
			return origin;
		}

		@Override
		public int compareTo(DistanceOrigin o) {
			return Double.compare(distance, o.distance);
		}
	}

	private ClosestPoints findClosestPointsBetweenTwoTrees(
			QuadTree enemyTerritory, QuadTree myTerritory) {

		QuadTreeNode outer_closest = null;

		if (enemyTerritory.getRoot().getSize() <= 5d)
			return null;

		if (myTerritory.getRoot().getSize() <= gameInfo
				.getFleetDiameter(6))
			return null;

		Iterator<QuadTreeNode> nodeIterator = new QuadTreeBFSIterator(
				enemyTerritory);
		while (nodeIterator.hasNext()) {

			QuadTreeNode node = nodeIterator.next();

			if (!gameInfo.isValidTerritory(
					enemyId,
					node.getCenter(),
					true))
				continue;

			outer_closest = node;

			// PogamutJBotSupport.writeToConsole("enemyId: " + enemyId + " "
			// + node.getCenter());

			break;
		}

		if (outer_closest == null)
			return null;

		ClosestPoints closest = new ClosestPoints(new DefConLocation(
				outer_closest.getCenter()));
		PriorityQueue<DistanceOrigin> origins = new PriorityQueue<DistanceOrigin>();

		WidthLimitedQuadTreePostorderIterator inner_iter =
						new WidthLimitedQuadTreePostorderIterator(
								myTerritory,
								gameInfo
										.getFleetDiameter(6));

		while (inner_iter.hasNext()) {
			QuadTreeNode inner_node = inner_iter.next();

			if (!inner_node.isLabeled() || !gameInfo.isValidFleetPlacement(
						inner_node.getCenter(),
						6)) {
				continue;
			}

			double dist = gameInfo.getSailDistance(
						inner_node.getCenter(),
						outer_closest.getCenter());

			origins.add(new DistanceOrigin(dist, new DefConLocation(inner_node
					.getCenter())));
			/*
			 * log.info("Distance: " + dist + " " + inner_node.getCenter() + " "
			 * + outer_closest.getCenter());
			 */
		}

		closest.initOrigins(origins);

		return closest;
	}

	private void tickGetClosestPointsWorker() {
		if (ownTerritoriesIterator != null
				&& ownTerritoriesIterator.hasNext()) {

			Entry<Integer, Pair<List<QuadTree>, List<QuadTree>>> ownTerritory = ownTerritoriesIterator
					.next();

			log.info("MyTerritory: " + ownTerritory.getKey());

			List<ClosestPoints> points;
			if (!closestPointsToAnEnemyTerritory.containsKey(ownTerritory
					.getKey())) {
				points = new ArrayList<ClosestPoints>();

				closestPointsToAnEnemyTerritory.put(
						ownTerritory.getKey(),
						points);
			} else {
				points = closestPointsToAnEnemyTerritory.get(ownTerritory
						.getKey());
			}

			for (QuadTree ownSea : ownTerritory.getValue().first) {

				log.info("MyQTree");

				ClosestPoints return_value = findClosestPointsBetweenTwoTrees(
						enemyQuadTree,
						ownSea);

				if (return_value != null) {
					boolean found = false;
					for (ClosestPoints point : points) {
						if (point.getTarget().equals(return_value.getTarget())) {
							found = true;
							point.addOrigins(return_value.getCompleteOrigins());
							break;
						}
					}

					if (!found) {
						points.add(return_value);
					}
				}
			}

			return;
		} else if (enemyQuadTreeIterator != null &&
				enemyQuadTreeIterator.hasNext()) {

			enemyQuadTree = enemyQuadTreeIterator.next();

			log.info("EnemyQTree " + enemyQuadTree.getRoot().getCenter());

			ownTerritoriesIterator = ownQuadTrees.entrySet().iterator();

			tickGetClosestPointsWorker();
		} else if (enemyTerritoryIterator.hasNext()) {

			enemyTerritory = enemyTerritoryIterator.next();

			enemyId = gameInfo.getTerritoryOwner(enemyTerritory.getKey());

			log.info("EnemyTerritory " + enemyTerritory.getKey() + " of "
					+ enemyId);

			// do only sea
			enemyQuadTreeIterator = enemyTerritory.getValue().first
					.iterator();


			closestPointsToAnEnemyTerritory = new TreeMap<Integer, List<ClosestPoints>>();

			closestPointsToEnemyTerritories.put(
					enemyTerritory.getKey(),
					closestPointsToAnEnemyTerritory);

			tickGetClosestPointsWorker();

		}

		if (!enemyTerritoryIterator.hasNext()) {

			if (enemyQuadTreeIterator == null
					|| !enemyQuadTreeIterator.hasNext()) {

				if (ownTerritoriesIterator == null
						|| !ownTerritoriesIterator.hasNext()) {
					isFinished = true;
				}
			}

		}
	}
}
