package cz.cuni.amis.pogamut.sposh.engine;

import cz.cuni.amis.pogamut.sposh.elements.Goal;
import cz.cuni.amis.pogamut.sposh.elements.Sense;
import cz.cuni.amis.pogamut.sposh.elements.Triggers;
import cz.cuni.amis.pogamut.sposh.executor.IWorkExecutor;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

/**
 * Executor that decicdes if goal or trigger are fulfilled. That happens only
 * is all its senses are fired.
 * 
 * @author Honza
 */
final class SenseListExecutor extends AbstractExecutor {
    private List<SenseExecutor> sensesExecutors = new ArrayList<SenseExecutor>();

    /**
     * Create executor for goal.
     * @param goal null for empty SenseListExecutor or source of senses
     * @param log logger to record actions of this executor, can be null
     */
    SenseListExecutor(Goal goal, VariableContext ctx, Logger log) {
        super(ctx, log);
        if (goal == null) {
            return;
        }

        for (Sense sense : goal.getSenses()) {
            sensesExecutors.add(new SenseExecutor(sense, ctx, log));
        }
    }

    /**
     * Create executor for triggers.
     * @param triggers source of senses, can be null, then no senses will be used.
     * @param log logger to record actions of this executor, can be null
     */
    SenseListExecutor(Triggers triggers, VariableContext ctx, Logger log) {
        super(ctx, log);
        if (triggers == null) {
            return;
        }

        for (Sense sense : triggers.getSenses()) {
            sensesExecutors.add(new SenseExecutor(sense, ctx, log));
        }
    }


    /**
     * Create executor for triggers.
     * @param triggers source of senses, can be null, then no senses will be used.
     * @param log logger to record actions of this executor, can be null
     */
    SenseListExecutor(List<Sense> triggers, VariableContext ctx, Logger log) {
        super(ctx, log);

        for (Sense sense : triggers) {
            sensesExecutors.add(new SenseExecutor(sense, ctx, log));
        }
    }

    /**
     * Evaluate all senses until first one fails. 
     * If no senses were specified, consider it a fail.
     * @param defaultReturn what to return if no senses were specified. In trigger true, in goal false
     * @return true if none of senses fails, false if at least one fails.
     */
    public boolean fire(IWorkExecutor workExecuter, boolean defaultReturn) {
        for (SenseExecutor senseExecutor : sensesExecutors) {
            defaultReturn = true;
            if (!senseExecutor.fire(workExecuter)) {
                return false;
            }
        }
        return defaultReturn;
    }

}
