package cz.cuni.amis.pogamut.ut2004.communication.translator;


import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Collection;
import java.util.LinkedList;
import java.util.logging.Level;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;

import com.google.inject.Guice;
import com.google.inject.Injector;
import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.io.xml.DomDriver;

import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.communication.messages.InfoMessage;
import cz.cuni.amis.pogamut.base.communication.translator.IWorldMessageTranslator;
import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldChangeEvent;
import cz.cuni.amis.pogamut.base.utils.logging.AgentLogger;
import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.base.utils.logging.LogPublisher;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.base3d.worldview.object.Velocity;
import cz.cuni.amis.pogamut.unreal.communication.messages.UnrealId;
import cz.cuni.amis.pogamut.ut2004.agent.params.UT2004AgentParameters;
import cz.cuni.amis.pogamut.ut2004.communication.messages.ItemType;
import cz.cuni.amis.pogamut.ut2004.communication.messages.ItemType.Category;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.AliveMessageMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.BombInfoMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.FlagInfoMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfoMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.HelloControlServerHandshake;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ItemCategory;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ItemCategoryEnd;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ItemCategoryStart;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.MapList;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.MapListEnd;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.MapListStart;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPointListEnd;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPointListStart;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.PasswdOk;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Password;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.PlayerListEnd;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.PlayerListStart;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.PlayerMessage;
import cz.cuni.amis.pogamut.ut2004.communication.translator.shared.events.ItemDescriptorObtained;
import cz.cuni.amis.pogamut.ut2004.communication.translator.testplan.ListsWrapperForConversion;
import cz.cuni.amis.pogamut.ut2004.communication.translator.testplan.ListsWrapperForConversionConverter;
import cz.cuni.amis.pogamut.ut2004.communication.translator.testplan.MyUnrealId;
import cz.cuni.amis.pogamut.ut2004.factory.guice.remoteagent.UT2004CommunicationModule;

/**
 * Test case for WorldMessageTranslators.
 * @author Radek 'Black_Hand' Pibil
 *
 */
@RunWith(value = Parameterized.class)
public class Test01_WorldMessageTranslator {
	protected UT2004CommunicationModule module;
	protected LinkedList<InfoMessage> messages = new LinkedList<InfoMessage>();
	protected LinkedList<IWorldChangeEvent> outputEvents = new LinkedList<IWorldChangeEvent>();	
	protected String planFileName;
	protected Class<? extends IWorldMessageTranslator> worldMessageTranslatorClass;
	
	private AgentLogger logger;
	private LogCategory log;
		
	/**
	 * Override this to set a module for Guice.
	 * It has to be set to an IWorldMessageTranslator to be tested.
	 */
	protected void setModule() {
		module = new WorldMessageTranslatorTestModule(worldMessageTranslatorClass);
	}
	
	public AgentLogger getLogger() {
		return logger;
	}
	
	public LogCategory getLog() {
		return log;
	}

	@SuppressWarnings("unchecked")
	public Test01_WorldMessageTranslator(String worldMessageTranslatorClassName,
			String planFileName) {
		
		this.planFileName = planFileName;
				
		try {
			worldMessageTranslatorClass =
				(Class<? extends IWorldMessageTranslator>) Class.forName(worldMessageTranslatorClassName);
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
			fail("Could not find class to test: " + worldMessageTranslatorClassName);
		}
	}
	
	@Test 
	public void WorldMessagesTranslatorTest() {
		doTest();
	}

	/**
	 * Test execution method.
	 */
	private void doTest() {
		
		// load test data
		deserializeTest(planFileName);
				
		setModule();
		assertTrue("Module is null", module != null);
		
		AgentId id = new AgentId("Test");
		
		module.prepareNewAgent(new UT2004AgentParameters().setAgentId(id).setWorldAddress(new SocketConnectionAddress("localhost", 3000))); // address is not used, dummy
		Injector injector = Guice.createInjector(module);
		
		// test subject
		IWorldMessageTranslator testSubject = injector.getInstance(IWorldMessageTranslator.class);
		LogCategory log = new LogCategory("Test");
		
		log.addHandler(new LogPublisher.ConsolePublisher(id));
				
		if (log.isLoggable(Level.INFO)) log.info("Starting WorldMessageTranslatorTest of " + worldMessageTranslatorClass.getName()
				+ " with plan " + planFileName);
		
		// prepare test data		
		LinkedList<IWorldChangeEvent> worldEventQueue = new LinkedList<IWorldChangeEvent>();		
		for (InfoMessage message : messages) {
			
			if (log.isLoggable(Level.INFO)) log.info("Notifying translator: " + message.toString());
			
			IWorldChangeEvent[] worldEvents = null;
			worldEvents = testSubject.processMessage(message);
			
			for (int i = 0; i < worldEvents.length; ++i) {
				
				if (log.isLoggable(Level.INFO)) log.info("Received event: " + worldEvents[i].toString());
				worldEventQueue.add(worldEvents[i]);
			}
		}

		int i = 0;
		for (IWorldChangeEvent event : outputEvents) {
			boolean checkAll = true;
			String expected_message_string = event.toString();
			assertTrue("Too few events returned, those returned were OK", 0 < worldEventQueue.size());
			IWorldChangeEvent receivedWorldEvent = worldEventQueue.remove();
			if (event instanceof ItemDescriptorObtained) {
				if (receivedWorldEvent instanceof ItemDescriptorObtained) checkAll = false;
			}			
			if (checkAll) {				
				assertTrue("Wrong event returned at " + i + " (indexed from 0)\nReturned: " +
						receivedWorldEvent.toString() +
						"\nExpected: " + expected_message_string,
						receivedWorldEvent.getClass().isInstance(event) &&
						receivedWorldEvent.toString().equals(expected_message_string));
			}
			++i;
		}
		assertTrue("More events were returned than expected, preceeding events were OK",
				worldEventQueue.size() == 0);				
	}
	
	/**
	 * You might want to override this in order to achieve better aliasing
	 * for your own plans.
	 * @param xstream
	 */
	protected static void assignAliasesToXStream(XStream xstream) {
		xstream.alias("TestPlan", ListsWrapperForConversion.class);
		xstream.autodetectAnnotations(true);
		xstream.alias("Id", UnrealId.class, MyUnrealId.class);
		xstream.alias("BotId", UnrealId.class, MyUnrealId.class);
		xstream.alias("Holder", UnrealId.class, MyUnrealId.class);
		xstream.autodetectAnnotations(false);
		xstream.aliasPackage("ut2004messages",
				"cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages");
		xstream.aliasPackage("coremessages",
				"cz.cuni.amis.pogamut.base.communication.worldview.objects");
		xstream.aliasPackage("worldview",
				"cz.cuni.amis.pogamut.ut2004.communication.worldview");
		xstream.aliasPackage("translator",
				"cz.cuni.amis.pogamut.ut2004.communication.translator");
		
	}
	
	protected static void assignConvertersToXStream(XStream xstream) {
		xstream.registerConverter(
				new ListsWrapperForConversionConverter());	
	}

	protected void serializeTest(LinkedList<InfoMessage> messages,
			LinkedList<IWorldChangeEvent> defaultOutput, String planFileName) {
		
		assertTrue("planFileName has to contain the name of the plan to execute.",
				planFileName != null && !planFileName.isEmpty());
		
		assertTrue("Input messages list cannot be null or empty.",
				messages != null && !messages.isEmpty());
		
		assertTrue("Expected messages list cannot be null or empty.",
				defaultOutput != null && !defaultOutput.isEmpty());
		
		XStream xstream = new XStream(new DomDriver());
		
		assignConvertersToXStream(xstream);				
		assignAliasesToXStream(xstream);
								
		try {
			FileWriter writer = new FileWriter(
					"target/test-classes/testplans/WorldMessageTranslator/" + worldMessageTranslatorClass.getName() +
					"/" + planFileName);
			
			writer.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
					
			writer.write(xstream.toXML(
					new ListsWrapperForConversion(messages, defaultOutput)));
						
			writer.close();
		} catch (IOException e) {
			e.printStackTrace();
			
			fail("Failed to write plan: " + planFileName);
		}
	}
	
	protected void deserializeTest(String planFileName) {
		
		assertTrue("planFileName has to contain the name of the plan to execute.",
				planFileName != null && !planFileName.isEmpty());	
		
		XStream xstream = new XStream(new DomDriver());

		assignConvertersToXStream(xstream);
		assignAliasesToXStream(xstream);		
								
		try {
			FileReader reader = new FileReader(
					"target/test-classes/testplans/WorldMessageTranslator/" + planFileName);
			
			while (reader.read() != '\n');
			
			ListsWrapperForConversion wrapper =
				(ListsWrapperForConversion) xstream.fromXML(reader);
			
			messages = wrapper.messages;
			outputEvents = wrapper.defaultOutput;
			
			reader.close();
		} catch (IOException e) {			
			e.printStackTrace();
			fail("Failed to read plan file: " + planFileName);
		}
	}
	
	/**
	 * Example test plan
	 * @return a simple test plan
	 */
	public static LinkedList<InfoMessage> testPlan() {
		
		LinkedList<InfoMessage> messages = new LinkedList<InfoMessage>();
		/*
		messages.add(new HelloBotHandshake());
		messages.add(new GameInfo("BotFSMTest", "DM-Test", false, 0, 0, 0, 0, 0, false, false));
		messages.add(new PlayerListStart());
		messages.add(new Player(new UnrealId("debil"), "", "", "", false, null,
				null, null, 0, "", false, 0));
		messages.add(new Player(new UnrealId("blbecek"), "", "", "", false, null,
				null, null, 0, "", false, 0));
		messages.add(new PlayerListEnd());
		messages.add(new ConfigChange(new UnrealId("he"), new UnrealId("co?"), false, false,
				"to jako fakt",	1.0f, false, 10, true, false, false, false, false));
		messages.add(new InitedMessage(new UnrealId("inited"), 100, 150, 200, 20d, 100d,
				0, 0, 2, 1d, 1d, 1d, 1d, 1d, 1d, 10d, 10d, 100d, 50d, 60d, 70d));		
		messages.add(new BotKilled(new UnrealId("debil"), "", "", "", false,
				false, false, false, false));
		messages.add(new Spawn());
		messages.add(new BeginMessage(2d));
		messages.add(new PathListStart("first path"));
		messages.add(new PathListEnd());
		messages.add(new BotKilled(new UnrealId("debil"), "", "", "", false,
				false, false, false, false));
		messages.add(new EndMessage(2d));
			*/	
		
		messages.add(new HelloControlServerHandshake());
		messages.add(new Password());
		messages.add(new PasswdOk());
		messages.add(new GameInfoMessage());
		
		messages.add(new MapListStart());
		messages.add(new MapList("murder"));
		messages.add(new MapList("death"));
		messages.add(new MapList("kill"));
		messages.add(new MapListEnd());
		
		messages.add(new ItemCategoryStart());
		messages.add(new ItemCategory("cheese", new ItemType("dairy product"), Category.HEALTH,
				false, false, false, "", false,
				false, false, false, false, false, false, false, false, 0, 0, 0, 0, 0, 0, 0,
				0, 0, "", 0, 0, 0, "", false, false, false, false, false, false, "", 0, 0, 0,
				0, 0, 0, 0, 0, 0, "", false, false, false, false, false, false, false, false,
				false, 0, 0, 0, 0, 0, 0, 0, 0, 0, "", 0, 0, 0, "", false, false, false, false,
				false, false, "", 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false));
		messages.add(new ItemCategory("cabbage", new ItemType("vegetable"), Category.AMMO,
				false, false, false, "", false,
				false, false, false, false, false, false, false, false, 0, 0, 0, 0, 0, 0, 0,
				0, 0, "", 0, 0, 0, "", false, false, false, false, false, false, "", 0, 0, 0,
				0, 0, 0, 0, 0, 0, "", false, false, false, false, false, false, false, false,
				false, 0, 0, 0, 0, 0, 0, 0, 0, 0, "", 0, 0, 0, "", false, false, false, false,
				false, false, "", 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false));
		messages.add(new ItemCategory("cleese", new ItemType("comedian"), Category.WEAPON,
				false, false, false, "", false,
				false, false, false, false, false, false, false, false, 0, 0, 0, 0, 0, 0, 0,
				0, 0, "", 0, 0, 0, "", false, false, false, false, false, false, "", 0, 0, 0,
				0, 0, 0, 0, 0, 0, "", false, false, false, false, false, false, false, false,
				false, 0, 0, 0, 0, 0, 0, 0, 0, 0, "", 0, 0, 0, "", false, false, false, false,
				false, false, "", 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false));		
		messages.add(new ItemCategoryEnd());
		
		messages.add(new PlayerListStart());
		messages.add(new PlayerMessage(UnrealId.get("debil"), "", "", "", false, null,
				null, null, 0, "", false, 0));
		messages.add(new PlayerMessage(UnrealId.get("blbecek"), "", "", "", false, null,
				null, null, 0, "", false, 0));
		messages.add(new PlayerListEnd());
		
		messages.add(new NavPointListStart());
		messages.add(new NavPointListEnd());
		
		messages.add(new FlagInfoMessage(UnrealId.get("red flag"), null, null, 0, false, false, ""));
		messages.add(new AliveMessageMessage(2d));
		messages.add(new BombInfoMessage(UnrealId.get("what bomb?"), new Velocity(), new Location(),
				UnrealId.get("bomb holder"), 0, false, false, "free"));
		
		return messages;
	}
	
	@Parameters
	public static Collection<String[]> data() {
		
		
	    File mediator_test_dir = new File("target/test-classes/testplans/WorldMessageTranslator/");
	    
	    LinkedList<String[]> output = new LinkedList<String[]>();
	    FilenameFilter filter = new FilenameFilter() {
			
			@Override
			public boolean accept(File dir, String name) {
				return name.endsWith(".xml");
			}
		};
		
		for (String class_dir : mediator_test_dir.list()) {
		    File mediator_tests_for_class_dir =
		    	new File("target/test-classes/testplans/WorldMessageTranslator/" + class_dir + "/");
		    if (!mediator_tests_for_class_dir.isDirectory()) continue;
			if (class_dir.equalsIgnoreCase(".svn")) continue;
		    
		    
		    for (String file : mediator_tests_for_class_dir.list(filter)) {
		    	output.add(new String[] { class_dir,
		    			class_dir + "/" + file });
		    }		    
		}
		return output;
		
	}
}
