package cz.cuni.amis.pogamut.usar2004.agent.module.sensor;

import cz.cuni.amis.pogamut.usar2004.agent.module.datatypes.ClosestObstacle;
import cz.cuni.amis.pogamut.usar2004.agent.module.datatypes.DriveDirection;
import cz.cuni.amis.pogamut.usar2004.agent.module.datatypes.SensorType;
import java.util.List;

/**
 * Sensor message representative for Laser sensor.
 *
 * @author vejmanm
 */
public class SensorLaser extends SuperSensor
{
    //public static final String type="RangeScanner";
    public static final SensorType type = SensorType.LASER_SENSOR;

    /**
     * Ctor. Sensor type describes particular subject about which we want to
     * know about. It is used to distinguish incoming message from the server.
     */
    public SensorLaser()
    {
        super(type);
    }

    /**
     * Returns the usm of values from <b>list</b> from <b>from</b> to <b>to</b>.
     *
     * @param list List to
     * @param from
     * @param to
     * @return Returns sum of list of specified range.
     */
    private double sumList(List<Double> list, int from, int to)
    {
        double sum = 0;
        for(int i = from; i < to; i++)
        {
            sum += list.get(i);
        }
        return sum;
    }

    /**
     * Devides Laser ranges in half and sums both halves. If left is larger,
     * than the obstacle is on the Right. On the Left otherwise.
     *
     * @return Returns Left or right based on Laser range values.
     */
    public ClosestObstacle getClosestObstacleCourse()
    {
        List<Double> Ranges = this.lastMessage.getLaserRanges();
        int half = Ranges.size() / 2;
        double mleft = sumList(Ranges, 0, half);
        double mright = sumList(Ranges, half, Ranges.size());

        if(mleft < mright)
        {
            return ClosestObstacle.LEFT;
        }
        else
        {
            return ClosestObstacle.RIGHT;
        }

    }

    /**
     * Iterates through Laser ranges and returns the min value.
     *
     * @return Returns Closest range in meters from all Laser rays.
     */
    public double getClosestObstacleDistance()
    {
        List<Double> Ranges = this.lastMessage.getLaserRanges();
        double closest = Double.MAX_VALUE;
        for(Double double1 : Ranges)
        {
            if(closest > double1)
            {
                closest = double1;
            }
        }
        return closest;
    }

    /**
     * Devides Laser ranges to thirds and sums up each third. Based on these
     * sums it determines where is the most open space.
     *
     * @param backwardsTreshold If range in front is smaller than this value it
     * is clear there is wall in front of the robot. Therefore it Returns
     * Backwards.
     * @return Returns direction that is most pleasant spacewise.
     */
    public DriveDirection getMostOpenDirection(int backwardsTreshold)
    {
        List<Double> Ranges = this.lastMessage.getLaserRanges();
        int third = Ranges.size() / 3;
        double mright = sumList(Ranges, 0, third - 10);
        double mstraight = sumList(Ranges, third + 10, 2 * (third - 10));
        double mleft = sumList(Ranges, 2 * (third + 10), Ranges.size());


        //if there is an obstacle in front of the sensor robot
        if(mleft + mstraight < backwardsTreshold || mright + mstraight < backwardsTreshold || mstraight < backwardsTreshold / 4)
        {
            return DriveDirection.BACKWARD;
        }
        else if(mstraight > 400)
        {
            return DriveDirection.FORWARD;
        }
        else
        {
            if(mleft >= mright && mleft >= mstraight)
            {
                return DriveDirection.LEFT;
            }
            else if(mright >= mleft && mright >= mstraight)
            {
                return DriveDirection.RIGHT;
            }
            else //most space in the front
            {
                return DriveDirection.FORWARD;
            }
//            if(Math.max(mleft, mright) == mleft)
//            {
//                if(Math.max(mleft, mstraight) == mleft)
//                    return DriveDirection.Left;
//                else
//                    return DriveDirection.Forward;
//            }
//            else
//            {
//                if(Math.max(mright, mstraight) == mright)
//                    return DriveDirection.Right;
//                else
//                    return DriveDirection.Left;
//            }
        }
    }

    /**
     * It is the sensor’s resolution in radians. With FOV, we can calculate the
     * number of the data in the Range segment.
     *
     * @return Returns resolution of the sensor.
     */
    public double getResolution()
    {
        return lastMessage.getResolution();
    }

    /**
     * FOV is the sensor’s field of view in radians.
     *
     * @return Returns field of view in radians.
     */
    public double getFOV()
    {
        return lastMessage.getFOV();
    }

    /**
     * LaserRanges is a series of range values in meters.
     *
     * @param index
     * @return Returns double value representing distance in meters at specified
     * index.
     */
    public double getRangeAt(int index)
    {
        //TODO: discover if index is equal to angle! use FOV knowledge!

        if(index < getRangesSize())
        {
            return getRanges().get(index);
        }
        else
        {
            return -1;
        }
    }

    /**
     * Returns size of Laser range list.
     *
     * @return Returns size of Laser range list.
     */
    public int getRangesSize()
    {
        return lastMessage.getLaserRanges().size();
    }

    /**
     * LaserRanges is a series of range values in meters.
     *
     * @return Returns List of double values representing distance in meters.
     */
    public List<Double> getRanges()
    {
        return lastMessage.getLaserRanges();
    }

    /**
     * Computes avarage value from set of double values obtained from sector of
     * Laser ranges from the middle of the list.
     *
     * @param numOfRays Number of records around the middle one to compute the
     * avarage distance from.
     * @return Avarage distance computed from rays around the middle.
     */
    public double getNMidAvg(int numOfRays)
    {
        return getNMidAvg(numOfRays, 0);
    }

    /**
     * Computes avarage value from set of double values obtained from sector of
     * Laser ranges defined by <b>numOfRays</b>(count) and <b>offset</b>.
     *
     * @param numOfRays Number of records around the middle one to compute the
     * avarage distance from.
     * @param offset Number of rays to skip before computing.
     * @return Returns avarage value from laser ranges defined by specific
     * sector.
     */
    public double getNMidAvg(int numOfRays, int offset)
    {
        if(numOfRays >= getRangesSize())
        {
            numOfRays = getRangesSize();
        }
        double avg = 0;
        int from = getRangesSize() / 2 - numOfRays / 2 + offset;
        int to = getRangesSize() / 2 - numOfRays / 2 + numOfRays + offset;
        if(from < 0)
        {
            to -= (0 - from);
            from = 0;
        }
        else if(to > getRangesSize())
        {
            from -= (to - getRangesSize());
            to = getRangesSize();
        }
        for(int i = from; i < to; i++)
        {
            avg += getRangeAt(i);
        }
        return avg / numOfRays;
    }
}
