package cz.cuni.amis.pogamut.usar2004.agent.module.datatypes;

import cz.cuni.amis.pogamut.usar2004.agent.module.sensor.SuperSensor;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Container of sensor message representatives. Note that this is equvalent with
 * SensorsContainerQueued. The difference is that the non queued version throws
 * out records as soon as new comes along. It writes them over. This is needed
 * when we do not care about precision - when we want to know what is happening
 * right now!
 *
 * @author vejmanm
 */
public class SensorsContainer extends HashMap<String, Map<String, SuperSensor>>
{
    /**
     * Gets sensor message representatives from local hashmap. Returns null if
     * none matches or this hash map is empty.
     *
     * @param type String representing the type of sensor to return.
     * @return Returns List of specified type of Sensor module.
     */
    public List<SuperSensor> getSensorsByType(String type)
    {
        if(this.isEmpty() || type == null)
        {
            return null;
        }
        if(this.containsKey(type))
        {
            return new ArrayList<SuperSensor>(this.get(type).values());
        }
        return null;
    }

    /**
     * Iterates through local hashmap values and seeks match. Returns null if
     * this hash map is empty. Note, that if <B>type</B> = UNKNOWN_SENSOR it
     * returns all unknown sensors.
     *
     * @param type SensorType representing the type of sensor to return.
     * @return Returns List of all sensors that suit input SensorType.
     */
    public List<SuperSensor> getSensorsBySensorType(SensorType type)
    {
        if(this.isEmpty() || type == null)
        {
            return null;
        }
        List<SuperSensor> list = new ArrayList<SuperSensor>();
        for(Map<String, SuperSensor> map : this.values())
        {
            if(map == null || map.isEmpty())
            {
                continue;
            }
            if(map.values().iterator().next().getSensorType() == type)
            {
                list.addAll(map.values());
            }
        }
        return list;
    }

    /**
     * Adds every object that can be casted to initial class to the output list.
     * Note that if feeded with SuperSensor class it will return all available
     * submodules.
     *
     * @param c Class representing the type of which the return list should be.
     * @return Returns a list of eligible objects, that can be casted to Class
     * c.
     */
    public List<SuperSensor> getSensorsByClass(Class c)
    {
        if(c == null)
        {
            return null;
        }
        List<SuperSensor> list = new ArrayList<SuperSensor>();
        for(Map<String, SuperSensor> map : this.values())
        {
            for(SuperSensor sensor : map.values())
            {
                if(c.isInstance(sensor))
                {
                    list.add(sensor);
                }
            }
        }
        return list;
    }

    /**
     * Gets sensor message representatives from local hashmap specified by type
     * and by name. Returns null if none matches or this hash map is empty.
     *
     * @param type String representing the type of sensor to return.
     * @param name String representing the name of sensor to return.
     * @return Returns List of specified type of Sensor module.
     */
    public SuperSensor getSensorByTypeName(String type, String name)
    {
        if(this.isEmpty())
        {
            return null;
        }
        if(this.containsKey(type) && !this.get(type).isEmpty())
        {
            return this.get(type).get(name);
        }
        return null;
    }

    /**
     * For each type of sensor it adds all individuals to the returnee List as a
     * couple (Type, Name)
     *
     * @return returns Map of couples (Type/Name) of non empty sensor
     * representatives.
     */
    public List<MessageDescriptor> getNonEmptyDescription()
    {
        if(this.isEmpty())
        {
            return null;
        }
        List<MessageDescriptor> list = new ArrayList<MessageDescriptor>();
        for(String type : this.keySet())
        {
            for(String name : this.get(type).keySet())
            {
                list.add(new MessageDescriptor(type, name));
            }
        }
        return list;
    }
}
