package cz.cuni.amis.pogamut.udk.agent.navigation;

import java.util.logging.Level;

import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.udk.agent.UDKBotTestController;
import cz.cuni.amis.pogamut.udk.agent.navigation.martinnavigator.MartinNavigator;
import cz.cuni.amis.pogamut.udk.bot.impl.UDKBot;
import cz.cuni.amis.pogamut.udk.communication.messages.gbcommands.Initialize;
import cz.cuni.amis.pogamut.unreal.communication.messages.UnrealId;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.ConfigChange;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.GameInfo;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.InitedMessage;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.Self;

/**
 * Allows you to specify start-end navpoint that should be run through (tested).
 * <p><p>
 * Initialize with {@link NavigationTestBotParameters} or set start-end navpoint pairs
 * via {@link NavigationTestBot#setStartNavPointId(String)} and {@link NavigationTestBot#setEndNavPointId(String)}
 * (navpoint from the params are taken only iff navpoint pair is not already set).
 * 
 * @author Jimmy
 */
public class NavigationTestBot extends UDKBotTestController<UDKBot> {
	
	private enum State {
		INIT,
		PREPARE_TEST,
		SPAWNED,
		COMMAND_ISSUED,
		RUNNING_BACK
	}
	
	State state = State.INIT;
	
	NavPoint startNavPoint = null;
	NavPoint endNavPoint = null;
	
	UnrealId startNavPointId = null;
	UnrealId endNavPointId = null;
	
	Integer totalRepetitions = null;
	int repetitions = 0;
	boolean walkInCircles = false;
	
	String name;
	
	@Override
	public void prepareBot(UDKBot bot) {
		if (bot.getParams() instanceof NavigationTestBotParameters) {
            NavigationTestBotParameters navigationParams = (NavigationTestBotParameters)bot.getParams();
			if (startNavPointId == null) {
				setStartNavPointId((navigationParams).getStartNavPointId());
			}
			if (endNavPointId == null) {
				setEndNavPointId((navigationParams).getEndNavPointId());
			}
			if (totalRepetitions == null) {
				totalRepetitions = (navigationParams).getNumOfRepetitions();
			}
                        if(navigationParams.getPathPlanner() != null){
                            this.pathPlanner = navigationParams.getPathPlanner();
                        }
			walkInCircles = (navigationParams).isWalkInCircles();
		}
	}
	
	@Override
	protected void initializePathFinding(UDKBot bot) {
		super.initializePathFinding(bot);
	}
	
	@Override
	public void botInitialized(GameInfo gameInfo, ConfigChange currentConfig, InitedMessage init) {
		super.botInitialized(gameInfo, currentConfig, init);
		pathExecutor.getLog().setLevel(Level.ALL);
	}
	
	@Override
	public void botSpawned(GameInfo gameInfo, ConfigChange currentConfig, InitedMessage init, Self self) {
		name = self.getName();
	}

    @Override
    public Initialize getInitializeCommand() {
        startNavPoint = getWorldView().get(startNavPointId, NavPoint.class);
        if (startNavPoint == null) {
            throw new IllegalStateException("Could not find start NavPoint: " + startNavPointId);
        }
        return new Initialize().setLocation(startNavPoint.getLocation());
    }
	
        
        
	@Override
	public void logic() {
		if (isFailure()) {
			getLog().severe("FAILED");
			return;
		} else 
		if (isSuccess()) {
			getLog().severe("SUCCEEDED");
			return;
		}
		switch (state) {
		case INIT:
			if (startNavPointId == null) {
				setFailure("startNavPointId not set!");
				return;
			}
			if (endNavPointId == null) {
				setFailure("endNavPointId not set!");
			}			
			
			startNavPoint = world.getAll(NavPoint.class).get(startNavPointId);			
			if (startNavPoint == null) {
				setFailure("Could not find start navpoint '" + startNavPointId.getStringId() + "'!");
				return;
			}
			
			endNavPoint = world.getAll(NavPoint.class).get(endNavPointId);
			if (endNavPoint == null) {
				setFailure("Could not find end navpoint '" + endNavPointId.getStringId() + "'!");
				return;
			}
			
			if (totalRepetitions == null) {
				setFailure("Number of repetitions for the test was not set!");
				return;
			}
			
			repetitions = totalRepetitions;
			if (repetitions <= 0) {
				setFailure("Number of repetitions for the test <= 0!");
				return;
			}
			
			//body.getAction().respawn(startNavPoint);
			state = State.PREPARE_TEST;
			return;
			
		case PREPARE_TEST:
			//body.getAction().respawn(startNavPoint);
			state = State.SPAWNED;
			return;
			
		case SPAWNED:
			if (!info.atLocation(startNavPoint.getLocation())) {
				getLog().warning("Bot is not at " + startNavPoint.getId().getStringId() + ", respawning again!");
				body.getAction().respawn(startNavPoint);
				return;
			}		
			getLog().warning("Navigation test " + (totalRepetitions - repetitions + 1) + " / " + totalRepetitions);
			config.setName(name + " " + (totalRepetitions - repetitions + 1) + " / " + totalRepetitions);
			pathExecutor.followPath(pathPlanner.computePath(info.getLocation(), endNavPoint));
			state = State.COMMAND_ISSUED;
			return;
			
		case COMMAND_ISSUED:
			if (pathExecutor.isExecuting()) return;
			if (info.atLocation(endNavPoint.getLocation())) {
				if (walkInCircles) {
					pathExecutor.followPath(pathPlanner.computePath(info.getLocation(), startNavPoint));
					state = State.RUNNING_BACK;
					return;
				}
				--repetitions;				
				if (repetitions == 0) {
					setSuccess("Successfully arrived to '" + endNavPointId.getStringId() + "'.");
				} else {
					getLog().info("Successfully arrived to '" + endNavPointId.getStringId() + "'.");
					state = State.PREPARE_TEST;
				}
				return;
			} else {
				setFailure("Failed to arrive to '" + endNavPointId.getStringId() + "'.");
				return;
			}
			
		case RUNNING_BACK:
			if (pathExecutor.isExecuting()) return;
			if (info.atLocation(startNavPoint.getLocation())) {
				--repetitions;				
				if (repetitions == 0) {
					setSuccess("Successfully arrived back to '" + startNavPointId.getStringId() + "'.");
				} else {
					getLog().info("Successfully arrived back to '" + startNavPointId.getStringId() + "'.");
					state = State.SPAWNED;
				}
				return;
			} else {
				setFailure("Failed to arrive back at '" + startNavPointId.getStringId() + "'.");
				return;
			}
		}
		
	}

	public UnrealId getStartNavPointId() {
		return startNavPointId;
	}

	public void setStartNavPointId(String startNavPointId) {
		this.startNavPointId = UnrealId.get(startNavPointId);
	}

	public UnrealId getEndNavPointId() {
		return endNavPointId;
	}

	public void setEndNavPointId(String endNavpointId) {
		this.endNavPointId = UnrealId.get(endNavpointId);
	}

	public void setStartNavPointId(UnrealId startNavPointId) {
		this.startNavPointId = startNavPointId;
	}

	public void setEndNavPointId(UnrealId endNavpointId) {
		this.endNavPointId = endNavpointId;
	}

	public int getRepetitions() {
		return totalRepetitions == null ? 0 : totalRepetitions;
	}

	public void setRepetitions(int repetitions) {
		this.totalRepetitions = repetitions;
	}
	
}
