package cz.cuni.amis.pogamut.udk.communication.worldview;

import java.io.File;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.junit.Assert;
import org.junit.Test;

import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldChangeEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.component.bus.ComponentBus;
import cz.cuni.amis.pogamut.base.utils.logging.AgentLogger;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.udk.communication.worldview.testplan.wrapper.EventWrapper;
import cz.cuni.amis.pogamut.udk.communication.worldview.testplan.wrapper.EventWrapperWithCallCount;
import cz.cuni.amis.pogamut.udk.communication.worldview.testplan.wrapper.InputEventWrapper;
import cz.cuni.amis.pogamut.udk.component.ComponentStub;
import cz.cuni.amis.utils.Const;
import cz.cuni.amis.utils.NullCheck;

/**
 * Top level abstract class for worldview test cases.
 * @author Radek 'Black_Hand' Pibil
 *
 */
public abstract class AbstractWorldViewTest {
	
	private WorldViewTestContext ctx;
	
	private AgentId agentId;

	private IAgentLogger logger;

	private ComponentBus bus;

	private ComponentStub worldViewController;

	private LogCategory log;
	
	public AgentId getAgentId() {
		return agentId;
	}

	public IAgentLogger getLogger() {
		return logger;
	}
	
	public Logger getLog() {
		return log;
	}

	public ComponentBus getBus() {
		return bus;
	}
	
	public ComponentStub getWorldViewController() {
		return worldViewController;
	}

	public WorldViewTestContext getContext() {
		return ctx;
	}
	
	@Test
	public void testWorldView() {				
		doTest();
	}
	
	protected abstract IWorldView createWorldView(IAgentLogger logger, ComponentBus bus, ComponentStub worldViewController, MockMediator mockMediator);

	protected AbstractWorldViewTest(File planFile) {
		Assert.assertTrue("planFile is null.", planFile != null);
		
		this.agentId = new AgentId("WorldViewTest");
		this.logger = new AgentLogger(agentId);
		this.log = this.logger.getCategory("Test");
		this.logger.addDefaultConsoleHandler();
		this.logger.setLevel(Level.ALL);
		
		if (log.isLoggable(Level.INFO)) log.info("Inicializing: " + getClass().getSimpleName());
		if (log.isLoggable(Level.INFO)) log.info("Plan: " + planFile.getAbsolutePath());
		
		this.bus = new ComponentBus(logger);
		this.worldViewController = new ComponentStub(logger, bus);
		this.ctx = new WorldViewTestContext(planFile, log, createWorldView(getLogger(), getBus(), getWorldViewController(), new MockMediator()));
	}
	
	protected IWorldChangeEvent clone(IWorldChangeEvent msg) {
		NullCheck.check(msg, "msg");
		try {
			return msg.getClass().getConstructor(msg.getClass()).newInstance(msg);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	protected void doTest() {				
		
		getWorldViewController().getController().manualStart("Starting the worldview.");
		
		ctx.getRunningFlag().setFlag(true);
		
		preTest();
		
		for (EventWrapper eventWrapper : ctx.getContainer().getEventWrappers()) {
			eventWrapper.registerToWorldView(ctx.getWorldView());
		}

		preNotifications();
		for (InputEventWrapper inputEvent : ctx.getContainer().getInputEventsPlan()) {
			for (int i = 0; i < inputEvent.getRaiseTimes(); ++i) {
				if (ctx.getLog().isLoggable(Level.INFO)) ctx.getLog().info("Notifying wrapper: " + inputEvent.getEvent().toString());
				ctx.getWorldView().notify(clone(inputEvent.getEvent()));				
			}
		}
		if (ctx.getLog().isLoggable(Level.INFO)) ctx.getLog().info("postNotifications");

		postNotifications();
		
		ctx.getRunningFlag().setFlag(false);

		if (ctx.getLog().isLoggable(Level.INFO)) ctx.getLog().info("postTest");
		postTest();
		
		getWorldViewController().getController().manualStop("Stopping the worldview.");
		
		// verify call counts
		for (EventWrapperWithCallCount callCountsWrapper : ctx.getContainer().getExpectedEventsPlan()) {
			if (!callCountsWrapper.hasBeenCalledExpectedNumberOfTimes()) {
				String str = "Event has not been called enough times." + Const.NEW_LINE +
				"Expected event:" + callCountsWrapper.getEventWrapper().getExpectedEventString() + Const.NEW_LINE +
				"Called: " + callCountsWrapper.getCalled() + " expected calls: "
				+ callCountsWrapper.getExpectedNumberOfCalls();
				if (log.isLoggable(Level.SEVERE)) log.severe(str);
				Assert.fail(str);
			}
		}
		
		System.out.println("---/// TEST OK ///---");
	}
		
	protected final synchronized void fail(String message) {
		if (message != null) {
				Assert.fail(message);
		} else {
				Assert.fail();
		}
	}
	
	/**
	 * Called before calling any "notify" on tested worldView.
	 */	
	protected abstract void preNotifications();
	/**
	 * Called after calling any "notify" on tested worldView.
	 */		
	protected abstract void postNotifications();	
	
	/**
	 * Called before the test starts, and setUp has not yet been called on module.
	 */
	protected abstract void preTest();
	/**
	 * Called after the test is over, running is false, and setDown has been called on module.
	 */
	protected abstract void postTest();

}
