package cz.cuni.amis.pogamut.udk.agent.navigation;

import java.util.List;

import cz.cuni.amis.pogamut.base.agent.navigation.IPathExecutor;
import cz.cuni.amis.pogamut.base.agent.navigation.IPathExecutorHelper;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.udk.bot.impl.UDKBot;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.Self;

/**
 * Navigator purpose is to actually move the bot through the UDK environment - if you did not read {@link IPathExecutor} 
 * and {@link UDKPathExecutor} documentation, do it now. If you did, read on.
 * <p><p>
 * The {@link IUDKPathNavigator} navigator is the easiest-to-implement piece of {@link UDKPathExecutor} in
 * terms that it has only two methods to implement. On the other hand - it is acrually very hard to navigate the bot
 * through UDK environment. But! If you need to actually change the way how bot is running inside the UDK,
 * implementing own {@link IUDKPathNavigator} is the best thing to do (you should probably start
 * by copy-pasting the code from {@link UDKPathExecutorNavigator} into your new class and experiment with it a bit).
 * <p><p>
 * This navigator interface is actually used by {@link UDKPathExecutor} that covers the tricky part when and how 
 * to call its methods {@link IUDKPathNavigator#navigate()} and {@link IUDKPathNavigator#reset()}.
 * 
 * @author Jimmy
 *
 * @param <PATH_ELEMENT>
 */
public interface IUDKPathNavigator<PATH_ELEMENT extends ILocated> {

	/**
	 * Sets the {@link UDKBot} instance that the navigator should navigate. Use its {@link UDKBot#getAct()}
	 * to pass commands to the bot.
	 * 
	 * @param bot
	 */
	public void setBot(UDKBot bot);
	
	/**
	 * Sets the {@link IPathExecutorHelper} who is using the navigator, i.e., are calling its
	 * {@link IUDKPathNavigator#navigate(Self)} and {@link IUDKPathNavigator#reset()}
	 * methods.
	 * <p><p>
	 * Used by {@link IPathExecutorHelper} implementation to inject its instance into the navigator,
	 * so the navigator may call methods such as {@link IPathExecutorHelper#checkStuckDetectors()},
	 * {@link IPathExecutorHelper#switchToAnotherPathElement(int)}, {@link IPathExecutorHelper#stuck()}
	 * and {@link IPathExecutorHelper#targetReached()}.
	 *  
	 * @param owner
	 */
	public void setExecutor(IPathExecutorHelper<PATH_ELEMENT> owner);
	
	/**
	 * This method is regularly called by {@link UDKPathExecutor} to continue the navigation of the bot
	 * inside the UDK.
	 * 
	 * @param bot instance of the bot we're navigating
	 * @param self {@link Self} object of the bot the navigator should move
	 */
	public void navigate();
	
	/**
	 * {@link UDKPathExecutor} reports that execution of current path has been terminated - clean up your internal data
	 * structure and prepare to navigate the bot along the new path in the future. 
	 */
	public void reset();
	
	/**
	 * {@link UDKPathExecutor} reports that new path has been received and the {@link IUDKPathNavigator#navigate()}
	 * is about to be called in near future. The new path is passed as a parameter.
	 * 
	 * @param path
	 */
	public void newPath(List<PATH_ELEMENT> path);
	
	
}
