/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package cz.cuni.amis.pogamut.unreal.t3dgenerator;

import cz.cuni.amis.pogamut.unreal.t3dgenerator.annotations.FieldName;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.annotations.StaticText;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.annotations.UnrealBean;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.annotations.UnrealChild;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.annotations.UnrealChildCollection;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.annotations.UnrealComponent;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.annotations.UnrealDataType;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.annotations.UnrealHeaderField;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.annotations.UnrealProperty;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.datatypes.UnrealReference;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.elements.IUnrealReferencable;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.elements.IUnrealReferencableByName;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.lang.annotation.Annotation;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import org.apache.commons.beanutils.PropertyUtils;

/**
 *
 * @author Martin Cerny
 */
public class DefaultT3dGenerator extends AbstractT3dGenerator {

    protected static String indentString(int indent) {
        StringBuilder sb = new StringBuilder(indent);
        for (int i = 0; i < indent; i++) {
            sb.append("\t");
        }
        return sb.toString();
    }

    protected String getPropertyNameForField(Field f, String annotationValue) {
        String propertyName = annotationValue;
        //if the property name is not specified derive it from field name - the first letter is uppercased.
        if (propertyName == null || propertyName.equals("")) {
            propertyName = f.getName().substring(0, 1).toUpperCase() + f.getName().substring(1);
            if (f.getType() == Boolean.class) {
                //booleans in UDK are always prefixed with "b"
                propertyName = "b" + propertyName;
            }
        }
        return propertyName;
    }

    private void preProcess(Object unrealObject, IT3dGeneratorContext context) {
        if (unrealObject instanceof IUnrealReferencableByName) {
            IUnrealReferencableByName referencableObject = (IUnrealReferencableByName) unrealObject;
            if (referencableObject.getNameForReferences() == null) {
                if (referencableObject.getClassName() != null) {
                    referencableObject.setNameForReferences(context.getNamingFactory().getName(referencableObject.getClassName()));
                } else {
                    throw new T3dGeneratorException("Element with generated name must specify an element class.");
                }
            }
        }
        for (Object child : getChildren(unrealObject)) {
            preProcess(child, context);
        }
    }

    protected String getPropertyValueString(final Object value, boolean escapeStrings) {
        String valueString;
        if (value == null) {
            throw new NullPointerException("Null values are not supported");
        } else if (value instanceof String) {
            if (escapeStrings) {
                valueString = '"' + value.toString() + '"';
            } else {
                valueString = value.toString();
            }
        } else if (value instanceof Boolean) {
            Boolean bVal = (Boolean) value;
            if (bVal == null) {
                throw new InvalidPropertyValueException("Boolean values can't be null");
            }
            if (bVal) {
                valueString = "True";
            } else {
                valueString = "False";
            }
        } else if (value instanceof Number) {
            valueString = value.toString();
        } else if (value instanceof UnrealReference) {
            valueString = ((UnrealReference) value).toReferenceString();
        } else if (value instanceof IUnrealReferencable) {
            valueString = ((IUnrealReferencable) value).getReference().toReferenceString();
        } else if (value.getClass().isAnnotationPresent(UnrealDataType.class)) {

            final StringBuilder valueBuilder = new StringBuilder("(");
            ReflectionUtils.processEachDeclaredField(value, Object.class, new ReflectionUtils.ProcessFieldCallback<T3dGeneratorException>() {

                boolean appendedSomething = false;

                @Override
                public void processField(Field f) throws T3dGeneratorException {
                    if (Modifier.isTransient(f.getModifiers())) {
                        return;
                    }
                    String fieldNameFromAnnotation = "";
                    if (f.isAnnotationPresent(FieldName.class)) {
                        fieldNameFromAnnotation = f.getAnnotation(FieldName.class).value();
                    }
                    String propertyName = getPropertyNameForField(f, fieldNameFromAnnotation);
                    Object fieldValue;
                    try {
                        fieldValue = PropertyUtils.getProperty(value, f.getName());
                    } catch (Exception ex) {
                        throw new T3dGeneratorException("Could not read property value for " + f.getName() + " in T3dSerializableValue annotated class" + f.getDeclaringClass().getName() + "\nThe property should have public getter or it should be marked as transient.", ex);
                    }

                    //Null values are ignored
                    if (fieldValue == null) {
                        return;
                    }

                    String subPropString = getPropertyString(propertyName, fieldValue);
                    if (!subPropString.isEmpty()) {
                        if (appendedSomething && valueBuilder.charAt(valueBuilder.length() - 1) != ',') {
                            valueBuilder.append(",");
                        }
                        appendedSomething = true;
                    }
                    valueBuilder.append(subPropString);
                }
            });
            valueBuilder.append(")");
            valueString = valueBuilder.toString();
        } else if (value instanceof Collection) {
            Collection collectionValue = (Collection) value;
            StringBuilder valueBuilder = new StringBuilder("(");
            boolean appendedSomething = false;
            for (Object collectionElement : collectionValue) {
                if (appendedSomething) {
                    valueBuilder.append(",");
                } else {
                    appendedSomething = true;
                }
                valueBuilder.append(getPropertyValueString(collectionElement));
            }
            valueBuilder.append(")");
            valueString = valueBuilder.toString();
        } else {
            throw new InvalidPropertyValueException("Unsupported value class. Only String, Boolean, Number, T3dReference, IUnrealReferencable, UnrealDataType and collections of those are supported.");
        }
        return valueString;
    }

    protected String getPropertyValueString(Object value) {
        return getPropertyValueString(value, true);
    }

    protected String getPropertyString(String key, Object value, boolean escapeStrings) {
        //default hodnoty se vubec do properties nezapisuji - ignoruji je
        if (value == null) {
            return "";
        }
        return key + "=" + getPropertyValueString(value, escapeStrings);
    }

    protected String getPropertyString(String key, Object value) {
        return getPropertyString(key, value, true);
    }

    protected void generateProperty(OutputStreamWriter out, String indetStream, String propertyName, Object propertyValue) throws IOException {
        if (propertyValue.getClass().isArray()) {
            for (int i = 0; i < Array.getLength(propertyValue); i++) {
                out.append("\t").append(indetStream).append(getPropertyString(propertyName + "(" + i + ")", Array.get(propertyValue, i))).append("\n");
            }
        } else if (propertyValue instanceof List) {
            List listValue = (List) propertyValue;
            for (int i = 0; i < listValue.size(); i++) {
                out.append("\t").append(indetStream).append(getPropertyString(propertyName + "(" + i + ")", listValue.get(i))).append("\n");
            }
        } else {
            out.append("\t").append(indetStream).append(getPropertyString(propertyName, propertyValue)).append("\n");
        }
    }

    protected Map<String, Object> getProperties(final Object object) throws IOException {
        final Map<String, Object> result = new TreeMap<String, Object>();
        ReflectionUtils.processEachDeclaredField(object,
                new ReflectionUtils.ProcessFieldCallback<IOException>() {

                    @Override
                    public void processField(Field f) throws IOException {

                        //skip transient and static fields
                        if (Modifier.isTransient(f.getModifiers()) || Modifier.isStatic(f.getModifiers())) {
                            return;
                        }
                        //ignore static text
                        if(f.isAnnotationPresent(StaticText.class)){
                            return;
                        }
                        if (!f.isAnnotationPresent(UnrealProperty.class)) {
                            //such annotated fields are not considered property unless explicitly specified
                            if (f.isAnnotationPresent(UnrealChild.class)
                                    || (f.isAnnotationPresent(UnrealChildCollection.class))
                                    || (f.isAnnotationPresent(UnrealComponent.class))
                                    || (f.isAnnotationPresent(UnrealHeaderField.class))) {
                                return;
                            }
                        }

                        Object fieldValue = null;
                        try {
                            fieldValue = PropertyUtils.getProperty(object, f.getName());
                        } catch (Exception ex) {
                            throw new T3dGeneratorException("Could not read property value for property " + f.getName() + " in class" + f.getDeclaringClass().getName() + "\nThe property should have public getter.", ex);
                        }

                        if (fieldValue == null) { //unset values are not exported
                            return;
                        }

                        String nameFromAnnotation = "";
                        if (f.isAnnotationPresent(FieldName.class)) {
                            nameFromAnnotation = f.getAnnotation(FieldName.class).value();
                        }
                        String propertyName = getPropertyNameForField(f, nameFromAnnotation);
                        result.put(propertyName, fieldValue);
                    }
                });


        ReflectionUtils.processEachAnnotatedDeclaredMethod(object, UnrealProperty.class, new ReflectionUtils.ProcessAnnotatedMethodCallback<UnrealProperty>(){

            @Override
            public void processMethod(Method m, UnrealProperty annotation) {
                if(!Modifier.isPublic(m.getModifiers()) || m.getParameterTypes().length > 0 || m.getReturnType() == Void.class){
                    throw new T3dGeneratorException("Methods annotated with @UnrealProperty must be public, take zero arguments and have non-void return value.");
                }
                String propertyName = null;
                if(m.isAnnotationPresent(FieldName.class)){
                    propertyName = m.getAnnotation(FieldName.class).value();
                } else {
                    if(!m.getName().startsWith("get")){
                        throw new T3dGeneratorException("Methods annotated with @UnrealProperty must either have @FieldName or conform to Java Beans getter name conventions");
                    } else {
                        propertyName = m.getName().substring(3);
                    }
                }
                try {
                    result.put(propertyName, m.invoke(object));
                } catch(Exception ex){
                    throw new T3dGeneratorException("Exception reading property value from method '" +m.getName() + "' at class " + m.getDeclaringClass(), ex);    
                }
            }
        
        });

        if(result.containsKey("Components")){
            throw new T3dGeneratorException("Property 'Components' cannot be defined by user. Use @UnrealComponent instead");
        }
        
        final List<Object> components = new ArrayList<Object>();
        //process the components property
        ReflectionUtils.processEachAnnotatedDeclaredField(object, UnrealComponent.class, new ReflectionUtils.ProcessAnnotatedFieldCallback<UnrealComponent, T3dGeneratorException>() {

            @Override
            public void processField(Field f, Object fieldValue, UnrealComponent annotation) throws T3dGeneratorException {
                //skip unset values
                if (fieldValue == null) {
                    return;
                }
                components.add(fieldValue);
            }
        });
        
        result.put("Components",components);


        return result;
    }

    protected void generateProperties(Object object, final OutputStreamWriter out, int indent) throws IOException {
        Map<String, Object> properties = getProperties(object);
        String indentString = indentString(indent);
        for (Map.Entry<String, Object> entry : properties.entrySet()) {
            generateProperty(out, indentString, entry.getKey(), entry.getValue());
        }
    }

    protected void generateHeaderPropertiesFromAnnotatedFields(Object object, final OutputStreamWriter out) throws IOException {
        ReflectionUtils.processEachAnnotatedDeclaredField(object, UnrealHeaderField.class,
                new ReflectionUtils.ProcessAnnotatedFieldCallback<UnrealHeaderField, IOException>() {

                    @Override
                    public void processField(Field f, Object fieldValue, UnrealHeaderField annotation) throws IOException {
                        if (fieldValue == null) { //unset values are not exported
                            return;
                        }
                        String propertyName = getPropertyNameForField(f, annotation.value());
                        out.append(" ").append(getPropertyString(propertyName, fieldValue, false));
                    }
                });
    }

    protected List<Object> getChildren(Object unrealObject) {

        final List<Object> childrenFromAnnotations = new ArrayList<Object>();

        ReflectionUtils.processEachAnnotatedDeclaredField(unrealObject, Object.class, UnrealChildCollection.class, new ReflectionUtils.ProcessAnnotatedFieldCallback<UnrealChildCollection, T3dGeneratorException>() {

            @Override
            public void processField(Field f, Object fieldValue, UnrealChildCollection annotation) throws T3dGeneratorException {
                if (!Collection.class.isAssignableFrom(f.getType())) {
                    throw new T3dGeneratorException("Field " + f.getName() + " in class " + f.getDeclaringClass()
                            + "is annotated with @UnrealChildCollection, but is not a collection");
                }
                //skip unset values
                if (fieldValue == null) {
                    return;
                }
                for (Object item : (Collection) fieldValue) {
                    childrenFromAnnotations.add(item);
                }
            }
        });


        ReflectionUtils.processEachAnnotatedDeclaredField(unrealObject, UnrealChild.class, new ReflectionUtils.ProcessAnnotatedFieldCallback<UnrealChild, T3dGeneratorException>() {

            @Override
            public void processField(Field f, Object fieldValue, UnrealChild annotation) throws T3dGeneratorException {
                //skip unset values
                if (fieldValue == null) {
                    return;
                }
                childrenFromAnnotations.add(fieldValue);
            }
        });

        //UnrealComponents are children as well
        ReflectionUtils.processEachAnnotatedDeclaredField(unrealObject, UnrealComponent.class, new ReflectionUtils.ProcessAnnotatedFieldCallback<UnrealComponent, T3dGeneratorException>() {

            @Override
            public void processField(Field f, Object fieldValue, UnrealComponent annotation) throws T3dGeneratorException {
                //skip unset values
                if (fieldValue == null) {
                    return;
                }
                childrenFromAnnotations.add(fieldValue);
            }
        });

        return childrenFromAnnotations;


    }

    private void generateStaticText(Object object, final OutputStreamWriter out, final int indent) throws IOException {
        ReflectionUtils.processEachAnnotatedDeclaredField(object, StaticText.class, new ReflectionUtils.ProcessAnnotatedFieldCallback<StaticText, IOException>() {

            @Override
            public void processField(Field f, Object fieldValue, StaticText annotation) throws IOException {
                //skip unset values
                if (fieldValue == null) {
                    return;
                }
                out.append(fieldValue.toString());
            }
        });

    }

    private String getElementType(Object unrealObject) {
        UnrealBean beanAnnotation = unrealObject.getClass().getAnnotation(UnrealBean.class);
        if (beanAnnotation == null) {
            throw new T3dGeneratorException("Cannot generate T3d for objects not annotated with UnrealBean");
        } else {
            return beanAnnotation.value();
        }
    }

    private void generateT3dInternal(Object object, OutputStreamWriter out, int indent) throws IOException {
        String is = indentString(indent);
        String elementType = getElementType(object);
        out.append(is).append("Begin ").append(elementType);
        generateHeaderPropertiesFromAnnotatedFields(object, out);
        out.append("\n");


        for (Object child : getChildren(object)) {
            generateT3dInternal(child, out, indent + 1);
        }

        generateStaticText(object, out, indent);
        generateProperties(object, out, indent);

        out.append(is).append("End ").append(elementType).append("\n");
    }

    @Override
    public void generateT3d(List elements, OutputStreamWriter out) throws IOException {
        IT3dGeneratorContext context = new DefaultT3dGeneratorContext(new SequenceNamingFactory());
        for (Object element : elements) {
            preProcess(element, context);
        }
        for (Object element : elements) {
            generateT3dInternal(element, out, 0);
        }
    }
}
