/*
 * Copyright (C) 2011 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.udk.experiments.impl;

import cz.cuni.amis.pogamut.udk.experiments.IValueConverter;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.ReflectionUtils;
import java.io.File;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;

/**
 * A CSV writer that uses reflection to write all non-transient non-volatile fields of result and parameter classes
 * to CSV with appropriate headers.
 * @author Martin Cerny
 */
public class ReflectionCSVResultWriter<RESULT, EXPERIMENT_PARAMS> extends AbstractCSVResultWriter<RESULT, EXPERIMENT_PARAMS> {

    List<Field> parametersFields;

    List<Field> resultFields;

    public ReflectionCSVResultWriter(File targetFile, IValueConverter valueConverter, Class<RESULT> resultClass, Class<EXPERIMENT_PARAMS> paramsClass) {
        super(targetFile, valueConverter);
        initFields(resultClass, paramsClass);
    }

    public ReflectionCSVResultWriter(File targetFile, Class<RESULT> resultClass, Class<EXPERIMENT_PARAMS> paramsClass) {
        super(targetFile);
        initFields(resultClass, paramsClass);
    }

    private void initFields(Class<RESULT> resultClass, Class<EXPERIMENT_PARAMS> paramsClass) {
        parametersFields = new ArrayList<Field>();
        resultFields = new ArrayList<Field>();

        ReflectionUtils.processEachDeclaredFieldOfClass(resultClass, new ReflectionUtils.ProcessFieldCallback<RuntimeException>() {

            @Override
            public void processField(Field f) throws RuntimeException {
                if (Modifier.isTransient(f.getModifiers()) || Modifier.isVolatile(f.getModifiers())) {
                    return;
                }
                f.setAccessible(true);
                resultFields.add(f);
            }
        });

        if (paramsClass != null) {
            ReflectionUtils.processEachDeclaredFieldOfClass(paramsClass, new ReflectionUtils.ProcessFieldCallback<RuntimeException>() {

                @Override
                public void processField(Field f) throws RuntimeException {
                    if (Modifier.isTransient(f.getModifiers()) || Modifier.isVolatile(f.getModifiers())) {
                        return;
                    }
                    f.setAccessible(true);
                    parametersFields.add(f);
                }
            });
        }
    }

    @Override
    protected List<String> getParametersHeaders() {
        List<String> headers = new ArrayList<String>(parametersFields.size());
        for (Field f : parametersFields) {
            headers.add(f.getName());
        }
        return headers;
    }

    @Override
    protected List<Object> getParametersValues(EXPERIMENT_PARAMS params) {
        List<Object> values = new ArrayList<Object>(parametersFields.size());
        for (Field f : parametersFields) {
            try {
                values.add(f.get(params));
            } catch (IllegalArgumentException ex) {
                throw new IllegalStateException("Error reading field " + f);
            } catch (IllegalAccessException ex) {
                throw new IllegalStateException("Error reading field " + f);
            }
        }
        return values;
    }

    @Override
    protected List<String> getResultHeaders() {
        List<String> headers = new ArrayList<String>(resultFields.size());
        for (Field f : resultFields) {
            headers.add(f.getName());
        }
        return headers;
    }

    @Override
    protected List<Object> getResultValues(RESULT result) {
        List<Object> values = new ArrayList<Object>(resultFields.size());
        for (Field f : resultFields) {
            try {
                values.add(f.get(result));
            } catch (IllegalArgumentException ex) {
                throw new IllegalStateException("Error reading field " + f);
            } catch (IllegalAccessException ex) {
                throw new IllegalStateException("Error reading field " + f);
            }
        }
        return values;
    }
}
