/*
 * Copyright (C) 2011 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.udk.experiments.impl;

import cz.cuni.amis.pogamut.udk.experiments.IExperimentRunResult;
import cz.cuni.amis.pogamut.udk.experiments.IResultWriter;
import cz.cuni.amis.pogamut.udk.experiments.IValueConverter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Date;
import java.util.List;

/**
 * A simple helper class for creating result writers that write to CSV files (to be read by R or something)
 * @author Martin Cerny
 */
public abstract class AbstractCSVResultWriter<RESULT, EXPERIMENT_PARAMS> implements IResultWriter<RESULT, EXPERIMENT_PARAMS> {

    private File targetFile;

    private FileWriter writer;
    
    protected String fieldDelimiter = ";";
    protected String lineDelimiter = "\r\n";
    protected String quote = "\"";
        
    private int numResultColumns;
    
    private IValueConverter valueConverter;
    
    private boolean writtenSomethingOnCurrentLine = false;

    public AbstractCSVResultWriter(File targetFile) {
        this(targetFile, new DefaultValueConverter());
    }

    public AbstractCSVResultWriter(File targetFile, IValueConverter valueConverter) {
        this.targetFile = targetFile;
        this.valueConverter = valueConverter;
    }
    
    
    protected abstract List<String> getParametersHeaders();
    protected abstract List<String> getResultHeaders();
    protected abstract List<Object> getParametersValues(EXPERIMENT_PARAMS params);
    protected abstract List<Object> getResultValues(RESULT result);
    
    @Override
    public void close() throws IOException {
        writer.close();
    }

    protected void writeField(String value) throws IOException{
        if(writtenSomethingOnCurrentLine){
            writer.write(fieldDelimiter);
        } else {
            writtenSomethingOnCurrentLine = true;
        }
        //quote is escaped by doubling it
        writer.append(quote).append(value.replace(quote, quote + quote)).append(quote);
    }
    
    protected void endLine() throws IOException {
        writer.write(lineDelimiter);
        writer.flush();
        writtenSomethingOnCurrentLine = false;
    }
    
    @Override
    public void init() throws IOException {
        writer = new FileWriter(targetFile);
        writeField("#");
        writeField("Start time");
        for(String header: getParametersHeaders()){
            writeField(header);
        }
        writeField("Status");
        writeField("Running time");
        List<String> resultHeaders = getResultHeaders();
        numResultColumns = resultHeaders.size();        
        for(String header: resultHeaders){
            writeField(header);
        }
        endLine();
    }

    @Override
    public void writeResult(int experimentNumber, IExperimentRunResult<RESULT> result, EXPERIMENT_PARAMS experimentParams) throws IOException {
        writeField(valueConverter.valueToString(experimentNumber));
        writeField(valueConverter.valueToString(new Date(result.getStartTime())));        
        
        if(experimentParams != null){
            for(Object field : getParametersValues(experimentParams)){
                writeField(valueConverter.valueToString(field));
            }
        }
        
        writeField(result.getState().toString());        
        writeField(valueConverter.valueToString(result.getRunningTime()));
        
        if(result.getState() == IExperimentRunResult.State.SUCCESS) {
            for(Object field : getResultValues(result.getResult())){
                writeField(valueConverter.valueToString(field));
            }
        } else {
            for(int i = 0; i < numResultColumns; i++){
                writeField("");
            }
        }
        
        endLine();
    }
    
}
