/*
 * Copyright (C) 2011 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.udk.experiments.impl;

import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.udk.communication.messages.gbcommands.StopRecord;
import cz.cuni.amis.pogamut.udk.experiments.IExperiment;
import cz.cuni.amis.pogamut.udk.experiments.IExperimentRunResult;
import cz.cuni.amis.pogamut.udk.experiments.IExperimentRunner;
import cz.cuni.amis.pogamut.udk.factory.direct.remoteagent.UDKServerFactory;
import cz.cuni.amis.pogamut.udk.server.IUDKServer;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper.UCCWrapperConf;
import cz.cuni.amis.pogamut.udk.utils.UDKServerRunner;

/**
 *
 * @author Martin Cerny
 */
public class ExperimentRunner<RESULT, PARAMETERS> implements IExperimentRunner<RESULT, PARAMETERS> {

    protected LogCategory log;

    /**
     * Timeout for the experiment in ms
     */
    private long timeout;

    public ExperimentRunner(LogCategory log, long timeout) {
        this.log = log;
        this.timeout = timeout;
    }

    @Override
    public IExperimentRunResult<RESULT> runExperiment(IExperiment<RESULT, PARAMETERS> experiment) {
        
        long startTime = System.currentTimeMillis();
        
        experiment.setLog(log);
        
        UCCWrapper uccWrapper = null;
        IUDKServer server = null;        
        String experimentDescription = experiment.getDescription();
        try {
            log.info("ExperimentRunner: Running experiment " + experimentDescription);
            log.info("ExperimentRunner: Performing beforeServerStart()");
            experiment.beforeServerStart();

            log.info("ExperimentRunner: Starting server.");
            UCCWrapperConf wrapperConfiguration = experiment.getUCCWrapperConfiguration();
            uccWrapper = new UCCWrapper(wrapperConfiguration);

            UDKServerFactory serverFactory = new UDKServerFactory();

            UDKServerRunner serverRunner = new UDKServerRunner(serverFactory, "UDKServerRunner", uccWrapper.getHost(), uccWrapper.getControlPort());
            server = serverRunner.startAgent();

            log.info("ExperimentRunner: Waiting for 1 sec to give UDKServer some time to catch up...");

            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
            }

//		log.info("ExperimentRunner: Checking required map " + requiredMap);
//		
//		if (server.getMapName().equals(requiredMap)) {
//			log.info("ExperimentRunner: Map ok!");
//		} else {
//			fail("ExperimentRunner: GBUDK is running wrong map " + server.getMapName() + ", required map is " + requiredMap); 
//		}
//		

            log.info("ExperimentRunner: Server started.");
            experiment.serverStarted(uccWrapper, server);
            experiment.startExperiment();

            experiment.getFinished().waitFor(timeout, Boolean.TRUE);

            long runningTime = System.currentTimeMillis() - startTime;

            if (!experiment.getFinished().getFlag()) {
                log.info("ExperimentRunner:  Experiment timed out");
                return ExperimentRunResult.timeout(runningTime);
            } else {
                if(experiment.isSuccess()){
                    log.info("ExperimentRunner:  Experiment success");
                    return ExperimentRunResult.succes(experiment.getResult(), runningTime);
                } else {
                    log.info("ExperimentRunner:  Experiment failed");
                    return ExperimentRunResult.failure("Failed.", runningTime);
                }
            }
        } catch (Exception ex) {
            long runningTime = System.currentTimeMillis() - startTime;
            return ExperimentRunResult.exception(ex, runningTime);
        } finally {
            experiment.cleanup();

            if (server != null) {
                try {
                    server.getAct().act(new StopRecord());
                } catch (Exception e) {
                    log.severe("Failed to stop recording." + e);
                }
                server.kill();
            }
            if (uccWrapper != null) {
                try {
                    uccWrapper.stop();
                } catch (Exception ex) {
                    log.severe("Failed to stop UCC." + ex);
                }
            }

        }
    }
}
