/*
 * Copyright (C) 2011 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.udk.experiments.impl;

import cz.cuni.amis.pogamut.base.agent.IAgent;
import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.agent.params.IAgentParameters;
import cz.cuni.amis.pogamut.base.agent.utils.runner.IAgentDescriptor;
import cz.cuni.amis.pogamut.base.agent.utils.runner.impl.MultipleAgentRunner;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.factory.IAgentFactory;
import cz.cuni.amis.pogamut.udk.agent.params.UDKAgentParameters;
import cz.cuni.amis.pogamut.udk.bot.IUDKBot;
import cz.cuni.amis.pogamut.udk.bot.impl.UDKBot;
import cz.cuni.amis.pogamut.udk.experiments.utils.IWatchAgentCallback;
import cz.cuni.amis.pogamut.udk.experiments.utils.WatchAgent;
import cz.cuni.amis.pogamut.udk.experiments.utils.WatchAgentCallbackAdapter;
import cz.cuni.amis.pogamut.udk.factory.guice.remoteagent.UDKBotFactory;
import cz.cuni.amis.pogamut.udk.factory.guice.remoteagent.UDKBotModule;
import java.util.ArrayList;
import java.util.List;

/**
 *
 * @author Martin Cerny
 */
public class AbstractBotExperiment<RESULT, PARAMETERS> extends AbstractExperiment<RESULT, PARAMETERS> {

    protected IAgentDescriptor<UDKAgentParameters, UDKBotModule> agentDescriptors[];

    protected List<UDKBot> runningBots = new ArrayList<UDKBot>();

    protected List<WatchAgent> watchAgents = new ArrayList<WatchAgent>();
    
    boolean failIfBotKilled;

    public AbstractBotExperiment(boolean failIfBotKilled, PARAMETERS params, IAgentDescriptor<UDKAgentParameters, UDKBotModule>... bots) {
        super(params);
        this.agentDescriptors = bots;
        this.failIfBotKilled = failIfBotKilled;
    }

    public AbstractBotExperiment(PARAMETERS params, IAgentDescriptor<UDKAgentParameters, UDKBotModule>... bots) {
        this(false, params, bots);
    }

    protected void startBots(IAgentDescriptor<UDKAgentParameters, UDKBotModule> bots[]) {
        AgentRunner runner = new AgentRunner();
        try {
            runner.startAgents(bots);
        } catch (Exception ex) {
            fail("Bot start failed", ex);
        }

    }

    @Override
    public final void startExperiment() {
        startBots(agentDescriptors);
        afterBotsStarted();
    }

    protected void afterBotsStarted() {
    }

    @Override
    public void cleanup() {
        for (WatchAgent watch : watchAgents) {
            watch.stopWatching();
        }
        for (UDKBot bot : runningBots) {
            bot.kill();
        }
    }

    private class MonitorAgentAliveCallback extends WatchAgentCallbackAdapter {

        @Override
        public void agentKilled(IAgent agent) {
            if (failIfBotKilled) {
                fail("Bot " + agent.getName() + " killed.");
            }
        }

        @Override
        public void agentStopped(IAgent agent) {
            fail("Bot " + agent.getName() + " stopped.");
        }
    }

    private class AgentRunner extends MultipleAgentRunner<UDKBot, UDKAgentParameters, UDKBotModule> {

        @Override
        protected IAgentFactory newAgentFactory(UDKBotModule agentModule) {
            return new UDKBotFactory(agentModule);
        }

        @Override
        protected IAgentParameters newDefaultAgentParameters() {
            return new UDKAgentParameters().setWorldAddress(new SocketConnectionAddress(ucc.getHost(), ucc.getBotPort())).setAgentId(new AgentId("Agent"));
        }

        @Override
        protected void postStartedHook(List<UDKBot> agents) {
            super.postStartedHook(agents);

            IWatchAgentCallback monitorAliveCallback = new MonitorAgentAliveCallback();

            runningBots.addAll(agents);
            for (IUDKBot bot : agents) {
                WatchAgent watchAgent = new WatchAgent(bot, monitorAliveCallback);
                watchAgent.startWatching();
                watchAgents.add(watchAgent);

            }
        }
    }
}
