package cz.cuni.amis.pogamut.emohawk.agent.module.essence;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObject;
import cz.cuni.amis.pogamut.base.communication.worldview.object.WorldObjectId;
import cz.cuni.amis.pogamut.base3d.worldview.IVisionWorldView;
import cz.cuni.amis.pogamut.base3d.worldview.object.IViewable;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.ControllerInfoReplica;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.PawnReplica;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.action.ActionRegistryReplica;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.essence.IEssenceReplica;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Self;

/** Map of associations of replicable objects
 * 
 * This map allows pogamut-emohawk to associate essences with world objects without the need to modify existing pogamut-ut2004 info messages.
 * For example, ReplicableObject in Unreal script may send name of the associated world object and Java receive() can associate the name with the essence.
 * 
 * @author Paletz
 */
public class EssenceMap implements IEssenceMap {

	protected IVisionWorldView worldView;
	protected HashMap<String,PawnReplica> pawns; 	// bot ID (unreal script Controller.name) -> pawn 
	protected HashMap<String,ControllerInfoReplica> controllers; 	// bot ID (unreal script Controller.name) -> controller 
	protected ActionRegistryReplica actionRegistry;
	protected HashMap<String,IEssenceReplica> essences; // entity ID (unreal script Object name) -> essence

	public EssenceMap( IVisionWorldView worldView ) {
		this.worldView = worldView;
		pawns = new HashMap<String,PawnReplica>();
		controllers = new HashMap<String,ControllerInfoReplica>();
		essences = new HashMap<String,IEssenceReplica>();
	}
	
	/** Associate Pawn object with a bot ID
	 * 
	 * @param botId bot ID
	 * @param pawn Pawn object
	 */
	public void associatePawn( String botId, PawnReplica pawn ) {
		pawns.put( botId, pawn );
	}
	
	@Override
	public PawnReplica retrievePawn( String botId ) {
		return pawns.get( botId );
	}
	
	@Override
	public PawnReplica retrievePawn(IWorldObject worldObject) {
		if ( worldObject instanceof Self ) {
			return retrievePawn( ((Self)worldObject).getBotId().getStringId() );
		} else {
			return retrievePawn( worldObject.getId().getStringId() );
		}
	}

	/** Associate Controller object with a bot ID
	 * 
	 * @param botId bot ID
	 * @param controller Controller object
	 */
	public void associateController( String botId, ControllerInfoReplica controller ) {
		controllers.put( botId, controller );
	}
	
	@Override
	public ControllerInfoReplica retrieveController( String botId ) {
		return controllers.get( botId );
	}
	
	@Override
	public ControllerInfoReplica retrieveController(IWorldObject worldObject) {
		return retrieveController( worldObject.getId().getStringId() );
	}
	
	/** Associate action registry
	 * 
	 * @param actionRegistry action registry
	 */
	public void associateActionRegistry( ActionRegistryReplica actionRegistry ) {
		this.actionRegistry = actionRegistry;
	}
	
	@Override
	public ActionRegistryReplica retrieveActionRegistry() {
		return actionRegistry;
	}
		
	/** Associate essence to world object ID
	 *
	 * @param worldObjectId world object ID
	 * @param essence
	 */
	public void associateEssence( String worldObjectId, IEssenceReplica essence ) {
		essences.put( worldObjectId, essence );
	}

	@Override
	public IEssenceReplica retrieveEssence(String worldObjectId) {
		return essences.get( worldObjectId );
	}
	
	@Override
	public IEssenceReplica retrieveEssence( IWorldObject entity ) {
		return retrieveEssence( entity.getId().getStringId() );
	}

	@Override
	public <TEssence extends IEssenceReplica> Map<IWorldObject, TEssence> getAllVisible( Class<TEssence> essenceClass ) {
    	HashMap<IWorldObject,TEssence> retval = new HashMap<IWorldObject,TEssence>();
    	for ( Entry<WorldObjectId,IViewable> entry : worldView.getVisible().entrySet() ) {
    		IWorldObject worldObject = worldView.get( entry.getKey() );
    		IEssenceReplica essence = retrieveEssence( entry.getValue() );
    		if ( essence == null ) {
    			essence = retrievePawn( worldObject );
    		}
    		if ( essence != null && essenceClass.isAssignableFrom( essence.getClass() ) ) {
    			@SuppressWarnings("unchecked")
    			TEssence castedEssence = (TEssence) essence;
    			retval.put( worldObject, castedEssence );
    		}
    	}
    	return retval;
	}
}
