package cz.cuni.amis.pogamut.defcon.utils.quadtree;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import cz.cuni.amis.pogamut.base3d.worldview.object.Location;

/**
 * QuadTree implementation. Made iterable for convenience. The order of
 * evaluation is prefix with [ ul, ur, ll, lr ] as the order of nodes.
 * 
 * @author Radek 'Black_Hand' Pibil
 * 
 */
public class QuadTree implements Iterable<QuadTreeNode> {

	private QuadTreeNode root;

	/**
	 * Constructor
	 * 
	 * @param vertices
	 *            polygon to be represented by the QuadTree.
	 */
	@SuppressWarnings("unchecked")
	public QuadTree(List<? extends Location> vertices) {

		double min_x = Double.MAX_VALUE, max_x = -min_x;
		double min_y = Double.MAX_VALUE, max_y = -min_y;

		for (Location vertex : vertices) {
			if (vertex.getX() < min_x)
				min_x = vertex.getX();
			if (vertex.getX() > max_x)
				max_x = vertex.getX();
			if (vertex.getY() < min_y)
				min_y = vertex.getY();
			if (vertex.getY() > max_y)
				max_y = vertex.getY();
		}
		min_x -= 1d;
		min_y -= 1d;
		max_x += 1d;
		max_y += 1d;

		double diffx = max_x - min_x;
		double diffy = max_y - min_y;

		if (diffx > diffy) {
			max_y = min_y + diffx;
		} else {
			max_x = min_x + diffy;
		}

		LinkedList<List<Location>> vrts = new LinkedList<List<Location>>();
		vrts.add((List<Location>) vertices);

		root = new QuadTreeNode(this, min_x, min_y, max_x, max_y, null, vrts);
	}

	/**
	 * Retrieves the root node.
	 * 
	 * @return root node
	 */
	public final QuadTreeNode getRoot() {
		return root;
	}
	
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();

		builder.append(String.format("QuadTree: [\n%s\n]", root.toString()));
		
		return builder.toString();
	}

	/**
	 * Returns the iterator over QuadTree. The order of evaluation is prefix
	 * with [ ul, ur, ll, lr ] as the order of nodes.
	 */
	@Override
	public Iterator<QuadTreeNode> iterator() {
		return new QuadTreePostorderIterator(this);
	}
}
