package cz.cuni.amis.pogamut.defcon.communication.worldview;

import java.lang.reflect.Field;

import com.google.inject.Inject;
import com.google.inject.name.Named;

import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldChangeEvent;
import cz.cuni.amis.pogamut.base.component.bus.IComponentBus;
import cz.cuni.amis.pogamut.base.component.controller.ComponentDependencies;
import cz.cuni.amis.pogamut.base.utils.guice.AgentScoped;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.base3d.worldview.impl.LockableBatchAwareWorldView;
import cz.cuni.amis.pogamut.base3d.worldview.object.IViewable;
import cz.cuni.amis.pogamut.defcon.agent.module.sensor.GameInfo;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.DefConViewableObject;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.EventBatchBegin;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.EventBatchEnd;
import cz.cuni.amis.pogamut.defcon.communication.translator.DefConMessageProducer;
import cz.cuni.amis.utils.exception.PogamutException;

/**
 * Worldview implementation specific for defcon.
 * 
 * @author Radek 'Black_Hand' Pibil
 * 
 */
@AgentScoped
public class DefConWorldView extends LockableBatchAwareWorldView {
	public static final String WORLDVIEW_DEPENDENCY = "DefConWorldViewDependency";

	private Field visibleField = null;

	private DefConMessageProducer messageProducer;

	private final static double maxX = 180d;
	private final static double minX = -180d;
	private final static double maxY = 100d;
	private final static double minY = -100d;

	@Inject
	public DefConWorldView(
			@Named(WORLDVIEW_DEPENDENCY) ComponentDependencies dependencies,
			DefConMessageProducer messageProducer, IComponentBus bus,
			IAgentLogger log) {
		super(dependencies, bus, log);
		this.messageProducer = messageProducer;
	}

	@Override
	protected boolean isBatchBeginEvent(IWorldChangeEvent evt) {
		return evt instanceof EventBatchBegin;
	}

	@Override
	protected boolean isBatchEndEvent(IWorldChangeEvent evt) {
		return evt instanceof EventBatchEnd;
	}

	@Override
	protected void setDisappearedFlag(IViewable obj) {
		if (obj instanceof DefConViewableObject) {
			try {
				getField((DefConViewableObject) obj).set(obj, false);
			} catch (Exception e) {
				throw new PogamutException("Can't set 'visible' field: "
						+ e.getMessage(), e, this);
			}
		}
	}

	private Field getField(DefConViewableObject obj) {
		if (visibleField == null) {
			try {
				visibleField = DefConViewableObject.class
						.getDeclaredField("visible");
				visibleField.setAccessible(true);
			} catch (Exception e) {
				throw new PogamutException("DefConViewableObject " + obj
						+ " doesn't contain 'visible' field???", e, this);
			}
		}
		return visibleField;
	}

	/**
	 * Updates message producer and reports the event.
	 */
	public void update() {
		if (eventBus.isRunning()) {
			lock();
			IWorldChangeEvent event;
			while ((event = messageProducer.getEvent()) != null) {
				notify(event);
			}
			unlock();
		}
	}

	public float getStartTime() {
		return messageProducer.getStartTime();
	}

	public float getCurrentTime() {
		return messageProducer.getCurrentTime();
	}

	/**
	 * Returns game info object used to query defcon.
	 * 
	 * @return
	 */
	public GameInfo getGameInfo() {
		return messageProducer.getGameInfo();
	}

	/**
	 * Returns minimal X value.
	 * 
	 * @return
	 */
	public double getMinX() {
		return minX;
	}

	/**
	 * Returns maximal X value.
	 * 
	 * @return
	 */
	public double getMaxX() {
		return maxX;
	}

	/**
	 * Returns minimal Y value.
	 * 
	 * @return
	 */
	public double getMinY() {
		return minY;
	}

	/**
	 * Returns maximal Y value.
	 * 
	 * @return
	 */
	public double getMaxY() {
		return maxY;
	}

	/**
	 * Returns width of the map.
	 * 
	 * @return
	 */
	public double getXSize() {
		return maxX - minX;
	}

	/**
	 * Returns height of the map.
	 * 
	 * @return
	 */
	public double getYSize() {
		return maxY - minY;
	}

	/**
	 * Updates the message producer.
	 */
	public void updateProducer() {
		messageProducer.update();
	}
}
