package javabot;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import javabot.JBot.EventData;
import javabot.JBot.FleetData;
import javabot.JBot.UnitData;
import javabot.events.DefConBasicUpdate;
import javabot.events.IDefConBasicEvent;
import cz.cuni.amis.pogamut.base.agent.state.impl.AgentState;
import cz.cuni.amis.pogamut.base.agent.state.impl.AgentStateStarted;
import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldChangeEvent;
import cz.cuni.amis.pogamut.defcon.agent.DefConAgent;
import cz.cuni.amis.pogamut.defcon.agent.module.sensor.GameInfo;
import cz.cuni.amis.pogamut.defcon.agentmanager.DefConAgentManager;
import cz.cuni.amis.pogamut.defcon.agentmanager.exception.CantInstantiateAgentException;
import cz.cuni.amis.pogamut.defcon.agentmanager.exception.ModuleForAgentClassNotFoundException;
import cz.cuni.amis.pogamut.defcon.base3d.worldview.object.DefConLocation;
import cz.cuni.amis.pogamut.defcon.communication.messages.commands.DefConCommand;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.City;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.DefConEvent;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.DefConObject;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.DefConUnitObject;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.Fleet;
import cz.cuni.amis.pogamut.defcon.consts.Event;
import cz.cuni.amis.pogamut.defcon.consts.UnitType;
import cz.cuni.amis.pogamut.defcon.factory.DefConAgentModule;
import cz.cuni.amis.pogamut.defcon.utils.SyncMethodExecContainer;
import cz.cuni.amis.utils.ExceptionToString;
import cz.cuni.amis.utils.exception.PogamutInterruptedException;
import cz.cuni.amis.utils.flag.Flag;
import cz.cuni.amis.utils.flag.IFlag;
import cz.cuni.amis.utils.flag.WaitForFlagChange;
import cz.cuni.amis.utils.flag.WaitForFlagChange.IAccept;

/**
 * Layer that takes care of incoming and outgoing requests on defcon.
 * 
 * @author Radek 'Black_Hand' Pibil
 * 
 */
public class PogamutJBotSupport {

	private static final String NEW_LINE = System.getProperty("line.separator");

	public static final String PROPERTY_CLASS = "module_class";
	public static final String PROPERTY_START_TIMEOUT_SECS = "start_timeout";

	private static final int DEFAULT_START_TIMEOUT_SECS = 10;
	private static final long MAX_UPDATE_TIME = 75;

	private static DefConAgent bot = null;

	private static final ConcurrentLinkedQueue<SyncMethodExecContainer> queries =
			new ConcurrentLinkedQueue<SyncMethodExecContainer>();

	private static final LinkedBlockingQueue<IDefConBasicEvent> events = new LinkedBlockingQueue<IDefConBasicEvent>();

	private static final LinkedBlockingQueue<IWorldChangeEvent> unitUpdates = new LinkedBlockingQueue<IWorldChangeEvent>();

	private static final int COMMANDS_PER_TICK = 20;

	/**
	 * Leave as not-immutable for testing purposes!
	 */

	public static Flag<Boolean> botIsRunning = new Flag<Boolean>(false);

	private static ConcurrentLinkedQueue<DefConCommand> commands = new ConcurrentLinkedQueue<DefConCommand>();

	private static ActExecutor actExecutor = new ActExecutor();

	private static String mapToString(Map<?, ?> map) {
		StringBuffer sb = new StringBuffer();
		for (Entry<?, ?> entry : map.entrySet()) {
			sb.append(entry.getKey() + " = " + entry.getValue());
			sb.append(NEW_LINE);
		}
		return sb.toString();
	}

	private static boolean start(Map<String, String> options) {
		logInit(Level.INFO, "Instantiating bot with options: " + NEW_LINE
				+ mapToString(options));

		String clsStr = options.get(PROPERTY_CLASS);
		Class<DefConAgentModule> cls = null;
		if (clsStr != null) {
			try {
				cls = (Class<DefConAgentModule>) Class.forName(clsStr);
			} catch (Exception e) {
				logInit(
						Level.SEVERE,
						"Bot module class '"
								+ clsStr
								+ "' could not be found, did you put your jar into 'java' directory?");
				logInit(Level.SEVERE, "BOT CAN NOT BE STARTED!");
				logInit(Level.SEVERE, "RESTART DEFCON! " + e.toString() + " "
						+ System.getProperty("java.class.path"));
				return false;
			}
			logInit(Level.INFO, "Bot module class '" + clsStr + "'.");
		} else {
			logInit(Level.INFO, "Bot module class ('" + PROPERTY_CLASS
					+ "' options) not specified.");
		}

		logInit(Level.INFO, "Instantiating bot.");

		try {
			bot = (DefConAgent<?>) DefConAgentManager.getInstance()
					.getAgentInstance(cls);

		} catch (CantInstantiateAgentException e) {
			logInit(Level.SEVERE, ExceptionToString.process(e));
			logInit(Level.SEVERE, "BOT CAN NOT BE STARTED!");
			logInit(Level.SEVERE, "RESTART DEFCON!");
			return false;
		} catch (ModuleForAgentClassNotFoundException e) {
			logInit(Level.SEVERE, ExceptionToString.process(e));
			logInit(Level.SEVERE, "BOT CAN NOT BE STARTED!");
			logInit(Level.SEVERE, "RESTART DEFCON!");
			return false;
		}

		logInit(Level.INFO, "Bot was instantiated.");

		String startTimeoutStr = options.get(PROPERTY_START_TIMEOUT_SECS);
		int startTimeoutSecs = DEFAULT_START_TIMEOUT_SECS;
		if (startTimeoutStr != null) {
			try {
				startTimeoutSecs = Integer.parseInt(startTimeoutStr);
			} catch (Exception e) {
				logInit(Level.WARNING, "Option '" + PROPERTY_START_TIMEOUT_SECS
						+ "' does not contain integer number.");
			}
		}

		logInit(Level.INFO, "Starting bot (" + startTimeoutSecs
				+ " secs timeout).");

		bot.setOptions(options);
		try {
			bot.start();
			try {
				new WaitForFlagChange<AgentState>((IFlag) bot.getState(),
						new IAccept<AgentState>() {
							@Override
							public boolean accept(AgentState flagValue) {
								return flagValue
										.isState(AgentStateStarted.class);
							}
						}).await(startTimeoutSecs, TimeUnit.SECONDS);
			} catch (PogamutInterruptedException e) {
				logInit(Level.SEVERE, "Bot fails to start in "
						+ startTimeoutSecs + " secs, killing bot.");
				bot.kill();
				logInit(Level.SEVERE, "BOT CAN NOT BE STARTED!");
				logInit(Level.SEVERE, "RESTART DEFCON!");
				return false;
			}
		} catch (Exception e) {
			logInit(Level.SEVERE, ExceptionToString.process(
					"Bot fails to start.", e));
			bot = null;
			logInit(Level.SEVERE, "BOT CAN NOT BE STARTED!");
			logInit(Level.SEVERE, "RESTART DEFCON!");
			return false;
		}
		logInit(Level.INFO, "Bot running!");
		botIsRunning.setFlag(true);
		return true;
	}

	public static DefConAgent<?> getBot() {
		return bot;
	}

	private static String[] split(String msg, int len) {
		if (msg.length() < len)
			return new String[] { msg };
		String[] result = new String[msg.length() / len + 1];
		for (int i = 0; i < msg.length() / len + 1; ++i) {
			if ((i + 1) * len < msg.length()) {
				result[i] = msg.substring(i * len, (i + 1) * len);
			} else {
				result[i] = msg.substring(i * len);
			}
		}
		return result;
	}

	public static void logInitException(Throwable e) {
		StringWriter writer = new StringWriter();
		e.printStackTrace(new PrintWriter(writer));
		writeToConsole(writer.toString());
	}

	public static void logGameException(Throwable e) {
		StringWriter writer = new StringWriter();
		e.printStackTrace(new PrintWriter(writer));
		writeToConsole(writer.toString());
	}

	/**
	 * Formats and send a log message during INIT phase.
	 * 
	 * @param level
	 * @param msg
	 */
	public static void logInit(Level level, String msg) {
		if (botIsRunning.getFlag()) {
			String[] msgs = split("[" + level + "] " + msg, 50);
			for (String m : msgs) {
				writeToConsole(m);
			}
		} else {
			System.out.println("[" + level + "] " + msg);
		}
	}

	/**
	 * Formats and send a log message during GAME phase.
	 * 
	 * @param level
	 * @param msg
	 */
	public static void logGame(Level level, String msg) {
		if (botIsRunning.getFlag()) {
			JBot.DebugLog("[" + level + "] " + msg);
		} else {
			System.out.println("[" + level + "] " + msg);
		}
	}

	public static GameInfo gameInfo = null;

	/**
	 * Called from JBot.initialise().
	 * 
	 * @param commandLineOptions
	 * @return
	 */
	public static boolean initialise(String[][] commandLineOptions) {
		PogamutJBotSupport
				.writeToConsole("Executing PogamutJBotSupport initialise");
		logGame(Level.INFO, "Executing PogamutJBotSupport initialise");
		StringBuffer sb = new StringBuffer();
		sb.append("Command line options:");
		Map<String, String> optionMap = new HashMap<String, String>();
		for (String[] option : commandLineOptions) {
			sb.append(NEW_LINE);
			if (option.length > 0) {
				logInit(Level.INFO, option[0]
						+ (option.length > 1 && option[1].length() > 0 ? " = "
								+ option[1] : ""));
				optionMap.put(option[0], option.length > 1 ? option[1] : null);
			}
		}
		logGame(Level.INFO, "Calling start");
		boolean result = start(optionMap);
		JBot.WriteToConsole("Finished start");

		Thread.currentThread().setPriority(
				(int) ((Thread.MAX_PRIORITY - Thread.NORM_PRIORITY) / 3d)
						+ Thread.NORM_PRIORITY);

		return result;
	}

	/**
	 * Called from JBot.addEvent().
	 * 
	 * @param data
	 */
	public static void addEvent(EventData data) {
		synchronized (events) {

			if (data.m_eventType != JBot.EventDestroyed) {	

				DefConEvent event = Event.getInstanceOfUnitTypeFromEventType(
						data,
						bot
								.getWorldView().getCurrentTime());

				events.add(event);
			} else {

				UnitData unitdata = new UnitData();

				unitdata.m_objectId = data.m_targetObjectId;
				unitdata.m_teamId = JBot.GetTeamId(data.m_targetObjectId);
				unitdata.m_type = JBot.GetType(data.m_targetObjectId);
				unitdata.m_currentState = -1;
				unitdata.m_visible = true;
				unitdata.m_longitude = data.m_longitude;
				unitdata.m_latitude = data.m_latitude;				

				if (UnitType.naval.contains(UnitType
						.getEnum(unitdata.m_type))) {

					int fleetId = JBot.GetFleetId(data.m_targetObjectId);

					int[] fleetMembers = JBot.GetFleetMembers(fleetId);

					if (fleetMembers.length == 1) {

						Fleet fleet_object = new Fleet(fleetId,
								unitdata.m_teamId,
								new DefConLocation(data.m_longitude,
										data.m_latitude),
								false, fleetMembers, bot
										.getWorldView().getCurrentTime());

						unitUpdates.add(fleet_object.createDestroyedEvent());
					}

				}

				DefConUnitObject<?> update_object = (DefConUnitObject<?>) UnitType
						.getInstanceOfUnitTypeFromUnitData(unitdata, bot
								.getWorldView().getCurrentTime());
				unitUpdates.add(update_object.createDestroyedEvent());
			}
		}
	}

	/**
	 * Called from JBot.update().
	 * 
	 * @return
	 */
	public static boolean update() {
		// JBot.WriteToConsole("PogamutJBotSupport.update(); Time: " +
		// System.currentTimeMillis());
		long current_time = System.currentTimeMillis();

		if (JBot.IsVictoryTimerActive() && JBot.GetVictoryTimer() == 0) {
			writeToConsole("Match result");
			for (int teamId : JBot.GetTeamIds()) {
				writeToConsole("TeamId: " + teamId);
				writeToConsole("EnemyKills: " + JBot.GetEnemyKills(teamId));
				writeToConsole("CollateralDamage: "
						+ JBot.GetCollateralDamage(teamId));
				writeToConsole("Casualties: "
						+ JBot.GetFriendlyDeaths(teamId));
				writeToConsole("RemainingPopulation: "
						+ JBot.GetRemainingPopulation(teamId));
			}
			try {
				Thread.sleep(1000);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
			System.exit(0);
		}

		performCommands();

		prepareUnitsUpdate();

		synchronized (events) {
			// TODO: FIX! IDIOTIC!
			events.add(new DefConBasicUpdate((long) JBot.GetGameTime()));
		}

		while (System.currentTimeMillis() - current_time < MAX_UPDATE_TIME) {
			checkQueries();
		}

		return true;
	}

	/**
	 * Collects all updates of all visible units.
	 * 
	 * @return LinkedList of all collected events.
	 */
	private static void prepareUnitsUpdate() {

		synchronized (unitUpdates) {
			List<UnitData> unit_data = JBot.GetAllUnitData();

			float time = JBot.GetGameTime();
			for (UnitData unit : unit_data) {
				DefConObject object = UnitType
						.getInstanceOfUnitTypeFromUnitData(
								unit,
								time);
				unitUpdates.add(object.createUpdateEvent());
			}


			List<FleetData> fleet_data = JBot.GetAllFleetData();

			for (FleetData fleet : fleet_data) {

				Fleet object = new Fleet(fleet.m_fleetId,
						fleet.m_teamId,
						new DefConLocation(fleet.m_longitude, fleet.m_latitude),
						fleet.m_visible, fleet.m_fleetMembers, time);

				if (object.getFleetMembers().length > 0)
					unitUpdates.add(object.createUpdateEvent());
			}
			for (int cityId : JBot.GetCityIds()) {
				DefConLocation cityLocation = new DefConLocation(
						JBot.GetLongitude(cityId),
						JBot.GetLatitude(cityId));

				int ownerId = JBot.GetTeamId(cityId);
				City city = new City(cityId, ownerId, cityLocation, true,
						JBot.GetCityPopulation(cityId), JBot.GetGameTime());

				unitUpdates.add(city.createUpdateEvent());
			}
		}
	}

	public static List<IWorldChangeEvent> getUnitsUpdate() {
		List<IWorldChangeEvent> update = new ArrayList<IWorldChangeEvent>();
		unitUpdates.drainTo(update);

		return update;
	}

	private static void checkQueries() {
		SyncMethodExecContainer container = queries.poll();

		if (container != null) {
			container.execute();			
		}
	}

	public synchronized static void addQuery(SyncMethodExecContainer container) {
		queries.add(container);
	}

	private static void performCommands() {		
		int counter = COMMANDS_PER_TICK;

		while (--counter > 0 && !commands.isEmpty()) {
			actExecutor.sendCommand(commands.poll());
		}
	}

	public static void addCommand(DefConCommand command) {
		synchronized (commands) {
			commands.add(command);
		}
	}

	public static List<IDefConBasicEvent> getEvents() {
		List<IDefConBasicEvent> result = new ArrayList<IDefConBasicEvent>();

		synchronized (events) {
			events.drainTo(result);
		}
		return result;
	}

	public static void writeToConsole(String logLine) {
		if (logLine != null) {
			JBot.WriteToConsole(logLine);
		} else {
			JBot.WriteToConsole("NULL LOGLINE");
			for (StackTraceElement element : Thread.currentThread()
					.getStackTrace()) {
				JBot.WriteToConsole(element.toString());
			}
		}
	}

	public static ActExecutor getDefConActExecutor() {
		return actExecutor;
	}

	public static void setName(String name) {
		JBot.SendChatMessage(String.format("/name [Bot]%s", name), 0);
	}

}