package cz.cuni.amis.pogamut.defcon.communication.translator;

import java.util.LinkedList;
import java.util.List;

import javabot.PogamutJBotSupport;
import javabot.events.DefConBasicUpdate;
import javabot.events.IDefConBasicEvent;

import com.google.inject.Inject;

import cz.cuni.amis.pogamut.base.communication.exception.CommunicationException;
import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldChangeEvent;
import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldChangeEventOutput;
import cz.cuni.amis.pogamut.base.component.bus.IComponentBus;
import cz.cuni.amis.pogamut.base.component.controller.ComponentControlHelper;
import cz.cuni.amis.pogamut.base.component.controller.ComponentController;
import cz.cuni.amis.pogamut.base.component.controller.ComponentDependencies;
import cz.cuni.amis.pogamut.base.component.controller.IComponentControlHelper;
import cz.cuni.amis.pogamut.base.utils.guice.AgentScoped;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.defcon.agent.module.sensor.GameInfo;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.EventBatchBegin;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.EventBatchEnd;
import cz.cuni.amis.pogamut.defcon.communication.worldview.DefConWorldView;
import cz.cuni.amis.utils.token.IToken;
import cz.cuni.amis.utils.token.Token;
import cz.cuni.amis.utils.token.Tokens;

/**
 * This class retrieves events from GameInfo class and prepares them for the next stage of
 * processing in {@link DefConWorldView}.
 *  
 * @author Radek 'Black_Hand' Pibil
 *
 */
@AgentScoped
public class DefConMessageProducer implements IWorldChangeEventOutput {
	
	public static final Token COMPONENT_ID = Tokens.get("DefConMessageProducer");	

	/**
	 * Enqueued events ready for processing.
	 */
	private LinkedList<IWorldChangeEvent> eventQueue = new LinkedList<IWorldChangeEvent>();
	
	private float startTime;
	
	private boolean defConStarted = false;
	
	private IComponentBus eventBus;

	private ComponentController controller;	
	
	private LogCategory log;
	
	private GameInfo gameInfo;	
			
	@Inject
	public DefConMessageProducer(
			IComponentBus eventBus,
			IAgentLogger logger) {
		
		log = logger.getCategory(COMPONENT_ID.getToken());
		this.eventBus = eventBus;
		this.controller = new ComponentController(this, control, this.eventBus, log,
				new ComponentDependencies());
		
		gameInfo = new GameInfo();
	}

	private IComponentControlHelper control = new ComponentControlHelper() {
	
		@Override
		public void start() {
			eventQueue.clear();
			eventQueue.add(new EventBatchBegin(0));		
			startTime = gameInfo.getGameTime();
			defConStarted = false;
		}
		
		@Override
		public void reset() {
			start();
		}		
	};	
	
	/**
	 * Used by worldview to retrieve one enqueued event.
	 * Works only if the game has been started.
	 * 
	 * @return dequeued event
	 */
	@Override
	public IWorldChangeEvent getEvent() throws CommunicationException {
		if (!defConStarted) {
			if (!gameInfo.getRunning())
				return null;

			startTime = gameInfo.getGameTime();
			defConStarted = true;
		}

		if (eventQueue.size() != 0)
			return eventQueue.poll();
		else
			return null;
	}
	
	/**
	 * Retrieves all events from GameInfo instance and stores them in the eventQueue.
	 * Unit updates are retrieved as well by {@link getUnitsUpdate()}.
	 * <p><p>
	 * It also creates {@link EventBatchEnd} and {@link EventBatchBegin} between each of the retrievals
	 * to ensure proper batch processing.
	 */
	private void populateQueue() {
		List<IDefConBasicEvent> events = PogamutJBotSupport.getEvents();
		for (IDefConBasicEvent event : events) {
			if (event instanceof DefConBasicUpdate) {
				// BATCH END
				eventQueue.addAll(gameInfo.getEvents());
				eventQueue.addAll(getUnitsUpdate());
				eventQueue.add(new EventBatchEnd(gameInfo.getGameTime()));
				eventQueue.add(new EventBatchBegin(gameInfo.getGameTime()));
			} else {
				eventQueue.add(event);
			}
		}
	}

	/**
	 * Collects all updates of all visible units.
	 * 
	 * @return LinkedList of all collected events.
	 */
	private List<IWorldChangeEvent> getUnitsUpdate() {
		return PogamutJBotSupport.getUnitsUpdate();
	}

	@Override
	public String toString() {
		return "DefConMessageProducer[startTime=" + startTime + "]";
	}

	@Override
	public IToken getComponentId() {
		return COMPONENT_ID;
	}
	
	/**
	 * Since update is called periodically from worldview, we use it to call populateQueue() periodically
	 */
	public void update() {
		populateQueue();
	}

	public float getStartTime() {
		return startTime;
	}

	public float getCurrentTime() {
		return gameInfo.getGameTick();
	}
	
	public GameInfo getGameInfo() {
		return gameInfo;
	}
}
