package cz.cuni.amis.pogamut.defcon.agent.module.sensor;

import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import javabot.JBot;
import javabot.JBotMethodsRepository;
import javabot.PogamutJBotSupport;
import javabot.events.IDefConBasicEvent;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.defcon.base3d.worldview.object.DefConLocation;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.DebugIsReplayingGameChanged;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.DefConChanged;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.GameRunningChanged;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.GameSpeedChanged;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.VictoryTimerActiveChanged;
import cz.cuni.amis.pogamut.defcon.consts.GameSpeed;
import cz.cuni.amis.pogamut.defcon.consts.UnitType;
import cz.cuni.amis.pogamut.defcon.consts.state.IState;
import cz.cuni.amis.pogamut.defcon.utils.AdvancedFlagListener;
import cz.cuni.amis.pogamut.defcon.utils.SyncMethodExecContainer;
import cz.cuni.amis.pogamut.defcon.utils.SimpleFlag;

/**
 * This class implements functionality that is not present within standard worldview,
 * but it is important since it provides crucial information about the games' state.
 * 
 * @author Radek 'Black_Hand' Pibil
 *
 */
public class GameInfo {

	private final static int MAX_FLEET_SIZE = 6;

	/**
	 * Contains reverse mapping from territory to territory owner.
	 */
	private final int[] territoryOwners = new int[JBot.NumTerritories];
	/**
	 * Holds fleet diameters.
	 */
	private final double[] fleetDiameters = new double[MAX_FLEET_SIZE];

	/**
	 * Holds fleet member offsets.
	 */
	private final Location[][] fleetMemberOffsets = new Location[MAX_FLEET_SIZE][];

	/**
	 * Keeps track of game time.
	 */
	private Float gameTime = 0f;

	/**
	 * Holds the amount of ticks (updates) the AI has passed.
	 */
	private Integer gameTick = 0;

	/**
	 * Time remaining in game after victory timer has been started.
	 */
	private Float victoryTimer = 0f;

	/**
	 * Own team id.
	 */
	private int ownTeamId;

	/**
	 * Enemy cities ids list.
	 */
	private final SortedMap<Integer, List<Integer>> enemiesCityIds = new TreeMap<Integer, List<Integer>>();

	/**
	 * Own cities ids list.
	 */
	private final List<Integer> ownCityIds = new ArrayList<Integer>();

	/**
	 * List of cities locations.
	 */
	private final SortedMap<Integer, Location> cityLocations = new TreeMap<Integer, Location>();

	/**
	 * Enemy team ids list.
	 */
	private int[] enemyIds;

	/**
	 * Holds current defcon.
	 */
	private final SimpleFlag<Integer> defcon = new SimpleFlag<Integer>(5);
	/**
	 * Adds defcon changed event into event queue, which then gets passed to the worldview.
	 */
	private final AdvancedFlagListener<Integer> defconListener = new AdvancedFlagListener<Integer>() {

		@Override
		public void flagChanged(Integer oldValue, Integer changedValue) {
			events.add(new DefConChanged(oldValue, changedValue, getGameTime()));
		}

	};

	/**
	 * Holds current game speed.
	 */
	private final SimpleFlag<GameSpeed> gameSpeed = new SimpleFlag<GameSpeed>(
			GameSpeed.PAUSED);
	/**
	 * Adds game speed changed event into event queue, which then gets passed to the worldview.
	 */	
	private final AdvancedFlagListener<GameSpeed> gameSpeedListener = new AdvancedFlagListener<GameSpeed>() {

		@Override
		public void flagChanged(GameSpeed oldValue, GameSpeed changedValue) {
			events.add(new GameSpeedChanged(oldValue, changedValue,
					getGameTime()));
		}

	};

	/**
	 * Holds current state of victory timer.
	 */
	private final SimpleFlag<Boolean> victoryTimerActive = new SimpleFlag<Boolean>(
			false);
	/**
	 * Adds victory timer changed event into event queue, which then gets passed to the worldview.
	 */
	private AdvancedFlagListener<Boolean> victoryTimerActiveListener = new AdvancedFlagListener<Boolean>() {

		@Override
		public void flagChanged(Boolean oldValue, Boolean changedValue) {
			events.add(new VictoryTimerActiveChanged(changedValue,
					getGameTime()));
		}

	};

	/**
	 * Holds current state of debug replaying.
	 */	
	private final SimpleFlag<Boolean> debugIsReplayingGame = new SimpleFlag<Boolean>(false);
	/**
	 * Adds debug is replaying game changed event into event queue, which then gets passed to the worldview.
	 */	
	private final AdvancedFlagListener<Boolean> debugIsReplayingGameListener = new AdvancedFlagListener<Boolean>() {

		@Override
		public void flagChanged(Boolean oldValue, Boolean changedValue) {
			events.add(new DebugIsReplayingGameChanged(changedValue,
					getGameTime()));
		}

	};

	/**
	 * Holds current state of debug replaying.
	 */	
	private final SimpleFlag<Boolean> gameRunning = new SimpleFlag<Boolean>(
			false);
	/**
	 * Adds victory timer changed event into event queue, which then gets passed to the worldview.
	 */	
	private AdvancedFlagListener<Boolean> gameRunningListener = new AdvancedFlagListener<Boolean>() {

		@Override
		public void flagChanged(Boolean oldValue, Boolean changedValue) {
			if (changedValue) {
				cacheOwnTeamId();
				populateTerritoryOwners();
				populateCityLocations();
			}
			events.add(new GameRunningChanged(changedValue, getGameTime()));
		}
	};

	/**
	 * Holds the queue of events to be passed to worldview.
	 */
	private final BlockingQueue<IDefConBasicEvent> events = new LinkedBlockingQueue<IDefConBasicEvent>();

	/**
	 * Instantiates and prepares GameInfo for operation.
	 */
	public GameInfo() {		
		gameTime = JBot.GetGameTime();
		gameTick = JBot.GetGameTick();
		victoryTimer = 0f;
		gameRunning.addStrongListener(gameRunningListener);
		defcon.addStrongListener(defconListener);
		gameSpeed.addStrongListener(gameSpeedListener);
		victoryTimerActive.addStrongListener(victoryTimerActiveListener);
		debugIsReplayingGame.addStrongListener(debugIsReplayingGameListener);
		populateFleetMemberOffsets();
		populateFleetDiameters();
	}

	/**
	 * Caches all cities locations.
	 */
	private void populateCityLocations() {


		int[] cities = getCityIds();

		for (int cityId : cities) {

			Location location = getObjectLocation(cityId);

			int territoryId = getTerritoryId(location);

			int teamId = territoryOwners[territoryId];

			cityLocations.put(cityId, location);

			if (teamId == -1)
				continue;

			if (teamId == getOwnTeamId()) {
				ownCityIds.add(cityId);
			} else {
				enemiesCityIds.get(teamId).add(cityId);
			}
		}
	}

	/**
	 * Caches all territory to owner mappings.
	 */
	private final void populateTerritoryOwners() {

		for (int i = 0; i < territoryOwners.length; ++i) {
			territoryOwners[i] = -1;
		}

		ArrayList<Integer> enemies = new ArrayList<Integer>();
		int own_id = getOwnTeamId();

		for (int teamId : JBot.GetTeamIds()) {

			if (teamId != own_id) {
				enemies.add(teamId);
				enemiesCityIds.put(teamId, new ArrayList<Integer>());
			}

			for (int territoryId : getTeamTerritories(teamId)) {
				territoryOwners[territoryId] = teamId;
			}
		}

		enemyIds = new int[enemies.size()];

		int i = 0;
		for (int enemyId : enemies) {
			enemyIds[i++] = enemyId;
		}
	}

	/**
	 * Caches fleet members offsets.
	 */
	private final void populateFleetMemberOffsets() {
		for (int i = 0; i < MAX_FLEET_SIZE; ++i) {
			fleetMemberOffsets[i] = new Location[i + 1];
			for (int j = 0; j <= i; ++j) {
				fleetMemberOffsets[i][j] = getFleetMemberOffsetWorker(i + 1, j);
			}
		}
	}

	/**
	 * Caches fleet diameters per fleet size.
	 */
	private final void populateFleetDiameters() {
		for (int i = 0; i < MAX_FLEET_SIZE; ++i) {
			fleetDiameters[i] = getFleetDiameterWorker(i + 1);
		}
	}

	/**
	 * Caches own Id.
	 */
	private void cacheOwnTeamId() {
		ownTeamId = JBot.GetOwnTeamId();
	}

	/**
	 * Updates all flags, which in turn generates events, which are then returned in a form
	 * of LinkedList<IDefconBasicEvent>.
	 * This method is periodically called by DefconMessageProdducer.populateQueue().
	 * @return LinkedList of acquired events.
	 */
	public LinkedList<IDefConBasicEvent> getEvents() {
		gameTime = JBot.GetGameTime();
		gameTick = JBot.GetGameTick();

		victoryTimerActive.setFlag(JBot.IsVictoryTimerActive());

		if (victoryTimerActive.getFlag()) {
			victoryTimer = JBot.GetVictoryTimer();
		}

		gameRunning.setFlag(true);
		defcon.setFlag(JBot.GetDefcon());
		gameSpeed.setFlag(GameSpeed.getEnum(JBot.GetGameSpeed()));
		victoryTimerActive.setFlag(JBot.IsVictoryTimerActive());
		debugIsReplayingGame.setFlag(JBot.DebugIsReplayingGame());

		LinkedList<IDefConBasicEvent> output = new LinkedList<IDefConBasicEvent>();
		events.drainTo(output);
		return output;
	}
	
	private final LinkedList<Location> aiPlacementPoints = new LinkedList<Location>();
	private final LinkedList<Location> targetCoords = new LinkedList<Location>();
	
	/**
	 * Initializes default aiPlacementPoints.
	 * Borrowed from the reimplementation of the original Defcon bot.
	 */
	private void populatePoints()
	{
		// This data was originally stored in bitmaps and then read out when the game starts.
		// However, the bitmap loading procedures are integrated deep into the game engine, so this has
		// been left away.
		checkedAddToAIPlacementPoints(new Location(-177.890625f,34.736842f));
		checkedAddToAIPlacementPoints(new Location(-177.187500f,46.666667f));
		checkedAddToAIPlacementPoints(new Location(-156.093750f,45.964912f));
		checkedAddToAIPlacementPoints(new Location(-151.875000f,35.438596f));
		checkedAddToAIPlacementPoints(new Location(-146.953125f,22.807018f));
		targetCoords.add(new Location(-142.031250f,45.964912f));
		targetCoords.add(new Location(-136.406250f,28.421053f));
		checkedAddToAIPlacementPoints(new Location(-133.593750f,20.701754f));
		checkedAddToAIPlacementPoints(new Location(-123.046875f,26.315789f));
		targetCoords.add(new Location(-118.828125f,9.473684f));
		checkedAddToAIPlacementPoints(new Location(-113.203125f,-9.473684f));
		checkedAddToAIPlacementPoints(new Location(-106.171875f,6.666667f));
		checkedAddToAIPlacementPoints(new Location(-93.515625f,-6.666667f));
		targetCoords.add(new Location(-93.515625f,-0.350877f));
		checkedAddToAIPlacementPoints(new Location(-91.406250f,-18.596491f));
		targetCoords.add(new Location(-88.593750f,-27.719298f));
		targetCoords.add(new Location(-64.687500f,29.122807f));
		checkedAddToAIPlacementPoints(new Location(-56.250000f,22.807018f));
		checkedAddToAIPlacementPoints(new Location(-55.546875f,35.438596f));
		targetCoords.add(new Location(-50.625000f,16.491228f));
		targetCoords.add(new Location(-47.812500f,36.842105f));
		checkedAddToAIPlacementPoints(new Location(-44.296875f,31.228070f));
		targetCoords.add(new Location(-43.593750f,48.771930f));
		checkedAddToAIPlacementPoints(new Location(-42.890625f,41.754386f));
		checkedAddToAIPlacementPoints(new Location(-40.078125f,4.561404f));
		checkedAddToAIPlacementPoints(new Location(-37.968750f,22.105263f));
		checkedAddToAIPlacementPoints(new Location(-37.265625f,-30.526316f));
		targetCoords.add(new Location(-31.640625f,-17.894737f));
		checkedAddToAIPlacementPoints(new Location(-31.640625f,35.438596f));
		targetCoords.add(new Location(-30.234375f,-35.438596f));
		checkedAddToAIPlacementPoints(new Location(-29.531250f,45.263158f));
		checkedAddToAIPlacementPoints(new Location(-28.828125f,9.473684f));
		checkedAddToAIPlacementPoints(new Location(-28.125000f,24.912281f));
		checkedAddToAIPlacementPoints(new Location(-26.718750f,0.350877f));
		checkedAddToAIPlacementPoints(new Location(-25.312500f,-17.894737f));
		targetCoords.add(new Location(-23.203125f,13.684211f));
		targetCoords.add(new Location(-22.500000f,31.228070f));
		checkedAddToAIPlacementPoints(new Location(-21.796875f,38.947368f));
		checkedAddToAIPlacementPoints(new Location(-21.796875f,52.280702f));
		targetCoords.add(new Location(-21.093750f,45.263158f));
		checkedAddToAIPlacementPoints(new Location(-14.765625f,35.438596f));
		checkedAddToAIPlacementPoints(new Location(-14.062500f,47.368421f));
		checkedAddToAIPlacementPoints(new Location(-11.953125f,-5.964912f));
		checkedAddToAIPlacementPoints(new Location(-10.546875f,60.000000f));
		targetCoords.add(new Location(-9.140625f,68.421053f));
		targetCoords.add(new Location(0.000000f,69.122807f));
		targetCoords.add(new Location(1.406250f,-4.561404f));
		checkedAddToAIPlacementPoints(new Location(2.812500f,73.333333f));
		checkedAddToAIPlacementPoints(new Location(3.515625f,-19.298246f));
		checkedAddToAIPlacementPoints(new Location(4.218750f,67.017544f));
		targetCoords.add(new Location(6.328125f,-31.929825f));
		checkedAddToAIPlacementPoints(new Location(12.656250f,71.929825f));
		checkedAddToAIPlacementPoints(new Location(28.828125f,76.140351f));
		targetCoords.add(new Location(35.859375f,75.438596f));
		checkedAddToAIPlacementPoints(new Location(38.671875f,74.736842f));
		checkedAddToAIPlacementPoints(new Location(48.515625f,75.438596f));
		targetCoords.add(new Location(49.218750f,-5.263158f));
		checkedAddToAIPlacementPoints(new Location(56.953125f,-10.175439f));
		checkedAddToAIPlacementPoints(new Location(58.359375f,1.754386f));
		targetCoords.add(new Location(64.687500f,8.771930f));
		targetCoords.add(new Location(70.312500f,-6.666667f));
		checkedAddToAIPlacementPoints(new Location(71.015625f,3.859649f));
		checkedAddToAIPlacementPoints(new Location(82.968750f,-3.859649f));
		targetCoords.add(new Location(90.000000f,-2.456140f));
		checkedAddToAIPlacementPoints(new Location(94.921875f,-12.280702f));
		targetCoords.add(new Location(132.890625f,21.403509f));
		targetCoords.add(new Location(134.296875f,10.175439f));
		checkedAddToAIPlacementPoints(new Location(144.843750f,27.017544f));
		checkedAddToAIPlacementPoints(new Location(149.062500f,15.087719f));
		targetCoords.add(new Location(150.468750f,31.228070f));
		checkedAddToAIPlacementPoints(new Location(154.687500f,34.035088f));
		checkedAddToAIPlacementPoints(new Location(157.500000f,21.403509f));
		targetCoords.add(new Location(158.906250f,41.052632f));
		checkedAddToAIPlacementPoints(new Location(163.828125f,46.666667f));
		checkedAddToAIPlacementPoints(new Location(165.234375f,8.771930f));
		checkedAddToAIPlacementPoints(new Location(167.343750f,32.631579f));
		checkedAddToAIPlacementPoints(new Location(172.265625f,20.701754f));
		checkedAddToAIPlacementPoints(new Location(172.968750f,45.964912f));
	}
	
	private final void checkedAddToAIPlacementPoints(Location location) {
		if (JBot.IsValidTerritory(
				JBot.GetOwnTeamId(),
				(float) location.getX(),
				(float) location.getY(),
				true) ||
				JBot.IsValidTerritory(
						JBot.GetOwnTeamId(),
						(float) location.getX(),
						(float) location.getY(),
						true))
			aiPlacementPoints.add(location);
	}

	/**
	 * Returns current set of AI placement points.
	 * @return LinkedList of Locations
	 */
	public final LinkedList<Location> getAIPlacementPoints() {
		return aiPlacementPoints;
	}

	/**
	 * Returns current set of AI target points.
	 * @return LinkedList of Locations
	 */	
	public final LinkedList<Location> getTargetCoords() {
		return targetCoords;
	}

	/**
	 * Current Defcon Stage, game starts with 5
	 * 
	 * @return int Defcon stage
	 */
	public int getDefconLevel() {
		return defcon.getFlag();
	}

	/**
	 * Current Game Time, measured in seconds. Each tick, the game progresses by
	 * 0.1 sec * GameSpeed
	 * 
	 * @return game time
	 */
	public float getGameTime() {
		return gameTime;
	}

	/**
	 * Amount of update cycles (ticks) passed since game start
	 * 
	 * @return number of game ticks
	 */
	public int getGameTick() {
		return gameTick;
	}

	/**
	 * Current speed-up factor of the game over the real time passed. Usually
	 * has values from 0 (paused), 1 (real time), 5, 10, 20, see enum GAMESPEED
	 * (GameSpeed class)
	 * 
	 * @return
	 */
	public GameSpeed getGameSpeed() {
		return gameSpeed.getFlag();
	}

	/**
	 * Time remaining in game, if victory timer was started. Test this with
	 * IsVictoryTimerStarted
	 * 
	 * @return
	 */
	public float getVictoryTimer() {
		return victoryTimer;
	}

	/**
	 * True iff the victory-timer has been started
	 * 
	 * @return
	 */
	public boolean isVictoryTimerActive() {
		return victoryTimerActive.getFlag();
	}

	/**
	 * Value of certain option.
	 * <p><p>
	 * Black_Hand: no idea what it does, might be related to the game settings
	 * 	(advanced options in game hosting) like CityPopulations. Expected
	 * to be called like: getOptionValue("CityPopulations").
	 * 
	 * @param name
	 * @return
	 */
	public int getOptionValue(String name) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetOptionValue(), new Object[] { name });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * True if the game is currently replayed (Timeline has been clicked).
	 * TODO: e.g. game has been paused?
	 * 
	 * @return
	 */
	public boolean debugIsReplayingGame() {
		return debugIsReplayingGame.getFlag();
	}

	/**
	 * Team Id of given unit.
	 * 
	 * @param id
	 * @return
	 */
	public int getTeamId(int id) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetTeamId(), new Object[] { id });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * Own fleet ids.
	 * 
	 * @return
	 */
	public int[] getOwnFleets() {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetOwnFleets(), null);
		return (int[]) container.syncCallInMainThread();
	}

	/**
	 * Fleet ids of given team. Only fleets ids with visible members are
	 * returned.
	 * 
	 * @param teamId
	 * @return
	 */
	public int[] getFleets(int teamId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetFleets(), new Object[] { teamId });
		return (int[]) container.syncCallInMainThread();
	}

	/**
	 * Ids of ships in given fleet.
	 * 
	 * @param fleetId
	 * @return
	 */
	public int[] getFleetMembers(int fleetId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetFleetMembers(),
				new Object[] { fleetId });
		return (int[]) container.syncCallInMainThread();
	}

	/**
	 * Id of fleet of given unit.
	 * 
	 * @param unitId
	 * @return
	 */
	public int getFleetId(int unitId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetFleetId(), new Object[] { unitId });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * Time until current state is active.
	 * 
	 * @param unitId
	 * @return
	 */
	public float getStateTimer(int unitId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetStateTimer(),
				new Object[] { unitId });
		return (Float) container.syncCallInMainThread();
	}

	/**
	 * Array of unitIds of currently queued actions, for example nukes in a silo
	 * or planes on a carrier
	 * 
	 * @param unitId
	 * @return
	 */
	public int[] getActionQueue(int unitId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetActionQueue(),
				new Object[] { unitId });
		return (int[]) container.syncCallInMainThread();
	}

	/**
	 * True iff given unit is automatically retaliating an earlier attack.
	 * 
	 * @param unitId
	 * @return
	 */
	public boolean isRetaliating(int unitId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.IsRetaliating(),
				new Object[] { unitId });
		return (Boolean) container.syncCallInMainThread();
	}

	/**
	 * True iff given unit is visible to given team. In full information mode,
	 * visibility information about other teams is available. In limited
	 * information mode, only visible units are accessible.
	 * 
	 * @param unitId
	 * @param byTeamId
	 * @return
	 */
	public boolean isVisible(int unitId, int byTeamId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.IsVisible(), new Object[] { unitId,
						byTeamId });
		return (Boolean) container.syncCallInMainThread();
	}

	/**
	 * Movement direction of given unit, in longitude and latitude parts. The
	 * vector has the length of the unit speed (see also SPEED_*).
	 * 
	 * @param unitId
	 * @return
	 */
	public Location getVelocity(int unitId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetVelocity(), new Object[] { unitId });
		return new Location((float[]) container.syncCallInMainThread());
	}

	/**
	 * Remaining range of unit. If unlimited, -1 is returned.
	 * 
	 * @param unitId
	 * @return
	 */
	public float getRange(int unitId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetRange(), new Object[] { unitId });
		return (Float) container.syncCallInMainThread();
	}

	/**
	 * Amount of remaining units of given type that can be placed.
	 * 
	 * @param typeId
	 * @return
	 */
	public int getRemainingUnits(UnitType typeId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetRemainingUnits(),
				new Object[] { typeId.id });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * True iff given location is valid for placement of given type. For fleets
	 * use getFleetMemberOffset to get offset from fleet center.
	 * 
	 * @param longitude
	 * @param latitude
	 * @param typeId
	 * @return
	 */
	public boolean isValidPlacementLocation(float longitude, float latitude,
			UnitType typeId) {

		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.IsValidPlacementLocation(),
				new Object[] { longitude, latitude, typeId.id });

		Object return_value = container.syncCallInMainThread();

		return (Boolean) return_value;
	}
	
	/**
	 * True iff given location is valid for placement of given type. For fleets
	 * use getFleetMemberOffset to get offset from fleet center.
	 * 
	 * @param longitude
	 * @param latitude
	 * @param typeId
	 * @return
	 */
	public boolean isValidPlacementLocation(double longitude, double latitude,
			UnitType typeId) {
		return isValidPlacementLocation(
				(float) longitude,
				(float) latitude,
				typeId);
	}

	/**
	 * True iff given location is valid for ship placement. For fleets use
	 * getFleetMemberOffset to get offset from fleet center.
	 * 
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	public boolean isValidShipPlacementLocation(Location location) {
		return isValidPlacementLocation(
				location.getX(),
				location.getY(),
				UnitType.BATTLE_SHIP);
	}

	/**
	 * True iff given location is valid for ship placement. For fleets use
	 * getFleetMemberOffset to get offset from fleet center.
	 * 
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	public boolean isValidShipPlacementLocation(double longitude,
			double latitude) {
		return isValidPlacementLocation(
				longitude,
				latitude,
				UnitType.BATTLE_SHIP);
	}

	/**
	 * True iff given location is valid for building placement. For fleets use
	 * getFleetMemberOffset to get offset from fleet center.
	 * 
	 * @param location
	 * @return
	 */
	public boolean isValidBuildingPlacementLocation(Location location) {
		return isValidBuildingPlacementLocation(
				location.getX(),
				location.getY());
	}


	/**
	 * True iff given location is valid for building placement. For fleets use
	 * getFleetMemberOffset to get offset from fleet center.
	 * 
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	public boolean isValidBuildingPlacementLocation(double longitude,
			double latitude) {
		return isValidPlacementLocation(
				longitude,
				latitude,
				UnitType.RADAR);
	}
	
	public boolean isValidFleetPlacementLocation(float longitude, float latitude,
			int fleetSize) {
		for( int i = 0; i < fleetSize; ++i )
		{
			Location offset = getFleetMemberOffset(fleetSize, i);

			// the placement validity is equal for battleships, carriers and
			// subs.
			if (!isValidShipPlacementLocation(longitude + offset.getX(),
					latitude + offset.getY()))
				return false;
		}
		return true;
	}

	/**
	 * Credits available for placement (if in variable unit mode).
	 * 
	 * @return
	 */
	public int getUnitCredits() {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetUnitCredits(), null);
		Object return_value = container.syncCallInMainThread();

		return (Integer) return_value;
	}

	/**
	 * Value of given unit type (if in variable unit mode).
	 * 
	 * @param typeId
	 * @return
	 */
	public int getUnitValue(UnitType typeId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetUnitValue(),
				new Object[] { typeId.id });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * Distance in game between given coordinates on sea (performs pathfinding).
	 * 
	 * @param locationA
	 * @param locationB
	 * @return distance
	 */
	public float getSailDistance(Location locationA, Location locationB) {

		if (locationA == null)
			throw new IllegalArgumentException(
					"GetSailDistance requires non-null arguments. [locationA]");
		if (locationB == null)
			throw new IllegalArgumentException(
					"GetSailDistance requires non-null arguments. [locationB]");

		return getSailDistance(locationA.getX(),
				locationA.getY(), locationB.getX(),
				locationB.getY());
	}

	/**
	 * Distance in game between given coordinates on sea (performs pathfinding).
	 * 
	 * @param locationA
	 * @param locationB
	 * @return distance
	 */
	public float getSailDistance(double locationA_x, double locationA_y,
			double locationB_x, double locationB_y) {
		return getSailDistance(((float) locationA_x),
				(float) locationA_y, (float) locationB_x,
				(float) locationB_y);
	}

	/**
	 * Distance in game between given coordinates on sea (performs pathfinding).
	 * 
	 * @param locationA_x
	 * @param locationB_x
	 * @return distance
	 */
	public float getSailDistance(float locationA_x, float locationA_y,
			float locationB_x, float locationB_y) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetSailDistance(),
				new Object[] { locationA_x, locationA_y, locationB_x,
						locationB_y });

		Object return_value = container.syncCallInMainThread();

		return (return_value instanceof Float) ? (Float) return_value : null;
	}


	private final static class LocationWithDistance {
				
		private double distance;
		private Location location;
		
		public LocationWithDistance(double distance, Location location) {
			this.distance = distance;
			this.location = location;
		}		
		
		public final double getDistance() {
			return distance;
		}
		
		public final void setDistance(double distance) {
			this.distance = distance;
		}
		
		public final Location getLocation() {
			return location;
		}
		
		public final void setLocation(Location location) {
			this.location = location;
		}
	}
	
	private final float MIN_STEP = 5f;
	
	/**
	 * Finds a closest point on given border to a given point. VERY ROUGH!!!
	 * That is because it uses getSailDistance and NOT the euclidean distance.
	 * 
	 * @param location
	 * @param border
	 * @return closest point
	 */
	public LocationWithDistance getClosestSailTarget(Location location,
			List<Location> border) {
		if (border == null || border.size() == 0)
			return null;

		if (border.size() == 1)
			return new LocationWithDistance(getSailDistance(location, border.get(0)), border.get(0));

		Iterator<Location> front = border.iterator();
		Location last = null;
		Location current = front.next();
		Location best = null;
		Location tmp = null;
		Location tmp2 = null;
		Location tmp3 = null;
		double tmp_distance = 0, tmp_distance2 = 0, tmp_distance3 = 0;
		double min_distance = Double.MAX_VALUE;

		while (front.hasNext()) {
			last = current;
			current = front.next();

			tmp = last;
			tmp2 = current;

			while (tmp.getDistance2D(tmp2) > MIN_STEP) {
				tmp3 = tmp2.sub(tmp).scale(0.5f).add(tmp);

				tmp_distance = getSailDistance(location, tmp);
				tmp_distance2 = getSailDistance(location, tmp2);
				tmp_distance3 = getSailDistance(location, tmp3);

				if (tmp_distance2 < tmp_distance) {
					tmp = tmp2;
					tmp_distance = tmp_distance2;
				}

				if (tmp_distance3 > tmp_distance) {
					break;
				}

				tmp2 = tmp3;
				tmp_distance2 = tmp_distance3;
			}

			if (tmp_distance2 < tmp_distance) {
				tmp_distance = tmp_distance2;
				tmp = tmp2;
			}

			if (tmp_distance < min_distance) {
				min_distance = tmp_distance;
				best = tmp;
			}
		}

		return new LocationWithDistance(min_distance, best);
	}

	/**
	 * Finds a closest point on given border to a given point. VERY ROUGH!!!
	 * That is because it uses getSailDistance and NOT the euclidean distance.
	 * 
	 * @param borderA
	 * @param borderB
	 * @return two closest points
	 */
	public LocationPair getClosestSailDistance(List<Location> borderA,
			List<Location> borderB) {

		double min_distance = Double.MAX_VALUE;
		LocationPair best = null;

		if (borderA == null || borderA.size() == 0)
			return null;

		Iterator<Location> front = borderA.iterator();
		Location current = front.next();
		Location last = null;
		LocationPair borderA_borderB = new LocationPair(current, borderB.get(0));
		LocationPair borderA_borderB_2 = new LocationPair(current,
				borderB.get(0));
		LocationPair borderA_borderB_candidate = null;
		double tmp_distance = 0, tmp_distance2 = 0, tmp_distance3 = 0;

		while (front.hasNext()) {
			last = current;
			current = front.next();						
			
			borderA_borderB.A = last;
			borderA_borderB_2.A = current;
			
			LocationWithDistance loc_with_dist;
			
			loc_with_dist = getClosestSailTarget(borderA_borderB.A, borderB);;
			tmp_distance = loc_with_dist.getDistance();
			borderA_borderB.B = loc_with_dist.getLocation();
			
			
			loc_with_dist = getClosestSailTarget(borderA_borderB_2.A, borderB);			
			tmp_distance2 = loc_with_dist.getDistance();
			borderA_borderB_2.B = loc_with_dist.getLocation();

			while (borderA_borderB.A.getDistance2D(borderA_borderB_2.A) > MIN_STEP) {
				// tmp2.A.sub(tmp.A).scale(0.5f).add(tmp.A) === half way between tmp and tmp2
				Location half_way = borderA_borderB_2.A.sub(borderA_borderB.A).scale(0.5f).add(borderA_borderB.A);
				loc_with_dist = getClosestSailTarget(half_way, borderB);
				borderA_borderB_candidate = new LocationPair(half_way,
						loc_with_dist.getLocation());

				tmp_distance3 = loc_with_dist.getDistance();

				if (tmp_distance2 < tmp_distance) {
					borderA_borderB = borderA_borderB_2;
					tmp_distance = tmp_distance2;
				}

				if (tmp_distance3 > tmp_distance) {
					break;
				}

				borderA_borderB_2 = borderA_borderB_candidate;
				tmp_distance2 = tmp_distance3;
			}

			if (tmp_distance2 < tmp_distance) {
				tmp_distance = tmp_distance2;
				borderA_borderB = borderA_borderB_2;
			}

			if (tmp_distance < min_distance) {
				min_distance = tmp_distance;
				best = borderA_borderB;
			}
		}
		System.gc();
		return best;
	}

	public class LocationPair {
		public Location A;
		public Location B;

		public LocationPair(Location A, Location B) {
			this.A = A;
			this.B = B;
		}
	}

	/**
	 * True if the given coordinates belong to the given Team. If seaArea is set
	 * to true, then Coordinates must be on sea area, otherwise land. If teamId
	 * = -1, then function returns if coordinates are land or sea terrain
	 * respectively. Note that there can be coordinates which are neither land
	 * nor sea
	 * 
	 * @param teamId
	 * @param longitude
	 * @param latitude
	 * @param seaArea
	 * @return
	 */
	public boolean isValidTerritory(int teamId, float longitude,
			float latitude, boolean seaArea) {

		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.IsValidTerritory(),
				new Object[] { teamId, longitude, latitude, seaArea });
		return (Boolean) container.syncCallInMainThread();
	}

	/**
	 * True if the given coordinates belong to the given Team. If seaArea is set
	 * to true, then Coordinates must be on sea area, otherwise land. If teamId
	 * = -1, then function returns if coordinates are land or sea terrain
	 * respectively. Note that there can be coordinates which are neither land
	 * nor sea
	 * 
	 * @param teamId
	 * @param longitude
	 * @param latitude
	 * @param seaArea
	 * @return
	 */
	public boolean isValidTerritory(int teamId, double longitude,
			double latitude, boolean seaArea) {
		return isValidTerritory(teamId, (float) longitude, (float) latitude,
				seaArea);
	}

	/**
	 * True if the given coordinates belong to the given Team. If seaArea is set
	 * to true, then Coordinates must be on sea area, otherwise land. If teamId
	 * = -1, then function returns if coordinates are land or sea terrain
	 * respectively. Note that there can be coordinates which are neither land
	 * nor sea
	 * 
	 * @param teamId
	 * @param location
	 * @param seaArea
	 * @return
	 */
	public boolean isValidTerritory(int teamId, Location location,
			boolean seaArea) {
		return isValidTerritory(teamId, (float) location.getX(),
				(float) location.getY(), seaArea);
	}

	/**
	 * True if given coordinates are on the border. Compare
	 * "data/earth/coastlines.bmp".
	 * 
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	public boolean isBorder(float longitude, float latitude) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.IsBorder(),
				new Object[] { longitude, latitude });
		return (Boolean) container.syncCallInMainThread();
	}

	/**
	 * Territory Id of territory at given coordinates.
	 * 
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	public int getTerritoryId(float longitude, float latitude) {

		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetTerritoryId(),
				new Object[] { longitude, latitude });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * Territory Id of territory at given coordinates.
	 * 
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	public int getTerritoryId(double longitude, double latitude) {
		return getTerritoryId((float) longitude, (float) latitude);
	}

	/**
	 * Own team id.
	 * 
	 * @return
	 */
	public int getOwnTeamId() {
		return ownTeamId;
	}

	/**
	 * List of Team Ids in the game.
	 * 
	 * @return
	 */
	public int[] getTeamIds() {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetTeamIds(),
				null);
		return (int[]) container.syncCallInMainThread();
	}

	/**
	 * List of Alliances Team Ids in the game.
	 * 
	 * @return List of enemy alliances and containing enemies WARNING: this
	 *         SortedMap changes when someone leaves/enters alliance
	 */
	public SortedMap<Integer, LinkedList<Integer>> getAlliancesWithTeamIds() {
		int[] teams = getTeamIds();

		SortedMap<Integer, LinkedList<Integer>> alliances = new TreeMap<Integer, LinkedList<Integer>>();
		for (int i : teams) {
			int alliance = getAllianceId(i);
			LinkedList<Integer> list = alliances.get(alliance);
			if (list == null)
				list = new LinkedList<Integer>();

			list.add(i);
			alliances.put(alliance, list);
		}
		return alliances;
	}

	/**
	 * List of Enemy Team Ids in the game.
	 * 
	 * @return List of enemy alliances and containing enemies WARNING: this
	 *         SortedMap changes when someone leaves/enters alliance
	 */
	public int[] getEnemyTeamIds() {
		return enemyIds;
	}

	/**
	 * List of Enemies city Ids in the game.
	 * 
	 * @return List of enemy alliances and containing enemies WARNING: this
	 *         SortedMap changes when someone leaves/enters alliance
	 */
	public SortedMap<Integer, List<Integer>> getEnemiesCityIds() {
		return Collections.unmodifiableSortedMap(enemiesCityIds);
	}

	/**
	 * List of Enemies city Ids in the game.
	 * 
	 * @return List of enemy alliances and containing enemies WARNING: this
	 *         SortedMap changes when someone leaves/enters alliance
	 */
	public List<Integer> getEnemyCityIds() {
		ArrayList<Integer> enemyCityIds = new ArrayList<Integer>(20);

		for (List<Integer> ids : enemiesCityIds.values()) {
			for (int id : ids) {
				enemyCityIds.add(id);
			}
		}
		return enemyCityIds;
	}

	/**
	 * List of Enemy city Ids in the game.
	 * 
	 * @return List of enemy alliances and containing enemies WARNING: this
	 *         SortedMap changes when someone leaves/enters alliance
	 */
	public List<Integer> getEnemyCityIds(int enemyId) {
		return Collections.unmodifiableList(enemiesCityIds.get(enemyId));
	}

	/**
	 * Number of territories for given team, usually 1.
	 * 
	 * @return
	 */
	public int getTeamTerritoriesCount(int teamId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetTeamTerritoriesCount(),
				new Object[] { teamId });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * 
	 * Territory Ids of territories that the given team owns. The enum
	 * TERRITORY_* relates the ids to starting positions
	 * 
	 * @param teamId
	 * @return territories belonging to the team
	 */
	public int[] getTeamTerritories(int teamId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetTeamTerritories(),
				new Object[] { teamId });
		return (int[]) container.syncCallInMainThread();
	}

	/**
	 * 
	 * Territory Ids of territories that the team the player is in owns. The
	 * enum TERRITORY_* relates the ids to starting positions
	 * 
	 * @return territories belonging to the team
	 */
	public int[] getOwnTeamTerritories() {
		return getTeamTerritories(getOwnTeamId());
	}

	/**
	 * Id of alliance. Each team belongs to exactly one alliance.
	 * 
	 * @param teamId
	 * @return
	 */
	public int getAllianceId(int teamId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetAllianceId(),
				new Object[] { teamId });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * Currently requested game speed of given team.
	 * 
	 * @param teamId
	 * @return
	 */
	public int getDesiredGameSpeed(int teamId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetDesiredGameSpeed(),
				new Object[] { teamId });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * Sum of enemy kills of the given team (for scoring).
	 * 
	 * @param teamId
	 * @return
	 */
	public int getEnemyKills(int teamId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetEnemyKills(),
				new Object[] { teamId });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * Sum of friendly deaths (deaths in allied populations) of the given team.
	 * 
	 * @param teamId
	 * @return
	 */
	public int getFriendlyDeaths(int teamId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetFriendlyDeaths(),
				new Object[] { teamId });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * Sum of collateral damage deaths (deaths in own population) of the given
	 * team.
	 * 
	 * @param teamId
	 * @return
	 */
	public int getCollateralDamage(int teamId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetCollateralDamage(),
				new Object[] { teamId });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * Name of the given team.
	 * 
	 * @param teamId
	 * @return
	 */
	public String getTeamName(int teamId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetTeamName(),
				new Object[] { teamId });
		return (String) container.syncCallInMainThread();
	}

	/**
	 * True iff the first team is sharing its radar with the second team.
	 * 
	 * @param teamId1
	 * @param teamId2
	 * @return
	 */
	public boolean isSharingRadar(int teamId1, int teamId2) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.IsSharingRadar(),
				new Object[] { teamId1, teamId2 });
		return (Boolean) container.syncCallInMainThread();
	}

	/**
	 * True iff the first team is in cease fire mode with the second team.
	 * 
	 * @param teamId1
	 * @param teamId2
	 * @return
	 */
	public boolean isCeaseFire(int teamId1, int teamId2) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.IsCeaseFire(),
				new Object[] { teamId1, teamId2 });
		return (Boolean) container.syncCallInMainThread();
	}

	/**
	 * Offset of ship number memberId from center of fleet, given fleet has
	 * memberCount ships
	 * 
	 * @param memberCount
	 * @param memberId
	 * @return
	 */
	public Location getFleetMemberOffset(int memberCount, int memberId) {
		return fleetMemberOffsets[memberCount - 1][memberId];
	}

	/**
	 * Used in prepopulation of fleet member offsets. DO NOT USE FROM OTHER
	 * THREADS THAN MAIN, SINCE IT IS NOT ENTIRELY STABLE!!!
	 * 
	 * @param memberCount
	 * @param memberId
	 * @return
	 */
	private Location getFleetMemberOffsetWorker(int memberCount, int memberId) {
		if (memberCount <= 0 || memberId >= memberCount)
			throw new InvalidParameterException(
					"GetFleetMemberOffset requires memberCount > 0 && "
							+ "memberId < membercount. memberCount: "
							+ memberCount + " memberId: " + memberId);

		return new Location((float[]) JBot.GetFleetMemberOffset(
				memberCount,
				memberId));
	}

	/**
	 * Queries gameRunning flag.
	 * @return true if game is running.
	 */
	public boolean getRunning() {
		return gameRunning.getFlag();
	}

	/**
	 * Simplifies queries for possibility of placing a fleet.
	 * Inspired by native bots implemented for Defcon API.
	 * @param placement location, where to check the placement validity.
	 * @param fleetSize number of units in the fleet.
	 * @return true if this placement is valid.
	 */
	public boolean isValidFleetPlacement(Location placement, int fleetSize) {
		for (int i = 0; i < fleetSize; ++i) {
			Location offset = getFleetMemberOffset(fleetSize, i);

			// the placement validity is equal for battleships, carriers and
			// subs.
			if (!isValidPlacementLocation(
					placement.getX() + offset.getX(),
					placement.getY() + offset.getY(),
					UnitType.BATTLE_SHIP)) {
				return false;
			}
		}
		return true;
	}


	/**
	 * Used in prepopulation of fleetSizes. DO NOT USE FROM OTHER THREADS THAN
	 * MAIN, SINCE IT IS NOT ENTIRELY STABLE!!!
	 * 
	 * @param fleetSize
	 * @return
	 */
	private double getFleetDiameterWorker(int fleetSize) {

		double max = 0;
		for (int i = 0; i < fleetSize; ++i) {
			Location offset = getFleetMemberOffset(fleetSize, i);
			double length = offset.getLength();
			if (max < length)
				max = length;
		}

		return max + 1d;
	}

	/**
	 * Gets the smallest circle diameter, which could contain the whole fleet.
	 * 
	 * @param fleetSize
	 * @return diameter of the fleet of given size
	 */
	public double getFleetDiameter(int fleetSize) {
		return fleetDiameters[fleetSize - 1];
	}

	/**
	 * Returns a flat list of territories possessed by enemies.
	 * 
	 * @return flat list of all enemy territories
	 */
	public LinkedList<Integer> getAllEnemyTerritories() {

		LinkedList<Integer> territoryIds = new LinkedList<Integer>();

		for (int enemyId : getEnemyTeamIds()) {
			for (int territoryId : getTeamTerritories(enemyId)) {
				territoryIds.add(territoryId);
			}
		}

		return territoryIds;
	}

	/**
	 * Checks validity of placement for structures.
	 * @param placement location, where to check the placement validity.
	 * @return true if this placement is valid.
	 */
	public boolean isValidStructureLocation(Location placement) {
		return isValidPlacementLocation(placement.getX(),
				placement.getY(), UnitType.AIR_BASE);
	}

	/**
	 * Returns amount of nukes present in the given unit.
	 * 
	 * @param unitId
	 * @return amount of nukes
	 */
	public int getNukeSupply(int unitId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetNukeSupply(),
				new Object[] { unitId });
		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * Returns type of unit with this id.
	 * 
	 * @param unitId
	 *            id of unit
	 * @return
	 */
	public UnitType getType(int unitId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetType(),
				new Object[] { unitId });
		return UnitType.getEnum((Integer) container.syncCallInMainThread());
	}

	/**
	 * Returns number of activations of this state for this unit.
	 * 
	 * @param unitId
	 *            id of unit
	 * @param state
	 *            state for the unit
	 * @return
	 */
	public int getStateCount(int unitId, IState state) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetStateCount(),
				new Object[] { unitId, state.getStateId() });

		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * Returns number of activations of this state for this unit.
	 * 
	 * @param unitId
	 *            id of unit
	 * @param stateId
	 *            stateId for the unit
	 * @return
	 */
	public int getStateCount(int unitId, int stateId) {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetStateCount(),
				new Object[] { unitId, stateId });

		return (Integer) container.syncCallInMainThread();
	}

	/**
	 * Returns the territory to owner mapping array.
	 * 
	 * @return
	 */
	public int[] getTerritoryOwners() {
		return territoryOwners;
	}

	/**
	 * Returns the owner of a land territory, which contains given location or
	 * -1 if noone owns it.
	 * 
	 * @param center
	 * @return
	 */
	public int getTerritoryOwner(Location center) {
		return getTerritoryOwner(center.getX(), center.getY());
	}

	/**
	 * Returns the owner of a land territory, which contains given location or
	 * -1 if noone owns it.
	 * 
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	public int getTerritoryOwner(double longitude, double latitude) {
		return getTerritoryOwner(getTerritoryId(longitude, latitude));
	}

	/**
	 * Returns the owner of a territory with a given Id.
	 * 
	 * @param territoryId
	 * @return
	 */
	public int getTerritoryOwner(int territoryId) {

		if (territoryId == -1)
			return -1;

		return territoryOwners[territoryId];
	}

	/**
	 * Get id of a land territory at a given location.
	 * 
	 * @param location
	 * @return
	 */
	public int getTerritoryId(Location location) {
		return getTerritoryId(location.getX(), location.getY());
	}

	/**
	 * Returns the number of territories.
	 * 
	 * @return
	 */
	public int getTerritoriesCount() {
		return JBot.NumTerritories;
	}

	/**
	 * Gets a list of cities.
	 * 
	 * @return
	 */
	public int[] getCityIds() {
		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetCityIds(),
				null);
		return (int[]) container.syncCallInMainThread();
	}

	/**
	 * Returns a location of the given object.
	 * 
	 * @param objectId
	 * @return
	 */
	public Location getObjectLocation(int objectId) {

		SyncMethodExecContainer container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetLongitude(),
				new Object[] { objectId });
		float longitude = (Float) container.syncCallInMainThread();

		container = new SyncMethodExecContainer(
				JBotMethodsRepository.GetLatitude(),
				new Object[] { objectId });
		float latitude = (Float) container.syncCallInMainThread();

		return new Location(longitude, latitude);
	}

	/**
	 * Checks whether are there enough units of each type to create a fleet of a
	 * given composition.
	 * 
	 * @param ships
	 * @return
	 */
	public boolean canCreateFleet(UnitType[] ships) {

		final Hashtable<Integer, Integer> table = new Hashtable<Integer, Integer>();

		for (UnitType type : ships) {

			if (table.containsKey(type.id)) {
				int newCount = table.get(type.id) + 1;

				int remaining = getRemainingUnits(type);

				if (newCount > remaining) {
					return false;
				}

				table.put(type.id, newCount);
			} else {
				table.put(type.id, 1);
			}
		}
		table.clear();

		return true;
	}

	/**
	 * Returns range of submarine nukes (SRBM).
	 * 
	 * @return
	 */
	public double getSubNukeRange() {
		return 45d;
	}

	/**
	 * Returns range of a bomber.
	 * 
	 * @return
	 */
	public double getBomberRange() {
		return 180d;
	}

	/**
	 * Returns range of a single radar.
	 * 
	 * @return
	 */
	public double getRadarRange() {
		return 30d;
	}

	/**
	 * Returns range of a fighter.
	 * 
	 * @return
	 */
	public double getFighterRange() {
		return 40;
	}

	/**
	 * Returns range of AA mode of a silo.
	 * 
	 * @return
	 */
	public double getSiloRange() {
		return 30d;
	}

	/**
	 * Returns center of gravity of given enemy's cities.
	 * 
	 * @param teamId
	 * @return
	 */
	public DefConLocation getCitiesGravityCenter(int teamId) {
		
		Location sum = new Location();

		List<Integer> cityList;
		if (teamId == getOwnTeamId())
			cityList = ownCityIds;
		else
			cityList = enemiesCityIds.get(teamId);

		if (cityList == null) {
			PogamutJBotSupport.writeToConsole("cityList is empty: " + teamId
					+ " " + enemiesCityIds);
		}

		for (int cityId : cityList) {
			DefConLocation location = getCityLocation(cityId);
			sum.x += location.getX();
			sum.y += location.getY();
		}

		return new DefConLocation(sum.scale(1d / cityList.size()));
	}

	/**
	 * Returns location of the given city.
	 * 
	 * @param cityId
	 * @return
	 */
	public DefConLocation getCityLocation(int cityId) {
		return new DefConLocation(cityLocations.get(cityId));
	}

}
