package decisionMakingSystem;

import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectListener;
import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.BeginMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Item;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ItemPickedUp;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import utils.HashMapWrapper;
import utils.MyHashMapEntry;
import bot.Bot;
import cz.cuni.amis.pogamut.unreal.communication.messages.UnrealId;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.logging.Logger;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import utils.TimeUtils;

/**
 * class which defines a tuple of integers
 * @author Ondrej
 */
class Tuple {

    public int a = 0;
    public int b = 0;

    Tuple() {
    }

    Tuple(int a, int b) {
        this.a = a;
        this.b = b;
    }

    @Override
    public String toString() {
        return "a: " + a + " b: " + b;
    }
}

/**
 * register listeners for items, players etc. handles them and add to them affordances etc. 
 * also manages list of all items and operations over it - most important action for other 
 * parts of DMS is seeingItems, which add to perceptive field items of an affordance agent 
 * is looking for
 * 
 * 
 * @author Ondrej
 */
public class ThingsManager {

    /** agent - for listeners */
    private Bot agent = null;
    /** log to have a place to log in*/
    private Logger log = null;
    /** visible items - by unique integer ID */
    public final HashMap<Long, EItem> visibleItems;
    public HashMap<UnrealId, Long> idTable;
    /** hash map which contains parameters of each item class - like that some particular Unreal class has affordance TO_EAT etc. */
    private HashMap<String, EItem> itemParameters = null;
    /** link on perceptive field */
    private PerceptiveField percField = null;
    private DecisionModuleImpl decisionModule = null;
    /**
     * used to invoke spontaneous action
     */
    private ArrayList<Action> possibleActions = null;
    /**
     * says if it is night or not = as bot starts in midnight, it is set to true
     */
    private boolean night = true;
    private boolean ready = false;

    /**
     */
    private class PlayerListener implements IWorldObjectListener<Player> {
        /**
         * Constructor. Registers itself on the given WorldView object.
         * @param worldView WorldView object to listent to.
         */
        public PlayerListener(IWorldView worldView) {
            worldView.addObjectListener(Player.class, WorldObjectUpdatedEvent.class, this);
        }

        //public void notify(WorldObjectUpdatedEvent<Player> event) {
        public void notify(IWorldObjectEvent<Player> event) {
            EItem newItem = processItem(event.getObject());
            if (newItem != null) {
                Long longId = idTable.get(event.getObject().getId());
                if (longId == null) {
                  Long newId = event.getObject().getId().getLongId();
                  idTable.put(event.getObject().getId(), newId);
                  longId = newId;
                }
                if (event.getObject().isVisible()) {
                    synchronized(visibleItems){
                        
                        visibleItems.put(longId, newItem);
                    }
                } else {
                    removeFromPerception(longId,newItem);
                }
            }
        }

       
    }
    /** Player listener */
    PlayerListener playerListener;

    /**
     */
    private class NavPointListener implements IWorldObjectListener<NavPoint> {

        /**
         * Constructor. Registers itself on the given WorldView object.
         * @param worldView WorldView object to listent to.
         */
        public NavPointListener(IWorldView worldView) {
            worldView.addObjectListener(NavPoint.class, WorldObjectUpdatedEvent.class, this);
        }

        //public void notify(WorldObjectUpdatedEvent<NavPoint> event) {
        public void notify(IWorldObjectEvent<NavPoint> event) {
            EItem newItem = processItem(event.getObject());
            if (newItem != null) {
                Long longId = idTable.get(event.getObject().getId());
                if (longId == null) {
                  Long newId = event.getObject().getId().getLongId();
                  idTable.put(event.getObject().getId(), newId);
                  longId = newId;
                }
                if (event.getObject().isVisible()) {
                    synchronized(visibleItems){
                        visibleItems.put(longId, newItem);
                    }
                } else {
                    removeFromPerception(longId,newItem);
                }
            }
        }

    }
    /** NavPoint listener */
    NavPointListener navPointListener;

    /**
     */
    private class ItemListener implements IWorldObjectListener<Item> {

        /**
         * Constructor. Registers itself on the given WorldView object.
         * @param worldView WorldView object to listent to.
         */
        public ItemListener(IWorldView worldView) {
            worldView.addObjectListener(Item.class, WorldObjectUpdatedEvent.class, this);
        }

   //     public void notify(WorldObjectUpdatedEvent<Item> event) {
        public void notify(IWorldObjectEvent<Item> event) {
            EItem newItem = processItem(event.getObject());

            if (newItem != null) {

                Long longId = idTable.get(event.getObject().getId());
                if (longId == null) {
                  Long newId = event.getObject().getId().getLongId();
                  idTable.put(event.getObject().getId(), newId);
                  longId = newId;
                }
                if (event.getObject().isVisible()) {
                    synchronized(visibleItems){
                        visibleItems.put(longId, newItem);
                    }
                } else {
                    removeFromPerception(longId,newItem);
                }
            }
        }

    }
    /** NavPoint listener */
    ItemListener itemListener;

    /**
     */
    private class BegListener implements IWorldEventListener<BeginMessage> {

        /**
         * Constructor. Registers itself on the given WorldView object.
         * @param worldView WorldView object to listent to.
         */
        public BegListener(IWorldView worldView) {
            worldView.addEventListener(BeginMessage.class, this);
        }

        public void notify(BeginMessage event) {
            //throw new UnsupportedOperationException("Not supported yet.");
        }
    }
    /** ItemPickedUp listener */
    BegListener begListener;

    public ThingsManager(Bot agent, Logger log, PerceptiveField field, String directory, DecisionModuleImpl decisionModule) {
        this.agent = agent;
        this.log = log;
        percField = field;
        visibleItems = new HashMap<Long, EItem>();
        idTable = new HashMap<UnrealId, Long>();

        playerListener = new PlayerListener(agent.getWorldView());
        navPointListener = new NavPointListener(agent.getWorldView());
        itemListener = new ItemListener(agent.getWorldView());
        //begListener = new BegListener(agent.getWorldView()); //here we will delete items, that are not visible anymore

        itemParameters = loadItemParameters(directory + agent.mapMarkings);
        log.config("Known types of items: " + itemParameters.keySet());
        this.decisionModule = decisionModule;
        possibleActions = new ArrayList<Action>();
        // initialize possible actions -> only those, who has some atomic actions -> as it should execute something
        for (Intention intention : decisionModule.allIntentions) {
            possibleActions.addAll(BasicIntentionLoader.getActionsWithAtomicActions(intention));
        }
        // and I don't need hard copies, as I will be copying them while adding
        int index = 0;
        // as I need only actions which has some affordances to satisfy, I prune the list a bit
        for (; index < possibleActions.size();) {
            if (possibleActions.get(index).satisfyingItems != null && possibleActions.get(index).satisfyingItems.isEmpty()) {
                possibleActions.remove(index);
            } else {
                index++;
            }
        }
    }

    public boolean isNight() {
        return night;
    }

    private void removeFromPerception(long id, EItem newItem) {

        synchronized(visibleItems){
            if (!visibleItems.containsKey(id)) {
                return;
            }
            this.visibleItems.remove(id);
        }
        //this.percField.removeSource(newItem);
    }


    public void everyRoundUpdate(int counter) {
    }

    /**
     * process Item to EItem
     * @param item
     * @return
     */
    public EItem processItem(Item item) {
        EItem eItem = null;
        //ItemType.ADRENALINE_PACK;
        if (this.itemParameters.containsKey(item.getType().getName())) {
            eItem = (EItem) itemParameters.get(item.getType().getName()); // clone?
            eItem.item = item;
            eItem.setLocation(item.getLocation());
            Long longId = idTable.get(item.getId());
            if (longId == null) {
                Long newId = (long)idTable.size() + 1;
                idTable.put(item.getId(), newId);
                longId = newId;
            }
            eItem.setId(longId);
            eItem.type = MessageType.ITEM;
            eItem.cathegory = ItemCathegory.PICKABLE;
        } else {
            log.severe("Unknown item! class: " + item.getType().getName());
        }
        return eItem;
    }

    /**
     * process AddItem to EItem - needed by inventory for instance
     * @param item
     * @return
     */
    public EItem processItem(ItemPickedUp item) {
        EItem eItem = null;
        String itemClass = item.getType().getName();// + "Pickup"; // pick-uped items differs just by that from normal items
        if (this.itemParameters.containsKey(itemClass)) {
            eItem = (EItem) itemParameters.get(itemClass);
            eItem.item = item;
            eItem.setLocation(item.getLocation());
            Long longId = idTable.get(item.getId());
            if (longId == null) {
                Long newId = (long)idTable.size() + 1;
                idTable.put(item.getId(), newId);
                longId = newId;
            }
            eItem.setId(longId);
            eItem.type = MessageType.ADD_ITEM;
            eItem.cathegory = ItemCathegory.INVENTORY;
        } else {
            log.warning("Unknown item! class: " + itemClass);
        }
        return eItem;
    }

    /**
     * process NavPoint to EItem
     * @param item
     * @return
     */
    public EItem processItem(NavPoint item) {
        EItem eItem = null;
        if (this.itemParameters.containsKey(item.getId().getStringId())) {
            eItem = (EItem) itemParameters.get(item.getId().getStringId());
            eItem.item = item;
            eItem.setLocation(item.getLocation());
            Long longId = idTable.get(item.getId());
            
            if (longId == null) {
                Long newId = (long)idTable.size() + 1;
                idTable.put(item.getId(), newId);
                longId = newId;
            }
            eItem.setId(longId);
            eItem.type = MessageType.NAV_POINT;
            eItem.cathegory = ItemCathegory.PLACE;
        }
        return eItem;
    }

    /**
     * process player to EItem
     * @param item
     * @return
     */
    public EItem processItem(Player item) {
        EItem eItem = null;
        if (item.getId().getStringId().contains("RemoteBot19")) {// remote bot - prey
            if (this.itemParameters.containsKey("Prey")) {
                eItem = (EItem) itemParameters.get("Prey");
                eItem.item = item;
                eItem.setLocation(item.getLocation());
                Long longId = idTable.get(item.getId());
                if (longId == null) {
                    Long newId = (long)idTable.size() + 1;
                    idTable.put(item.getId(), newId);
                    longId = newId;
                }
                eItem.setId(longId);
                eItem.type = MessageType.PLAYER;
                eItem.cathegory = ItemCathegory.PLAYER;
            }
        } else {
            if (this.itemParameters.containsKey("Player")) { // normal player - spectator
                eItem = (EItem) itemParameters.get("Player");
                eItem.item = item;
                eItem.setLocation(item.getLocation());
                Long longId = idTable.get(item.getId());
                if (longId == null) {
                    Long newId = (long)idTable.size() + 1;
                    idTable.put(item.getId(), newId);
                    longId = newId;
                }

                eItem.setId(longId);
                eItem.type = MessageType.PLAYER;
                eItem.cathegory = ItemCathegory.PLAYER;
            } else {
                log.warning("Unknown item! class: Player");
            }
        }
        return eItem;
    }

    /**
     * loads parameters from the file -> itemParameters.xml
     * @param fileName
     * @return
     */
    private static HashMap<String, EItem> loadItemParameters(String fileName) {
        HashMapWrapper wrapper = null;
        File file = new File(fileName);
        try {
            FileInputStream reader = new FileInputStream(file);
            JAXBContext context = JAXBContext.newInstance(HashMapWrapper.class);
            Unmarshaller u = context.createUnmarshaller();
            wrapper = (HashMapWrapper) u.unmarshal(reader);

        } catch (Exception e) {
            System.err.println("Error in loading things parameters! " + e);
        }
        HashMap<String, EItem> result = new HashMap<String, EItem>();

        for (MyHashMapEntry entry : wrapper.entry) {
            result.put(entry.key, entry.value);
        }
        return result;

    }

    /**
     * saves example of itemParameters to xml file to have a sample of itemParameters.xml
     */
    public static void makeFirstEntry(String fileName) {
        HashMap hashMap = new HashMap<String, EItem>();
        EItem item = new EItem();
        ArrayList<Affordance> affos = new ArrayList<Affordance>();
        affos.add(new Affordance(AffordanceType.TO_EAT));
        affos.add(new Affordance(AffordanceType.TO_WASH));
        item.setAffordances(affos);
        item.setFadeout(10);
        item.setBasicAttractivity(15);
        item.name = "Apple";
        item.classOfItem = "XWeapons.RocketLauncherPickup";
        EItem item2 = new EItem();
        affos.clear();
        affos.add(new Affordance(AffordanceType.TO_CLEAN));
        item2.setAffordances(affos);
        item2.setFadeout(10);
        item2.setBasicAttractivity(25);
        item2.name = "Hunting clothes";
        item2.classOfItem = "XWeapons.SniperRiflePickup";
        hashMap.put(item.classOfItem, item);
        hashMap.put(item2.classOfItem, item2);

        HashMapWrapper wrapper = new HashMapWrapper(hashMap);
        try {
            File file = new File(fileName);
            FileOutputStream outputFile = new FileOutputStream(file);
            JAXBContext context = JAXBContext.newInstance(HashMapWrapper.class);
            Marshaller m = context.createMarshaller();
            m.marshal(wrapper, outputFile);
        } catch (Exception e) {
            System.err.println("Error in creating things Entry! " + e);
        }
    }

    /**
     * just for testing - creates file with item parameters and verifies, that it is correct
     * @param args
     */
    public static void main(String[] args) {
        HashMap<String, EItem> items = null;
        makeFirstEntry("itemsParametersTest.xml");
        items = loadItemParameters("itemParametersTest.xml");
        System.out.println(items);
    }

    /**
     * it counts the probability, item will be spoted by agent
     * - probability is based on stress as well
     * - about a time to get some idea how this work:)
     * 
     * @param newItem
     * @return
     */
    public int probability(EItem newItem) {
        String logReport = "Probability count for: " + newItem.name + "\n";
        int probability = 0, s1 = 0, s2 = 0, p1, p2, attr1, attr2, coefficient;
        boolean print = false; // now logs are muted
        synchronized (visibleItems) {
            for (EItem item : this.visibleItems.values()) {
                coefficient = 1;
                for (Action action : this.percField.processArea) {
                    if (this.decisionModule.actualAction != null && this.decisionModule.actualAction.name.contains("_Search") && action.equals(this.decisionModule.actualAction)) {
                        for (Affordance aff : item.getAffordances()) {
                            if (this.decisionModule.actualAction.satisfyingItems.keySet().contains(aff.type)) {
                                coefficient += 4; // so far I do it static, then we can think about some improvement
                            }
                        }
                    } else {
                        for (Affordance aff : item.getAffordances()) {
                            if (action.enabled && action.satisfyingItems != null &&
                                    action.satisfyingItems.keySet().contains(aff.type)) {
                                coefficient += 1; // if item is important for another preactive action
                            }
                        }
                    }
                }
                s1 += item.getBasicAttractivity() - item.decreaseOfAttractivity;
                s2 += (item.getAttractivity() - item.decreaseOfAttractivity) * coefficient;
                logReport += " item:" + item.name + " s1: " + s1 + " s2: " + s2 + " \n";
            }
        }
        logReport += "Sum of S1, S2. S1: " + s1 + " S2: " + s2 + "\n";
        if (s1 < this.decisionModule.agentParameters.effortAttributeCoeficient * 100) // can't be higher than that
        {
            p1 = s1;
        } else {
            p1 = (int) (this.decisionModule.agentParameters.effortAttributeCoeficient * 100);
        }

        if (s2 < this.decisionModule.agentParameters.effortCoeficient * 100) {
            p2 = s2;
        } else {
            p2 = (int) (this.decisionModule.agentParameters.effortCoeficient * 100);
        }
        logReport += "P1: " + p1 + " P2 " + p2 + "\n";
        // so, I counted sum of basic attractivity and attractivity of items I currently see 
        // now the probability count
        coefficient = 1;
        for (Action action : this.percField.processArea) {
            if (this.decisionModule.actualAction != null && this.decisionModule.actualAction.name.contains("_Search") && action.equals(this.decisionModule.actualAction)) {
                for (Affordance aff : newItem.getAffordances()) {
                    if (this.decisionModule.actualAction.satisfyingItems.keySet().contains(aff.type)) {
                        coefficient += 4; // so far I do it static, then we can think about some improvement
                    }
                }
            } else {
                for (Affordance aff : newItem.getAffordances()) {
                    if (action.enabled && action.satisfyingItems != null &&
                            action.satisfyingItems.keySet().contains(aff.type)) {
                        coefficient += 1; // if item is important for another preactive action
                    }
                }
            }
        }
        attr1 = newItem.getBasicAttractivity() - newItem.decreaseOfAttractivity;
        attr2 = newItem.getAttractivity() - newItem.decreaseOfAttractivity;
        attr2 *= coefficient;
        logReport += "Attr1: " + attr1 + " Attr2 " + attr2 + "\n";
        if ((attr1 + attr2) == 0 || (s1 + s2) == 0 || (s1 == 0 || s2 == 0)) {
            probability = 0;
        } else if (attr1 == 0 || s1 == 0) {
            probability = attr2 * p2 / s2;
        } else if (attr2 == 0 || s2 == 0) {
            probability = attr1 * p1 / s1;
        } else {
            probability = attr1 * p1 / s1 + attr2 * p2 / s2;
        }
        probability = (int) Math.round(probability * (1 - this.percField.stress));
        logReport += "Probability: " + probability + " stress: " + this.percField.stress;
        if (print) {
            log.info(logReport);
        }
        return probability;
    }

    /**
     * called from SearchEnvironment as well as from SearchRandom
     * again a little bit of a dark magic, so look at it closely, study Klara's bachelor thesis
     * @param affordance - affordance, which we are looking for
     * @param action - action which will be satisfied by the item
     * @return
     */
    public EItem seeingItem(AffordanceType affordance) {
        HashMap<Long, Tuple> probabilityRange = new HashMap<Long, Tuple>();
        int start = 0, probability = 0, random;
        Tuple temp = null;
        EItem interestingThing = null, source = null;
        synchronized (visibleItems) {
            for (EItem item : this.visibleItems.values()) {
                probability = this.probability(item);
                temp = new Tuple(start, start + probability);
                start += probability;
                probabilityRange.put(item.getId(), temp);
            }
            random = Math.round((float) Math.random() * 100);
            for (long ID : probabilityRange.keySet()) {
                if (random > probabilityRange.get(ID).a && random < probabilityRange.get(ID).b) {
                    interestingThing = visibleItems.get(ID);
                    break;
                }
            }
        }
        if (interestingThing == null) {
            return null;
        }
        if (affordance != null) {
            for (Affordance aff : interestingThing.getAffordances()) {
                if (aff != null && aff.type != null && aff.type.equals(affordance)) {
                    source = interestingThing;
                }
            }
        }
        if (source != null) { // see item of the affordance
            this.log.fine("Found source for affordance: " + affordance + " | | " + source.getName());
            this.percField.addSource(source);
            return source;
        } else {
            this.addInterestingItem(interestingThing);
        }
        return null;
    }

    /**
     * used for instance in the increaseAttractivityToAllSources in perceptive field
     * gets an item of provided AffordanceType
     * 
     * @return
     */
    public ArrayList<EItem> getItemOfAffordance(AffordanceType aff) {
        ArrayList<EItem> result = new ArrayList<EItem>();
        synchronized (visibleItems) {
            for (EItem item : this.visibleItems.values()) {
                if (item == null) {
                    continue;
                }
                for (Affordance affordance : item.getAffordances()) {
                    try {
                        if (affordance.type.equals(aff)) {
                            result.add(item);
                        }
                    } catch (NullPointerException e) {
                        log.severe("Error in getItemOfAffordance! Input: " + aff + ". Item: " + item + " \nAffordance: " + affordance + " Affordance type: " + affordance.type);
                    }
                }
            }
        }
        return result;
    }

    void setDMS(DecisionModuleImpl dms) {
        this.decisionModule = dms;
    }

    void setLog(Logger log) {
        this.log = log;
    }

    void setPercField(PerceptiveField perceptiveField) {
        this.percField = perceptiveField;
    }

    /**
     * Adds spontaneous actions connected with the item
     * @param interestingThing
     */
    private void addInterestingItem(EItem interestingThing) {
        this.percField.addSource(interestingThing);
        Action newAction = null;
        if (night) // no spontaneous actions during the night
        {
            return;
        }
        if (this.decisionModule.counter % TimeUtils.minutesToTicksOfLogic(30) == 0) {
            for (Affordance aff : interestingThing.getAffordances()) {
                for (Action act : this.possibleActions) {
                    if (act.satisfyingItems == null || act.satisfyingItems.isEmpty()) {
                        continue;
                    }
                    if (act.satisfyingItems.keySet().contains(aff.type)) {
                        newAction = act.cloneBySerialize(decisionModule);
                        newAction.intention = null;
                        newAction.activity = interestingThing.getAttractivity();
                        percField.addAction(newAction);
                    }
                }
            }
        }
    }

    private String printVisibleItemsNames() {
        String result = "Visible items: \n";
        synchronized (visibleItems) {
            for (long ID : visibleItems.keySet()) {
                result += "ID:" + ID + " name:" + visibleItems.get(ID).name + " affordances: ";
                for (Affordance aff : visibleItems.get(ID).getAffordances()) {
                    result += aff.type + " ";
                }
                result += "\n";
            }
        }
        return result;
    }

    /**
     * @return the ready
     */
    public boolean isReady() {
        return ready;
    }

    /**
     * @param ready the ready to set
     */
    public void setReady(boolean ready) {
        this.ready = ready;
    }
}
