package decisionMakingSystem;

import atomicActions.AtomicAction;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.Configuration;
import utils.Interval;
import bot.DMSMemoryVO;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.logging.Logger;
import utils.TimeUtils;
import bot.Bot;
import java.io.Serializable;
import pogamutEndEvent.PogamutEndEventSource;
import pogamutEndEvent.PogamutEndEventType;
import scheduler.ProbabilisticScheduler;
import utils.FileSaveLoadUtils;

/**
 * main decision making module - presents the logic of the bot!
 * it takes care about everything about the main logic cycle
 * choice of intention to be performed, action to be performed, update of fadeout, attractivity,
 * rollback when something fails, update of attractivity of intentions etc.
 * 
 * basicaly all you need to know about the usage in Agent is that you has to call prepareLogic() in the prePrepareAgent()
 * and doReasoning() in the doLogic, before using this module you need to specify couple XML files and if necessary atomicActions.
 * The XML files are Loader.xml and then one for each root intention. All available atomic actions are in the atomicactions package
 * and all affordance types are in the Enum AffordanceType so you will might need to change those as well. Items are handeled in the
 * itemParameters.xml file. And note that if you add a new AtomicAction you need to add it to the BasicIntentionLoader to 
 * importIntentionForest() so it will be proceeded correctly.
 * 
 * @author Ondrej
 */
public class DecisionModuleImpl implements DecisionModule, Serializable {

    public PogamutEndEventSource eventSource = null;
    /** fast access to agent log */
    public transient Logger log;
    /** link to agent */
    public transient Bot agent = null;
    /** link on perceptive field to access actions, perceived items etc. */
    public transient PerceptiveField perceptiveField = null;
    /** list of basic intentions - for the current day */
    public ArrayList<Intention> basicIntentions = null;
    /** list of all intentions */
    public ArrayList<Intention> allIntentions = null;
    /** list of intentions which are currently active or preactive */
    public ArrayList<Intention> intentionArea = null;
    /** link to thing manager -> takes care about all visible items */
    public transient ThingsManager things = null;
    /** time counter */
    public int counter = 0;
    /** agent inventory - handles picked up things */
    public transient Inventory inventory = null;
    /** currently choosen intention */
    public Intention actualIntention = null;
    /** currently choosen action */
    public Action actualAction = null;
    /** creates plan for each day */
    Scheduler scheduler = null;
    /** agent parameters -> used for stres count and filter of attention, etc. */
    public AgentParameters agentParameters = null;
    /** */
    public HashMap<String, Intention> nameToIntention = null;
    /** for the test case running - see test() for details */
    private int testCaseCounter = 0;
    /** actual atomic action */
    private AtomicAction actualAtomicAction = null;
    /** previous atomic action */
    private AtomicAction oldAtomicAction = null;
    private transient DMSLogger dmsLogger = null; // injected from Main
    public ItemMemory memory;

    public DecisionModuleImpl() {
    }

    public DecisionModuleImpl(ItemMemory itemMemory) {
        this.memory = itemMemory;
    }

    public void changeLifestyle(String directory, String planningFilename, int i) {
        scheduler.changeLifestyle(directory, planningFilename, i);
        scheduler.scheduleNewDay(counter);
        log.config("New lifestyle prepared");
    }

    /**
     * quite close to the constructor, prepares everything in the logic
     * (loads intentions from XML, initialize all structures). it should be called
     * in the prePrepareAgent() in Agent
     * @param agent - pointer to agent
     * @param directory - path, where are located Loader.xml and XML files of root intentions
     */
    public void prepareLogic(Bot agent, String directory, String planningFilename) {
        this.log = agent.getLog();
        this.agent = agent;
        this.perceptiveField = new PerceptiveField(agent.getLog(), things, this);
        this.allIntentions = BasicIntentionLoader.importIntentionForest(directory, this);
        this.intentionArea = new ArrayList<Intention>();
        this.things = new ThingsManager(agent, log, perceptiveField, directory, this);
        this.perceptiveField.allThings = this.things;
        this.inventory = new Inventory(agent, this);
        this.scheduler = new ProbabilisticScheduler(allIntentions, log, directory, planningFilename);
        //this.scheduler = new Scheduler(allIntentions, log, directory, planningFilename);
        nameToIntention = new HashMap<String, Intention>();
        for (Intention intention : allIntentions) {
            nameToIntention.put(intention.getName(), intention);
        }
        agentParameters = new AgentParameters();
        GlobalParameters.getInstance();
        things.setReady(true);
        inventory.setReady(true);
        this.eventSource = new PogamutEndEventSource();
    }

    public void sensing() {
        counter++;
        synchronized (things.visibleItems) {
            if (!things.visibleItems.isEmpty() && counter % 20 == 0) {
                for (EItem item : things.visibleItems.values()) {
                    memory.addItemToMemory(item, 0, 1);
                }
            }
        }
    }

    /**
     * set GMT and replan the schedule for the day
     * @param GMTTime
     */
    public void setGMT(double gmtTime) {
        GMTTime.setGMT(gmtTime, this);
    }

    /**
     * Start of a recursive proceeding, where we start from intention and ends up with action which is ready to
     * execute its atomic actions.
     * This start is rather easy: it just checks if the intention has some actions (possibilities) to follow, if not it fails
     * and if yes I prepare the action and return it.
     *
     *
     * @param intention intention to start from
     * @return action ready to execute atomic action
     */
    protected Action chooseAction(Intention intention) {
        Action result = intention.chooseAction();
        // if intention can't choose next action means it failed
        if (result == null) {
            intention.setRealized(IntentionStates.FAILED);
            if (!intention.getName().contains("Want")) {
                String toLog = "";
                toLog += TimeUtils.counterToDate(counter + TimeUtils.minutesToTicksOfLogic(getGMTTime() * 60)) + " ";
                toLog += intention.getName() + " failed.\r\n";
                dmsLogger.writeAMessage(toLog);
            }
            return null;
        }
        // add it to preactive actions - guarded in PerceptiveField
        perceptiveField.addAction(result);
        return prepareAction(result);
    }

    /**
     * Part of recursive call. It prepares action to execute, returned action has to be ready to execute atomic action.
     * That means that all its sub-intentions are successfuly finished and all affordances are satisfied.
     * So this method takes care about putting in sub-intentions, creating Want intentions etc.
     *
     * The split of chooseAction to chooseAction and prepareAction enables inputing spontaneous actions elegantly
     * @param action which should be prepared
     * @return action (some son of the provided action) which can be executed - its atomic action can be executed
     */
    protected Action prepareAction(Action action) {
        Intention temp = null;
        boolean ready = true;
        // there are still some intentions to fullfill
        if (action.getCurrentIntention() != null) {
            temp = chooseIntention(action);
            return temp.getChosenAction();
        } else {
            // there are some unsatisfied affordances! we need to satisfy them before atomic action execution => create Want intention
            if (action.satisfyingItems != null && !action.name.contains("_Search")) { // no Want creating more Want:) (ethernal loop:))
                for (AffordanceType aff : action.satisfyingItems.keySet()) {
                    if (!perceptiveField.isAffordanceSatisfied(aff, action)) { // affordance is not satisfied
                        ready = false;
                        Intention want = new Want(aff, action, this); // create Want intention which is linked to current action
                        if (action.intentions == null) // leaves don't have intentions to do
                        {
                            action.intentions = new ArrayList<Intention>();
                        }
                        action.intentions.add(want);
                    }
                }
            }
            // not ready -> means that some Want intention was added => choose it => choose some Search action
            if (!ready) {
                temp = chooseIntention(action);
                return temp.getChosenAction();
            } else {
                // there are some atomic actions to do, so I found my action to execute
                if (action.getCurrentAtomicAction() != null) {
                    action.state = ActionStates.EXECUTE;
                    return action;
                } else {
                    if (!action.state.equals(ActionStates.FAILED)) // maybe not necessary, just for case, that execute set it to fail
                    {
                        action.state = ActionStates.TERMINATED; // means intentions are finished => affordances satisfied, no atomic actions to do => finish
                    }
                    return action;
                }
            }
        }
    }

    /**
     * recursive function to choose current intention to fullfill
     * @param action action to start from
     * @return parental intention of action which is ready to execute - null if not possible to choose next
     */
    protected Intention chooseIntention(Action action) {
        // choose intention from action
        Intention intention = action.chooseIntention();
        if (intention != null) {
            Action temp = null;
            // add it to intentionArea
            this.addIntention(intention);
            // choose preactive action for it
            temp = chooseAction(intention);
            return temp.intention;
        }
        return null;
    }

    /**
     * Recursive action which is executed when action fails. It removes action and
     * its descendants (shouldn't be necessary now). if spontaneous, then its finished
     * as it doesn't need to propagate fail up some tree.
     *
     * Normal action forces parental intention to choose new action and if there are not any,
     * it calls intentonFailed as it fails then as well.
     *
     * @param action action which failed
     */
    protected void actionFailed(Action action) {
        eventSource.fireEvent(PogamutEndEventType.ACTION_SUCCEEDED, action);
        // remove it and its descendants (Want) from perceptive field
        this.removeActionAndDescendantIntentions(action);
        // spontaneous
        if (action.intention == null) {
            return;
        }
        // if it contains Want, I force choice of new action just by setting chosenAction to null
        action.intention.setChosenAction(null);
        // for normal intention I need to remove action from notTriedActions
        if (!action.intention.getName().contains("Want")) {
            action.intention.getNotTriedActions().remove(action);
        }
        // choose new preactive action for the parental intention
        Action newAction = chooseAction(action.intention);
        // no other possibilities (actions to satisfy intention) => intention failed
        if (newAction == null) {
            action.intention.setRealized(IntentionStates.FAILED);
            intentionFailed(action.intention); // recursive call
        }
    }

    /**
     * success of an action means its removal from preactive actions and success of
     * parental intention
     * @param action action which succeeded
     */
    protected void actionSucceeded(Action action) {
        eventSource.fireEvent(PogamutEndEventType.ACTION_SUCCEEDED, action);
        // remove it and its descendants (Want) from perceptive field
        this.removeActionAndDescendantIntentions(action);
        this.throwAwayUsedThings(action);
        // spontaneous
        if (action.intention == null) {
            return;
        }
        // action succeeded so parental intention succeeded as well
        action.intention.setRealized(IntentionStates.YES);
        intentionSucceeded(action.intention); // recursive call
    }

    /**
     * Intention fails => parental action fails as well.
     * If there is not a one (root intention), it means dropout of the intention.
     * @param intention which failed
     */
    protected void intentionFailed(Intention intention) {
        eventSource.fireEvent(PogamutEndEventType.INTENTION_FAILED, intention);
        this.removeIntention(intention);
        // root
        if (intention.getAncestorAction() == null) {
            dropOut(intention); // DROPOUT even when failed?
        } else {
            intention.getAncestorAction().state = ActionStates.FAILED;
            actionFailed(intention.getAncestorAction()); // recursive call, parental action failed as well
        }
    }

    /**
     * When intention succeeds, parental action can continue with next intention.
     * When it is a root intention, than it's the end of execution of the intention => dropout.
     * @param intention
     */
    protected void intentionSucceeded(Intention intention) {
        eventSource.fireEvent(PogamutEndEventType.INTENTION_SUCCEEDED, intention);
        this.removeIntention(intention);
        // root -> dropout
        if (intention.getAncestorAction() == null) {
            dropOut(intention);
        } else {
            // move to next intention
            intention.getAncestorAction().unfullfiledIntentionsIndex += 1;
            // choose new intention and add respectfull preactive action to process area
            Intention newIntention = chooseIntention(intention.getAncestorAction());
            // if there are no intentions to choose
            if (newIntention == null) {
                // and if there are no more atomic actions to execute and as well no intention => action succeeded
                if (intention.getAncestorAction().getCurrentAtomicAction() == null) {
                    actionSucceeded(intention.getAncestorAction()); // recursive call
                }                // else there is something to execute, so I can stop for the moment and let doReasoning execute it
            }
        }

    }

    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /**
     * main function of DecisionModuleImpl, the only one that should be called from doLogic of Agent.
     * So it goes as follows:
     * <ol>
     * <li> recount activities of root intentions and get the most active of root intentions and of the preactive intentions in the intentionArea
     * <li> chooseAction - means to ensure that chosen intention has some preactive action in the perceptive field
     * <li> choose the most active action from preactive actions
     * <li> as actions are enabled only if they can already execute atomic action, execute it
     * <li> check if the execution of atomic action didn't lead to finish or fail of action (if yes, do the rollback)
     * <li> increase duration
     * <li> at the end update attractivity, activity, fadeout
     * </ol>
     */
    @Override
    public DMSMemoryVO doReasoning(int myCounter) {

       // eventSource.fireEvent();

        everyRoundUpdate(myCounter); // day/night, daily reports, neural memory learning etc.
        //also updates counter to myCounter
        counter = myCounter;

        this.things.seeingItem(null); // look around for attractive items
        perceptiveField.updateStress();
        perceptiveField.updateActivities(counter);
        perceptiveField.updateFadeout();
        updateIntentionFadeout(); // takes care about fadeouts of intentions from preactive intentions
        actualIntention = this.checkRootIntentions(counter);
        AtomicAction tmpAtomicAction = null;
        oldAtomicAction = null;
        if (actualIntention != null) { // sometimes nothing in the basic intentions and intentions, still can do some spontaneous actions
            this.addIntention(actualIntention);
            actualAction = this.chooseAction(actualIntention); // now I should have an action ready to execute in preactive actions of perceptive field
            if (!actualAction.intention.equals(actualIntention)) {
                actualIntention = actualAction.intention;
            }
        }
        actualAction = perceptiveField.chooseAction(counter); // choses the most active of preactive actions

        if (actualAction != null && actualAction.intention == null) // spontaneous action
        {
            actualAction = this.prepareAction(actualAction); // action execution preparation -> generates Wants etc.
        }        // /*
        synchronized (this) {
            log.finest("Counter: " + counter + "\n INTENTION AREA: " + this.intentionArea + " \n\n" +
                    "PROCESS AREA: " + perceptiveField.processArea + " \n\n" + perceptiveField.printPerceivedItems() + "\n\n GMT: " + GMTTime.getGMT());
        }
        if (counter % TimeUtils.minutesToTicksOfLogic(30) == 0) {
            log.finest("Basic intentions: " + basicIntentions);
        }
        // */
        if (actualAction == null) { // nothing to do -> change the agent name
            //this.agent.getAct().act(new Configuration().setName("Bored to death!"));
            //MB - changed this here, so we schedule new intentions when nothing to do
            if (scheduler.getClass().equals(ProbabilisticScheduler.class)) {
                this.basicIntentions = ((ProbabilisticScheduler) scheduler).scheduleNewIntention(counter);
            } else {
                this.agent.getAct().act(new Configuration().setName("Bored to death!"));
            }
            return new DMSMemoryVO(this);
        }
        this.perceptiveField.increaseAttractivityToAllSources(actualAction); // does as well updateAttractivities
        if (actualAtomicAction == null) {
            actualAtomicAction = actualAction.executeAtomicActions(this.log);
            tmpAtomicAction = actualAtomicAction;
        } else {
            tmpAtomicAction = actualAction.executeAtomicActions(this.log);
        }
        if (actualAtomicAction != null && !actualAtomicAction.equals(tmpAtomicAction)) {
            oldAtomicAction = actualAtomicAction;
            actualAtomicAction = tmpAtomicAction; // actual action furt null, tedy se sem nikdy nedostane!
        }
        // if it failed / succeeded after execution, I call actionFailed, actionSucceeded to carry out rollback
        if (actualAction == null || actualAction.state == null) {
            return new DMSMemoryVO(this);
        }
        if (actualAction.state.equals(ActionStates.FAILED)) {
            this.actionFailed(actualAction);
        }
        if (actualAction.state.equals(ActionStates.TERMINATED)) {
            this.actionSucceeded(actualAction);
        }
        // not spontaneous, I find root and increase its duration
        if (actualAction.intention != null) {
            actualAction.intention.rootIntention().increaseDuration();
        }
        return new DMSMemoryVO(this);
    }
    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * Reschedules basic actions so the activation intervals will correspond to
     * the new time zone.
     *
     * I need to ensure the continuity for the logic counter. So agent can directly continue performing the plan.
     *
     * For instance when agent has 6 a.m. and he changes the time-zone to +7 GMT he will have 1 p.m. on his watches.
     * Nevertheless the counter will remain the same. So we replan to make a plan which started a bit (7 hours) earlier
     * to allow for the illusion that the actions he is performing corresponds with 1 p.m.
     *
     * The time is recounted every time it's used after that - midnight would be 7 hours earlier as well. So the next day start
     * will be simply the counter at the earlier midnight.
     *
     * @param GMT
     */
    void replan(double GMT) {
        int newStart = counter - counter % GlobalParameters.LENGHT_OF_A_DAY;
        newStart -= TimeUtils.minutesToTicksOfLogic(GMT * 60);
        if (GMT < 0 && TimeUtils.counterToHours(counter) < -GMT) { // when it's too early in the morning and agent goes into the previous day
            newStart -= GlobalParameters.LENGHT_OF_A_DAY;
        }
        basicIntentions = scheduler.scheduleNewDay(newStart);

        log.config("Replanning the day! Time zone changed:" + GMT + " \nIn the ticks of logic: " + TimeUtils.minutesToTicksOfLogic(-GMT * 60.0) + " \nNew start: " + newStart + " counter: " + counter);

        intentionArea.clear();
        perceptiveField.clearProcessArea();
        log.config("New plan: " + basicIntentions);
    }

    /**
     * adds intention to preactive intentions in intentionArea - only if it is not already there
     * @param intention - intention to add
     */
    private void addIntention(Intention intention) {
        if (!intentionArea.contains(intention)) {
            // adding new intention -> disable grandfather
            if (intention.getAncestorAction() != null && intention.getAncestorAction().intention != null) {
                intention.getAncestorAction().intention.setEnabled(false);
            }
            this.intentionArea.add(intention);
            intention.setFadeout(GlobalParameters.FADEOUT_FOR_INTENTIONS);
        }
    }

    /**
     * updates the basic intentions (roots of intention-action trees)
     * and finds the most active intention among those basic intentions and the intentions in intentionArea
     *
     * NOTE! very important, it returns a hard copy, if the most active intention comes from basicIntentions!!!
     * NOTE: it makes a DEEP COPY of the intention!!! so it is added there as an exact copy of basic intention, thus it is
     * not used, thus you don't have to set anything starting from index of fullfilled actions, atomic actions, anything
     * @param timeCounter
     * @return most active intention from basic intentions and intentionArea
     */
    private Intention checkRootIntentions(int timeCounter) {
        Intention maxIntention = null;
        int maxActivity = 0;
        int index = 0;
        Intention temp = null;
        boolean fromBasicIntentions = false;
        for (; index < intentionArea.size();) {
            temp = intentionArea.get(index);
            // this is there to remove intention which failed due to timeout, thus it won't win a intention race
            // and thus won't be considered and checked during checkIntentionsAndActions() thus not removed
            if (temp.getRealized() == IntentionStates.FAILED || temp.getRealized() == IntentionStates.YES) {
                this.removeIntention(temp);
            } else {
                if (temp.isEnabled() && temp.getActivity() > maxActivity) {
                    maxIntention = temp;
                    maxActivity = temp.getActivity();
                }
                index++;
            }
        }
        for (Intention i : basicIntentions) {
            i.updateActivity(timeCounter);
            // what if an intention is already there but is disabled -> continue
            if (intentionArea.contains(i) && !intentionArea.get(intentionArea.indexOf(i)).isEnabled() && intentionArea.get(intentionArea.indexOf(i)).getActivity() == i.getActivity()) {
                continue;
            } else if (i.getActivity() > maxActivity) {
                fromBasicIntentions = true;
                maxIntention = i;
                maxActivity = i.getActivity();
            }
        }
        // basic intention got it -> need a hard copy, which will be added to intention area
        if (fromBasicIntentions) {
            log.info(TimeUtils.counterToDate(GMTTime.getRealTimeInTicks(counter)) + " Max intention choosen from basicIntentions: " + maxIntention.getName());
            return maxIntention.cloneBySerialize(agent, this); // now this deep copy is crucial - it has to be done this way, so actualIntention will not be changed
        }
        eventSource.fireEvent(PogamutEndEventType.INTENTION_AREA, intentionArea);
        return maxIntention;
    }

    /**
     * removes action from processArea
     * removes descendant intentions of action
     * be careful about Want intentions as they can be descendants of more than one action
     *
     * @param actualAction
     */
    protected void removeActionAndDescendantIntentions(Action actualAction) {
        this.log.fine("removing action: " + actualAction.name + " state: " + actualAction.state);
        this.perceptiveField.removeAction(actualAction); // we remove it first
        // when removing I must enable father and grandfather (father should be enabled already, but just in case)
        if (actualAction.intention != null) {
            actualAction.intention.setEnabled(true);
            if (actualAction.intention.getAncestorAction() != null) {
                actualAction.intention.getAncestorAction().enabled = true;
            }
        }

        int index = 0; // index to iterate through intentionArea - necessary to avoid concurrent modification exception while using iterator
        Intention temp = null; // intention considered to be removed
        for (; index < intentionArea.size();) {
            temp = intentionArea.get(index);
            if (temp.getAncestorAction() != null && temp.getAncestorAction().equals(actualAction)) {
                // if it has as a father actual intention it is a subject to be removed
                log.fine("removing " + intentionArea.get(index));
                // and I need to remove its children as well from process area
                this.perceptiveField.removeActionsOfIntention(temp);
                intentionArea.remove(index);
            } else {
                index++; // has no connection with actualAction - keep iterating
            }
        }
    }

    /**
     * so far just removes intention from intention area without anything else as intentions are rolling back differently
     * and when one fails it doesn't mean a disaster - so just the parent action failed as well.
     * intentions are not linked to any items, want intentions etc. so it is not that difficult to delete it
     */
    protected void removeIntention(Intention intention) {
        // when removing I must enable father and grandfather (father should be enabled already, but just in case)
        if (intention.getAncestorAction() != null) {
            intention.getAncestorAction().enabled = true;
            if (intention.getAncestorAction().intention != null) {
                intention.getAncestorAction().intention.setEnabled(true);
            }
        }
        this.intentionArea.remove(intention);
        this.perceptiveField.removeActionsOfIntention(intention);
    }

    /**
     * executes drop out on the basic intention which correspond to finished intention
     * @param actualIntention
     */
    private void dropOut(Intention actualIntention) {
        for (Intention i : basicIntentions) {
            if (i.getName().equals(actualIntention.getName())) {
                i.dropOut(counter);
                // log.info(counter + " counter. Dropped out intention: " + i);
                break;
            }
        }
    }

    /**
     * checks all preactive intentions and increases fadeout in those who are not executed yet active (enabled)
     */
    private void updateIntentionFadeout() {
        int index = 0;
        Intention intention = null;
        for (; index < this.intentionArea.size();) {
            intention = this.intentionArea.get(index);
            if (intention.isEnabled() && !intention.equals(actualIntention)) {
                intention.decreaseFadeout();
            }
            if (intention.getFadeout() < 0) {
                this.removeIntention(intention);
            } else {
                index++;
            }
        }
    }

    /**
     * call throw away on each item used by successful action -> the inventory handles whether it is used somewhere else
     * and call body.throw(...)
     * @param action
     */
    private void throwAwayUsedThings(Action action) {
        if (action.satisfyingItems == null) {
            return;
        }
        for (EItem item : action.satisfyingItems.values()) {
            this.inventory.throwAway(item);
        }
    }

    /**
     * This method adds dynamicaly new intention to the daily plan.
     * It shouldn't be called again until that intention activation interval expires.
     * How does it works? First I check if the intention is present in the daily plan,
     * if so I add new interval to it, if it is already active (there is an interval which is active
     * at the current time), I don't do anything.
     * Completely new intention is obtained from the hash map of (name,Intention) and I make a hard copy of it
     * as I will be modifying activation intervals
     * @param name - name of the intention
     * @param actInterval - activation interval of the inserted intention
     */
    @Override
    public void addIntention(String name, Interval actInterval) {
        boolean addInterval = true;
        boolean addIntention = true;
        // so first I check whether the intention is already present in the daily plan
        for (Intention intention : this.basicIntentions) {
            if (intention.getName().equals(name)) {
                addIntention = false;
                // if so, I will check if the interval is already there => e.g. I added it in the previous step
                for (Interval interval : intention.getActivationIntervals()) {
                    if (interval.isInInterval(actInterval.getLeftSide())) {
                        addInterval = false;
                        return;
                    }
                }
                // if not, I add the interval to intention's activationIntervals
                if (addInterval) {
                    intention.getActivationIntervals().add(actInterval);
                    log.config("intention: " + name + ". New activation interval added: " + actInterval + " \n intention: " + intention);
                    return;
                }
            }
        }
        // when the intention is not in the daily plan, I need to add it
        if (!addIntention) {
            return;
        }
        // so I make a copy from allIntentions (faster access throught the map)
        // if it is not there I can't add it
        if (!this.nameToIntention.containsKey(name)) {
            log.warning("About to add unknown intention " + name);
            return;
        }
        Intention intentionToAdd = this.nameToIntention.get(name).cloneBySerialize(agent, this);
        ArrayList<Interval> activationIntervals = new ArrayList<Interval>();
        // add it a new interval
        activationIntervals.add(actInterval);
        intentionToAdd.setActivationIntervals(activationIntervals);
        // and add it to the daily plan
        basicIntentions.add(intentionToAdd);
        log.config("intention: " + name + " added. Activation interval: " + actInterval + " \n Intention: " + intentionToAdd);
    }

    /**
     * procedure which executes all testcases
     * should be put in the agent.doLogic() instead of doReasoning()
     */
    public DMSMemoryVO test() {
        switch (testCaseCounter) {
            /**
            case 0:
            // if (testCaseReachPointInWant())
            testCaseCounter++;
            break;
            case 1:
            // if (testCaseStressAndConcentration())
            testCaseCounter++;
            break;
            case 2:
            // if (testCaseItemFadeout())
            testCaseCounter++;
            break;
            case 3:
            // if (testCaseItemAttractivity())
            testCaseCounter++;
            break;
            /*
            case 4:
            if (testCaseOne())
            counter++;
            break;
             * */
            default:
                return doReasoning(counter);
        }
    }

    private boolean testCaseItemAttractivity() {
        doReasoning(counter);
        String result = "";
        for (EItem item : this.things.visibleItems.values()) {
            result += "Item: " + item.name + " attractivity: " + item.getAttractivity();
            result += " basic attractivity: " + item.getBasicAttractivity();
            result += " decrease: " + item.decreaseOfAttractivity + " fadeout: " + item.getFadeout() + "\n";
        }
        log.config(result);
        log.config("Perceptive field: " + this.perceptiveField.printPerceivedItems());
        if (counter % 1000 == 999) {
            return true;
        }
        return false;
    }

    /**
     * @return
     */
    private boolean testCaseStressAndConcentration() {
        doReasoning(counter);
        if (counter % 1000 == 999) {
            return true;
        }
        return false;
    }
    /**
     * just for item scenario
     */
    private Intention searchIntention = null;

    /**
     * a test case which serves to check if Want works correctly - e.g. attention, search environment, pocket, memory etc.
     * good to do some basic tuning of memory - like how to count the trustfullness of an item
     */
    public void doReasoningItemScenario() {
        // some affordances to pick from
        if (counter == 0) {
            searchIntention = new Intention();
            ArrayList<Action> actions = new ArrayList<Action>();
            Action action = new Action();
            action.satisfyingItems.put(AffordanceType._TO_STUDY_ON, null);
            action.satisfyingItems.put(AffordanceType._TO_PLAY_ON, null);
            action.satisfyingItems.put(AffordanceType._TO_WASH_WITH, null);
            action.satisfyingItems.put(AffordanceType._TO_STUDY_AT, null);
            action.satisfyingItems.put(AffordanceType.TO_READ, null);
            action.satisfyingItems.put(AffordanceType.TO_RUN_IN, null);
            action.satisfyingItems.put(AffordanceType.TO_EAT, null);
            actions.add(action);
            action.timeLimit = 300;
            action.intention = searchIntention;
            action.name = "Test Search Action";

            searchIntention.setName("Search INTENTION");
            searchIntention.setActions(actions);
            ArrayList<Interval> activationIntervals = new ArrayList<Interval>();
            activationIntervals.add(new Interval(counter, counter + 100, 80));
            searchIntention.setActivationIntervals(activationIntervals);
            this.basicIntentions.add(searchIntention);
        }
        if (actualIntention != null && actualIntention.getName().equals("Search INTENTION") && ((actualIntention.getRealized().equals(IntentionStates.FAILED)) || (actualIntention.getRealized().equals(IntentionStates.YES)))) {
            ArrayList<Interval> activationIntervals = new ArrayList<Interval>();
            activationIntervals.add(new Interval(counter, counter + 100, 80));
            searchIntention.setActivationIntervals(activationIntervals);
        }
        doReasoning(counter);

    }

    private void everyRoundUpdate(int myCounter) {
        counter = myCounter;
        if (GMTTime.isMidnight(counter) || this.basicIntentions == null) {
            //if basic intentions are null it means we are at the beginning of bots life - we should schedule something
            this.basicIntentions = scheduler.scheduleNewDay(counter);
            log.warning("Plan of " + TimeUtils.counterToDate(GMTTime.getRealTimeInTicks(counter)) + ": " + this.basicIntentions);
            this.intentionArea.clear();
            this.perceptiveField.clearProcessArea();
        }
        this.things.everyRoundUpdate(counter); // does nothing so far
    }

    @Override
    public Action getActualAction() {
        return this.actualAction;
    }

    @Override
    public Intention getActualIntention() {
        return this.actualIntention;
    }

    @Override
    public int getCounter() {
        return this.counter;
    }

    @Override
    public AtomicAction getActualAtomicAction() {
        return this.actualAtomicAction;
    }

    @Override
    public AtomicAction getOldAtomicAction() {
        return this.oldAtomicAction;
    }

    @Override
    public double getGMTTime() {
        return GMTTime.getGMT();
    }

    public void saveToFile(String filename) {
        FileSaveLoadUtils.saveObject(this, filename, this.log);
    }

    public static DecisionModuleImpl loadFromFile(String filename, Logger log, Bot agent, String directory) {
        Object dms1 = FileSaveLoadUtils.loadObject(filename, log);
        if (!(dms1 instanceof DecisionModuleImpl)) {
            throw new IllegalStateException("The choosen file does not contain object of class DecisionModuleImpl");
        }
        DecisionModuleImpl dms = (DecisionModuleImpl) dms1;
        dms.inventory = new Inventory(agent, dms);
        dms.perceptiveField = new PerceptiveField(log, null, dms);
        dms.things = new ThingsManager(agent, log, dms.perceptiveField, directory, dms);
        dms.perceptiveField.setThings(dms.things);
        dms.memory = agent.itemMemory;
        dms.setLoggerAfterLoad(log);
        dms.setAgentAfterLoad(agent);
        dms.things.setReady(true);
        dms.inventory.setReady(true);
        dms.setToNullAllNeeded();
        dms.replan(dms.getGMTTime());
        return dms;
    }

    private void setToNullAllNeeded() {
        actualAction = null;
        actualIntention = null;
        actualIntention = null;
        oldAtomicAction = null;
        intentionArea.clear();
        perceptiveField.processArea.clear();
        perceptiveField.perceivedItems.clear();
        things.visibleItems.clear();
    }

    private void setLoggerAfterLoad(Logger log) {
        this.log = log;
        scheduler.setLog(log);
    }

    private void setAgentAfterLoad(Bot agent) {
        this.agent = agent;
    }

    public void setDMSLogger(DMSLogger logger) {
        dmsLogger = logger;
    }
}
        
        