package decisionMakingSystem;

import bot.DMSMemoryVO;
import atomicActions.AtomicAction;
import bot.FileLog;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;
import utils.DateUtils;
import utils.TimeUtils;

/**
 * DMSLogger logs agents activity to a file. The output is quite simple. It logs
 * just a time and a name of an action and its status - started, interrupted, finished, failed.
 * @author Ondrej
 */
public class DMSLogger implements FileLog {

    /** last written atomic action */
    private AtomicAction lastAtomicAction = null;
    /** a file to log to */
    private File file = null;
    private Logger userLog = null; // just for debug
    /** an output stream to write to */
    private FileOutputStream outputFile = null;
    /** if the logger is ready */
    private boolean ready = false;

    public DMSLogger(Logger log, String filename) {
        file = new File(filename);
        userLog = log;
        try {
            if (file.exists()) {
                outputFile = new FileOutputStream(file, true);
            } else {
                boolean createNewFile = file.createNewFile();
                if (!createNewFile) {
                    log.severe("File not initiated." + file.getAbsolutePath());
                }
                outputFile = new FileOutputStream(file);
            }
        } catch (IOException e) {
            log.severe("IO Exception when creating DMSLogger: " + e);
        }
        ready = true;
        userLog.info("File Logger initiated." + filename);
    }

    /**
     * Formats correctly a log output of old message.
     * @param action
     * @param counter
     * @return formatted message
     */
    private String getFormattedMessage(DMSMemoryVO memVO) {
        String toLog = "";
        AtomicAction action = memVO.getOldAtomicAction();
        if (action != null && !action.type.name().contains("SEARCH_")) {
            toLog += TimeUtils.counterToDate(memVO.getThoughtTimeTicks()) + " " + action.type;
            if (action.succeeded()) {
                toLog += " succeeded.";
            } else if (action.failed()) {
                toLog += " failed.";
            } else {
                toLog += " interrupted.";
            }
            toLog += "\r\n";
        }
        return toLog;
    }

    /**
     * This method should be called after every step of the logic. It accepts the
     * result of the step - DMSMemoryVO and logs a message to the file if necessary.
     * So far we are not logging SEARCH messages as they are many.
     * @param memoryVO
     */
    public void update(DMSMemoryVO memoryVO) {
        // if there was a change in the atomic actions (one replaced by another -> old atomic action not null)
        // log it
        String toLog = getFormattedMessage(memoryVO);
        if (!toLog.equals("")) {
            try {
                outputFile.write(toLog.getBytes());
                toLog = "";
            } catch (IOException ex) {
                Logger.getLogger(DMSLogger.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        // don't log SEARCH messages
        // log the start of a new action and set oldAtomicAction
        if (memoryVO.getActualAtomicAction() != null && !memoryVO.getActualAtomicAction().equals(lastAtomicAction)
                && !memoryVO.getActualAtomicAction().type.name().contains("SEARCH_")) {
            toLog += TimeUtils.counterToDate(memoryVO.getThoughtTimeTicks()) + " ";
            toLog += memoryVO.getActualAtomicAction().type + " started.\r\n";
            lastAtomicAction = memoryVO.getActualAtomicAction();
        }
        // write to the file
        if (!toLog.equals("")) {
            try {
                outputFile.write(toLog.getBytes());
            } catch (IOException ex) {
                Logger.getLogger(DMSLogger.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
    }

    /**
     * Enables to write a certain message from the DMS to the file.
     * @param message
     */
    public void writeAMessage(String message) {
        try {
            if (!ready) {
                return;
            } else {
                this.outputFile.write(message.getBytes());
            }
        } catch (IOException ex) {
            Logger.getLogger(DMSLogger.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    /**
     * Called after the end of agent's life. Close the log file.
     * @see episodibot.Main.shutdownAgent()
     */
    @Override
    public void cleanUp() {
        try {
            outputFile.close();
        } catch (IOException ex) {
            Logger.getLogger(DMSLogger.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    @Override
    public void logToFile(String text) {
        writeAMessage(DateUtils.nowHours() + ":  " + text + "\r\n");
    }

    @Override
    public void logToFilePlainText(String text) {
        writeAMessage(text);
    }
}