package utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.logging.Logger;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.WindowConstants;

/**
 * Utils which helps with saving loading objects to files - serialization.
 *
 * @author Ondrej Burkert
 */
public final class FileSaveLoadUtils {

  /**
   * Serialization of an object to a file of specified path
   * @param obj
   * @param file - path to a file to save to
   * @param log
   */
  public static void saveObject(Object obj, String file, Logger log) {
    ObjectOutputStream s = null;
    try {
      FileOutputStream f = new FileOutputStream(file);
      s = new ObjectOutputStream(f);
      s.writeObject(obj);
      s.flush();
    } catch (IOException ex) {
      log.severe("Exception occured while saving: " + ex);
    } finally {
      try {
        s.close();
      } catch (IOException ex) {
        log.severe("Exception occured while saving: " + ex);
      }
    }
  }

  /**
   * Loading a serialized object from a file on specified path.
   * @param file - path to a file
   * @param log
   * @return
   */
  public static Object loadObject(String file, Logger log) {
    ObjectInputStream s = null;
    try {
      FileInputStream in = new FileInputStream(file);
      s = new ObjectInputStream(in);
      return s.readObject();
    } catch (ClassNotFoundException ex) {
      log.severe("Exception occured while loading: " + ex);
    } catch (IOException ex) {
      log.severe("Exception occured while loading: " + ex);
    } finally {
      try {
        s.close();
      } catch (IOException ex) {
        log.severe("Exception occured while loading: " + ex);
      }
    }
    return null;
  }

  /**
   * creates a filename in the shape: path0001 for a fileCounter equals to 1
   * @param path
   * @param fileCounter
   * @return
   */
  public static String fileCounterPrefix(String path, int dayCounter, int hourCounter) {
    String result = path + "_";
    String day = Integer.toString(dayCounter);
    for (int i = 2; i > day.length(); --i) {
      result += "0";
    }
    result += day + "_";
    String hour = Integer.toString(hourCounter);
    for (int i = 2; i > hour.length(); --i) {
      result += "0";
    }
    result += hour;
    return result;
  }

  /**
   * Invokes a dialog to get a path to a file.
   * 
   * @param dialogName - a description in the header of the dialog.
   * @param directory - an initial directory of the dialog (by default Documents and Settings).
   * @return choosen path or null (Storno)
   */
  public static String setPathFromDialog(String dialogName, String directory) {
    JFrame frame = new JFrame();
    frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
    JFileChooser fc = new JFileChooser(dialogName);
    fc.setDialogTitle(dialogName);
    fc.setCurrentDirectory(new File(directory));
    // Show open dialog; this method does not return until the dialog is closed
    fc.showOpenDialog(frame);
    if (fc.getSelectedFile() == null) {
      return null;
    }
    return fc.getSelectedFile().getAbsolutePath();
  }
}