package decisionMakingSystem;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.logging.Logger;

/**
 * Perceptive field is taking care about the actions and their link with items (satisfaction of affordances)
 * 
 * therefore it contains link to ThingManager and structures to hold currently perceived items
 * and actions, which are active and compete to be executed
 * 
 * @author Ondrej
 */
public class PerceptiveField {

  /** processes which are active and about to be executed */
  public final ArrayList<Action> processArea;
  /** items which were obtained by the cue posed on memory */
  public ArrayList<EItem> memoryItems = null;
  /** perceived items - those who made it through the attention filter*/
  public final ArrayList<EItem> perceivedItems;
  /**
   * structure which gathers all currently visible items and handles messages from GB
   *  e.g. addition and deletion of an item
   */
  public ThingsManager allThings = null;
  /** global log to log anything into:) */
  public Logger log = null;
  /** actual value of stress - dependent on how much pressure is there on agent (how many actions to do)*/
  public double stress = 0;
  /** necessary for accessing memory */
  public DecisionModuleImpl dModule = null;

  public PerceptiveField(Logger log, ThingsManager things, DecisionModuleImpl dModule) {
    this.log = log;
    memoryItems = new ArrayList<EItem>();
    processArea = new ArrayList<Action>();
    perceivedItems = new ArrayList<EItem>();
    allThings = things;
    this.dModule = dModule;
  }

  /**
   * Adds action to the processArea, disable grandfather if there
   * @param action
   */
  public void addAction(Action action) {
    // check if it is not already there
    synchronized (processArea) {
      for (Action act : processArea) {
        if (act.equals(action)) // it is there, so I am not adding it
        {
          return;
        }
      }
      processArea.add(action);
    }
    action.fadeout = GlobalParameters.FADEOUT_FOR_ACTIONS;
    // adding new action -> disable grandfather
    if (action.intention != null) {
      action.activity = action.intention.getActivity();
      if (action.intention.getAncestorAction() != null) {
        action.intention.getAncestorAction().enabled = false;
      }
    }
  }

  /**
   * checks if the item is assigned to action
   * @param item
   * @param action
   * @return true if item is assigned to the action
   */
  public boolean isThingAssignedToAction(EItem item, Action action) {
    if (action == null || action.satisfyingItems == null) {
      return false;
    }
    for (AffordanceType af : action.satisfyingItems.keySet()) {
      if (action.satisfyingItems.get(af) != null) {

       // log.warning("First: " + action.satisfyingItems.get(af).toString() + "type " + action.satisfyingItems.get(af).type);
       // log.warning("Second: " + item.toString()+ "type " + item.type);

        if (action.satisfyingItems.get(af).equals(item)) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * checks if the affordance is satisfied by something
   * @param affordance
   * @param action
   * @return true if the affordance of the action has some item assigned - so its satisfied
   */
  public boolean isAffordanceSatisfied(AffordanceType affordance, Action action) {
    if (action == null || action.satisfyingItems == null) {
      return false;
    }
    if (!action.satisfyingItems.containsKey(affordance)) {
      return false;
    }
    if (action.satisfyingItems.get(affordance) != null) {
      return true;
    }
    return false;
  }

  /**
   * choose action from process field - so it is not the same as choosing action in the decision module
   */
  public Action chooseAction(int time) {
    synchronized (processArea) {
      if (this.processArea.isEmpty()) {
        return null;
      }
    }
    Action chosenAction = findMaxActivity();
    if (chosenAction == null) {
      return null;
    }
    chosenAction.time += 1;
    chosenAction.attention = Concentration.concentrationOnAction(chosenAction, time);
    return chosenAction;
  }

  /**
   * it adds new item to the perceptive field if it can fits in
   * - adds to memory - seeing item increases if it is already there
   * @param item
   */
  public void addSource(EItem item) {
    synchronized (perceivedItems) {
      // if it is not already perceived
      if (!perceivedItems.contains(item)) {
        if (perceivedItems.size() > AgentParameters.maxNumberOfPerceivedItems) {
          for (int i = 0; i < perceivedItems.size();) {
            if (isThingAssignedToAction(item)) {
              i++;
            } else {
              perceivedItems.remove(i);
            }
          }
        }
        this.addItemToMemory(item, 1);
        this.perceivedItems.add(item);
      }
    }
    item.setFadeout(GlobalParameters.FADEOUT_FOR_ITEMS);
  }

  /**
   * safely remove source from perceivedItems
   * so it removes itself from affordances
   * @param item
   */
  public void removeSource(EItem item) {
    boolean remove = false;
    AffordanceType keyOfRemoved = null;
    Action action = null;
    synchronized (processArea) {
      for (Action act : this.processArea) {
        if (this.isThingAssignedToAction(item, act)) {
          for (AffordanceType aff : act.satisfyingItems.keySet()) {
            for (Affordance itemAff : item.getAffordances()) {
              if (itemAff == null || aff == null) {
                continue;
              }
              if (aff.equals(itemAff.type)) { // direct remove will cause concurrent modification exception
                remove = true;
                keyOfRemoved = aff;
                action = act;
              }
            }
          }
        }
      }
    }
    if (remove) {
      action.satisfyingItems.remove(keyOfRemoved);
    }
    synchronized (perceivedItems) {
      this.perceivedItems.remove(item);
    }
  }

  /**
   * removes source with no assigned action or with assigned action of minimal activity
   */
  public void removeSource() {
    EItem toRemove = null;
    int minAct = 101; // maximal activity is a 100
    int itemMin = 101; // minimal activity for each item
    synchronized (perceivedItems) {
      for (EItem item : this.perceivedItems) {

        itemMin = 101;
        synchronized (processArea) {
          for (Action act : this.processArea) {
            if (this.isThingAssignedToAction(item, act)) {
              if (itemMin > act.activity) {
                itemMin = act.activity;
              }
            }
          }
        }
        if (itemMin == 101) // not assigned to any action
        {
          itemMin = 0;
        }
        if (itemMin < minAct) {
          minAct = itemMin;
          toRemove = item;
        }
      }
    }
    this.removeSource(toRemove);
  }

  /**
   * @return action with maximal activity among actions in process area
   */
  public Action findMaxActivity() {
    int maxActivity = 0;
    Action result = null;
    synchronized (processArea) {
      for (Action act : this.processArea) {
        if (act.activity > maxActivity && act.enabled) {
          maxActivity = act.activity;
          result = act;
        }
      }
    }
    return result;
  }

  /**
   * updates activities of enabled preactive actions - actions in process area
   * @param time
   */
  public void updateActivities(int time) {
    int intentionActivity = 0;
    boolean changed = false;
    synchronized (processArea) {
      for (Action act : this.processArea) {
        intentionActivity = 0;
        changed = false;
        if (!act.enabled) // not enabled actions are not processed
        {
          continue;
        }
        if (act.intention != null) {
          intentionActivity = act.intention.getActivity();
        } else { // action is there because of an interesting item
          for (AffordanceType aff : act.satisfyingItems.keySet()) {
            if (act.satisfyingItems.get(aff) != null) {
              if (act.satisfyingItems.get(aff).getAttractivity() > intentionActivity) {
                intentionActivity = act.satisfyingItems.get(aff).getAttractivity();
                changed = true;
              }
            }
          }
          if (!changed) // e.g. no satisfied item in the satisfying items
          {
            intentionActivity = act.activity;
          }
        }
        act.attention = Concentration.concentrationOnAction(act, time);
        act.intentionActivity = intentionActivity; // why?
        act.activity = Math.max(act.attention, act.intentionActivity);
        act.activity = Math.max(act.activity, 0);
      }
    }
  }

  /**
   * updates values of fadeout in all items - if the item is not perceived, it fades out of
   * agent perception -> removed from perceivedItems
   * same thing for processes - if process doesn't have activity big enough to get to execution, its fading out
   */
  public void updateFadeout() {
    boolean assigned = false;
    int index = 0;
    EItem item = null;
    // if item not assigned to anything
    synchronized (perceivedItems) {
      for (; index < this.perceivedItems.size();) {

        assigned = false;
        item = this.perceivedItems.get(index);
        for (Action act : this.processArea) {
          if (this.isThingAssignedToAction(item, act)) {
            assigned = true;
          }
        }
        if (!assigned) {
          item.setFadeout(item.getFadeout() - 1);
        }
        if (item.getFadeout() < 0) {
          this.perceivedItems.remove(item);
        } else // not removing -> increase index
        {
          index++;
        }
      }
    }
    // go through actions
    Action act = null;
    synchronized (processArea) {
      for (index = 0; index < this.processArea.size();) {
        act = this.processArea.get(index);
        // only spontaneous action or actions with negative activity got fadeout
        if (act.intention == null || act.activity <= 0) // so it will never win
        {
          act.fadeout -= 1;
        }
        if (act.fadeout < 0) { // remove action, its descendants, maybe even its predecessors
          this.removeFadeoutedAction(act);
          act.state = ActionStates.FADOUT;
        } else {
          index++;
        } // am not removing so I can increase index
      }
    }
  }

  /**
   * removes actions of the intention and call recursion, so it should erase whole tree when it comes to it
   * @param temp
   */
  public void removeActionsOfIntention(Intention temp) {
    int index = 0;
    Action action = null;
    synchronized (processArea) {
      for (; index < processArea.size();) {
        action = processArea.get(index);
        if (action.intention == null) { // spontaneous action
          index++;
          continue;
        }
        if (action.intention.equals(temp)) {
          processArea.remove(index);
          this.dModule.removeActionAndDescendantIntentions(action); // recursive call so it will erase whole tree
        } else {
          index++;
        }
      }
    }
  }

  private void addItemToMemory(EItem item, int i) {

      this.dModule.agent.itemMemory.addItemToMemory(item, dModule.counter, i);

  }
  /**
   * returns item of provided affordance, if it perceives any
   * @param affType
   * @return
   */
  public EItem getPerceivedItemOfAffordance(AffordanceType affType) {
    synchronized (perceivedItems) {
      for (EItem item : this.perceivedItems) {
        for (Affordance aff : item.getAffordances()) {
          if (aff.type.equals(affType)) {
            return item;
          }
        }
      }
    }
    return null;
  }

  public void clearProcessArea() {
    synchronized (processArea) {
      processArea.clear();
    }
  }

  /**
   * returns true if there is a single action using the tested item
   * @param item
   * @return
   */
  boolean isThingAssignedToAction(EItem item) {
    synchronized (processArea) {
      for (Action action : processArea) {
        if (action.satisfyingItems == null) {
          continue;
        }
        for (AffordanceType af : action.satisfyingItems.keySet()) {
          if (action.satisfyingItems.get(af) != null) {
            if (action.satisfyingItems.get(af).equals(item)) {
              return true;
            }
          }
        }
      }
    }
    return false;
  }

  void setDMS(DecisionModuleImpl dms) {
    this.dModule = dms;
  }

  void setThings(ThingsManager things) {
    this.allThings = things;
  }

  /**
   * adds item to item memory if there is a memory to add to
   * @param item
   * @param i
   *//*
  private void addItemToMemory(EItem item, int i) {
    if (dModule.hasMemory()) {
      this.dModule.getMemory().addItemToMemory(item, dModule.counter, i);
    } // adds to the memory
  }*/

  private void removeFadeoutedAction(Action action) {
    this.dModule.removeActionAndDescendantIntentions(action);
  }

  /**
   * removes action from preactive actions, plus if it has some items, decreases their attractivities
   * @param action
   */
  public void removeAction(Action action) {
    synchronized (processArea) {
      this.processArea.remove(action);
    }
    if (action.satisfyingItems == null) {
      return;
    }
    for (EItem item : action.satisfyingItems.values()) {
      if (item != null) {
        item.decreaseOfAttractivity = item.getBasicAttractivity();
      }
    }

  }

  /**
   * update attractivity of all items in the perceptive field
   */
  public void updateAttractivity() {
    EItem item = null;
    int tempAttr = 0;
    boolean assigned = false;
    int affAttractivity = 0;
    if (allThings.visibleItems.isEmpty()) {
      return;
    }
    synchronized (allThings.visibleItems) {
      for (Long itemID : allThings.visibleItems.keySet()) {
        affAttractivity = 0;
        item = allThings.visibleItems.get(itemID);
        for (Affordance aff : item.getAffordances()) {
          if (perceivedItems.contains(item)) {
            assigned = false;
            // if it is assigned to something, ok, else set affordance attractivity to 0
            synchronized (processArea) {
              for (Action act : this.processArea) {
                if (act.satisfyingItems == null) {
                  continue;
                }
                if (act.satisfyingItems.containsKey(aff.type)) {
                  assigned = true;
                }
              }
              if (!assigned) {
                aff.attractivity = 0;
              }
            }
          }
          // search the most attractive affordance of the item
          if (affAttractivity < aff.attractivity) {
            affAttractivity = aff.attractivity;
          }
        }
        if (item.decreaseOfAttractivity > 0 && !perceivedItems.contains(item)) {
          item.decreaseOfAttractivity -= 1;
        }
        tempAttr = item.getBasicAttractivity() - item.decreaseOfAttractivity + affAttractivity;
        if (tempAttr > 100) // this is not good:) attractivity over shouldn't be!!!
        {
          tempAttr = 100;
        }
        item.setAttractivity(tempAttr);
      }
    }
  }

  /**
   * increases attractivity to all visible (NOT only percieved) sources of the action
   * @param action
   */
  public void increaseAttractivityToAllSources(Action action) {
    if (action.intention != null && action.activity > 0 && action.satisfyingItems != null) {// not a spontaneous action
      ArrayList<EItem> items = null;
      for (AffordanceType aff : action.satisfyingItems.keySet()) {
        items = this.allThings.getItemOfAffordance(aff);
        for (EItem item : items) {
          item.changeAffordaceActivity(action.activity, aff);
        }
      }
    }
    this.updateAttractivity();
  }

  /**
   * counts the stress level of agent - based on some psychological study
   * approximated by the hyperbolic tangens of the sum of activities of actions in the processArea
   * @return
   */
  private double stressLevel() {
    double result = 0, temp = 0;
    synchronized (processArea) {
      for (Action act : processArea) {
        if (act.enabled) {
          temp = temp + act.activity / 100;
        }
      }
    }
    result = (1 - Math.exp(temp)) / (1 + Math.exp(temp));
    this.dModule.agentParameters.updateCoeficients(result);
    return result;
  }

  public void updateStress() {
    stress = stressLevel();
  }

  public String printPerceivedItems() {
    String result = "Perceived Items. Number of items: " + perceivedItems.size() + "\n";
    synchronized (perceivedItems) {
      for (EItem item : this.perceivedItems) {
        result += item.name + " attractivity: " + item.getAttractivity();
        result += " Affordances: ";
        for (Affordance affo : item.getAffordances()) {
          result += affo.type + " " + affo.counter + " " + affo.attractivity + "\t";
        }
        result += "\n";
      }
    }
    return result;
  }

  /**
   *
   * @param actualIntention
   * @return true if provided intention has some child action in the processArea
   */
  boolean hasDerivedAction(Intention actualIntention) {
    synchronized (processArea) {
      for (Action act : processArea) {
        if (act.intention != null && act.intention.equals(actualIntention)) {
          return true;
        }
      }
    }
    return false;
  }

  public void satisfyAffordance(Action action, AffordanceType af, EItem item) {
    if (action.satisfyingItems.containsKey(af) && action.satisfyingItems.get(af) == null) {
      log.fine("Satisfying affordance: " + af + " by item: " + item);
      action.satisfyingItems.put(af, item);
      this.addItemToMemory(item, 0);
    }
  }
}
