/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package decisionMakingSystem;

import decisionMakingSystem.Affordance;
import decisionMakingSystem.AffordanceType;
import decisionMakingSystem.AgentParameters;
import decisionMakingSystem.EItem;
import decisionMakingSystem.ItemCathegory;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.logging.Logger;

/**
 * This module is rather simple. It works over a HashMap which uses AffordanceType as a key and
 * contains lists of ItemRecords. The memory receives notifications about every perceived item, place,
 * player and stores them to the presented structure. It stores items as ItemRecord. ItemRecord
 * takes from item only  its location, affordances and cathegory and has some other atributes which
 * the episodic information. Those are seen, missed, found and time. As we can see from their names
 * they say how many times agent saw the item, looked for it successfuly/unsuccessfully and when the
 * last change happened. This information is then used to calculate the truthfullness � e.g. How much
 * does the agent think that the item/place/player will be at the stored location. The probability is
 * counted according to the following formula:
.
 * it is used to respond to the memory querry invoked by SearchMemory from Want intention
 *
 * @author Ondrej
 */
public class ItemMemory implements Serializable {

  /**
   * Storage of the information about the item - each affordance type is represented by
   * a list of itemRecords of items, which has that affordance
   */
  private HashMap<AffordanceType, ArrayList<ItemRecord>> itemInfo = null;
  /** */
  transient private Logger log;

  public ItemMemory(Logger log) {
    itemInfo = new HashMap<AffordanceType, ArrayList<ItemRecord>>();
    this.log = log;
  }

  /**
   * adds new item to the ItemMemory or updates its record
   * @param item - new item to add
   * @param time - when we are adding - number of steps of the logic
   * @param state 0 - use, 1 - seen
   */
  public void addItemToMemory(EItem item, int time, int state) {
    ItemRecord newRecord = new ItemRecord(item, time);
    // not processed by memory
    if (item.cathegory.equals(ItemCathegory.INVENTORY)) {
      return;
    }
    for (Affordance aff : item.getAffordances()) {
      if (!itemInfo.containsKey(aff.type)) {
        itemInfo.put(aff.type, new ArrayList<ItemRecord>());
      }
      // not already present, add it
      if (!itemInfo.get(aff.type).contains(newRecord)) {
        itemInfo.get(aff.type).add(0, newRecord);
      } else // already there, rise seen!!!
      {
        for (ItemRecord record : itemInfo.get(aff.type)) {
          if (record.equals(newRecord)) {
            newRecord = record;
          }
        }
      }
      // update seen, found according to the state -> means if the item was added when spoted or used
      if (state == 0) {
        newRecord.found(time);
      }
      if (state == 1) {
        newRecord.seen(time);
      }
    }
  }

  /**
   * just reports the state of the ItemMemory - to give a user some idea now many things are stored etc.
   */
  public void dailyReport() {
    String logResult = "Number of stored items: " + this.itemInfo.values().size() + "\n";
    ArrayList<ItemRecord> list = null;
    for (AffordanceType affordance : this.itemInfo.keySet()) {
      logResult += "List for an affordance: " + affordance + "\n";
      list = this.itemInfo.get(affordance);
      for (ItemRecord record : list) {
        logResult += record.toString() + " \t";
      }
      logResult += "\n";
    }
    this.log.config("Daily report from item memory: " + logResult);
  }

  /**
   * Note it return at least the most probable even thought it will have like 2%, as it is better to try that than go random
   * @param aff
   * @param counter - current time when it is called
   * @return list of item records with probability higher than the biasProbability or the highest probable record
   */
  public ArrayList<ItemRecord> getLocationOfAffordances(AffordanceType aff, int counter) {
    ArrayList<ItemRecord> result = new ArrayList<ItemRecord>();
    ItemRecord best = null;
    double max = 0;
    if (itemInfo == null || itemInfo.get(aff) == null) {
      return null;
    }
    String logReport = "Considered records: \n";
    for (ItemRecord record : itemInfo.get(aff)) {
      record.updateProbability(counter);
      logReport += record + " \n";
      if (record.probability > AgentParameters.biasProbability) {
        result.add(record);
      }
      if (record.probability > max) {
        max = record.probability;
        best = record;
      }
    }
    log.fine(logReport);
    if (result.isEmpty()) // will return at least the best found - so it will search on the places it seen something at (at least on one)
    {
      result.add(best);
    }
    return result;
  }

  public void oblivion(int counter) {
    for (AffordanceType aff : itemInfo.keySet()) {
      for (int i = 0; i < itemInfo.get(aff).size();) {
        ItemRecord record = itemInfo.get(aff).get(i);
        record.dailyUpdate();
        if (record.obsolete(counter)) {
          itemInfo.get(aff).remove(i);
        } else {
          i++;
        }
      }
    }
  }

  void setLoggerAfterLoad(Logger log) {
    this.log = log;
  }
}
