package decisionMakingSystem;


import utils.DeepCopyCreator;
import utils.Interval;
import atomicActions.AtomicAction;

import bot.Bot;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Random;
import javax.xml.bind.annotation.*;

/**
 * Intention is one of the crucial classes of this decision making model
 * it represents an "OR" node of the tree, so it contains a set of actions
 * by which it can be satisfied (it suffices to succeed with one of them)
 * <br>
 * 
 * @author Ondrej
 */
@XmlRootElement
public class Intention implements Comparable, Serializable {

  /** activity of the intention - depending on it an intention could be chosen to become the active intention */
  private int activity = 0;
  /** intention can be enabled - ready to execute or suspended - when some descendant is in progress */
  private boolean enabled = true;
  /** intention state defines whether intention is completed, failed, etc. */
  private IntentionStates realized = IntentionStates.NO;
  /** possibilities (pointers to next level of the tree) */
  private ArrayList<Action> actions = null;
  /** list of not yet tried action */
  private ArrayList<Action> notTriedActions = null;
  /** chosen action - e.g. pointer to next level */
  private Action chosenAction = null;
  /** activationIntervals are containing the function of basic activity of intention over the time. Intervals of inhibition and activation of the intention.*/
  private ArrayList<Interval> activationIntervals = null;
  /** dropoutInterval is used for dropping out the root intention when it is satisfied */
  private Interval dropoutInterval = null;
  /** pointer back in the tree structure */
  private Action ancestorAction = null;
  /** intention name */
  private String name = null;
  /** max lenght of the intention e.g. maximum of the timeLimits of the actions */
  private int timeLimit = 0;
  /** how long did it take to finish it */ 
  private int duration = 0;
  /** how long it should stay in the memory */
  private int memoryDuration = 0;
  /** fadeout for intention */
  private int fadeout = 0;
  private static Random rnd = new Random();

  /** specified in XML file - used to determine when to delete node from episodic memory  */
  public int attractivity = 0;
  /** specified in XML file - enables definition of common subtrees */
  public boolean extend = false;
  public boolean extension = false;

  public Intention() {
    fadeout = GlobalParameters.FADEOUT_FOR_INTENTIONS;
  }

  /**
   * makes hard copy of whole AND-OR tree via Serializable, as decision module and agent are transient
   * it sets those manually after acquiring hard copy
   * @param agent - link to agent
   * @param dModule - link to decision module
   * @return hard copy of the tree
   */
  public Intention cloneBySerialize(Bot agent, DecisionModuleImpl dModule) {
    Intention result = (Intention) DeepCopyCreator.copy(this);
    ArrayList<Action> actionsWithAtomicActions = BasicIntentionLoader.getActionsWithAtomicActions(result);
    for (Action action : actionsWithAtomicActions) {
      for (AtomicAction aAction : action.atomicActions) {
        aAction.agent = agent;
        aAction.dModule = dModule;
      }
    }
    return result;
  }

  @Override
  public String toString() {
    String result = "INTENTION: " + name + " | ";
    result += "Activity: " + this.activity + " | " + "Duration: " + this.duration + " | ";
    result += "Enabled: " + this.enabled + " | ";
    result += "Memory duration: " + this.memoryDuration + " | " + "Realized: " + this.realized + " | ";
    if (ancestorAction != null) {
      result += "Parent action name: " + this.ancestorAction.name + " | ";
    } else {
      result += "Root intention! | ";
    }
    if (actions != null && !actions.isEmpty()) {
      result += "Actions: ";
      for (Action i : actions) {
        result += i.name + " ";
      }
      if (this.notTriedActions != null) {
        result += " Not tried: " + this.notTriedActions.size() + " | ";
      } else {
        result += " Not tried: " + this.actions.size() + " | ";
      }
    }
    if (this.activationIntervals != null) {
      result += " ActivationIntervals: ";
      for (Interval interval : activationIntervals) {
        result += interval + " ";
      }
    }
    if (chosenAction != null) {
      result += "Chosen action: " + chosenAction.name;
    }
    result += " Fadeout: " + fadeout;
    return result + "\n";
  }

  /**
   * if it should be increased in the provided time
   * @param time
   * @return
   */
  public boolean increased(int time) {
    for (Interval interval : this.activationIntervals) {
      if (interval.isInInterval(time)) {
        return true;
      }
    }
    return false;
  }

  /**
   * updates activity of the intention according to the habituation/activation function
   * counts with dropout first - no need to restart it - it will be overriden later anyway
   */
  public void updateActivity(int time) {
    if (this.dropoutInterval != null && this.dropoutInterval.isInInterval(time)) {
      this.activity = 0;
      return;
    } else {
      this.dropoutInterval = null;
    }
    this.activity = 0;
    for (Interval i : activationIntervals) {
      if (i.isInInterval(time)) {
        if (i.valueInTime(time) > this.activity) {
          this.activity = i.valueInTime(time);
        }
      }
    }
  }

  /**
   * sets attractivity of all provided items which can satisfy any affordance of actions in the intention
   * @param things
   */
  public void increaseActivityToAllSources(ArrayList<EItem> things) {
    for (Action act : this.actions) {
      for (AffordanceType aff : act.satisfyingItems.keySet()) {
        if (act.satisfyingItems.get(aff) != null) {
          act.satisfyingItems.get(aff).setAttractivity(this.activity);
        }
      }
    }

    /** recount attractivity of all items - new attractivity is the maximum of activities of its affordances */
    int temp = 0;
    for (EItem it : things) {
      for (Affordance aff : it.getAffordances()) {
        if (aff.attractivity > temp) {
          temp = aff.attractivity;
        }
      }
      it.setAttractivity(temp + it.getBasicAttractivity());
    }

  }

  /**
   * @return root intention of presented intention - null if it is already in root
   */
  public Intention rootIntention() {
    Intention result = this;
    boolean go = true;
    while (go) { // and here I got my neverending cycle
      if (result.ancestorAction == null || result.ancestorAction.intention == null) {
        break;
      }
      result = result.ancestorAction.intention;
    }
    return result;
  }

  /**
   * @return true if intention is root of some AND-OR tree
   */
  public boolean isRoot() {
    if (this.ancestorAction == null) {
      return true;
    }
    return false;
  }

  /**
   * picks up an action to do next to try to satisfy the intention.
   * as intention can be satisfied by any of its action, it chooses it randomly
   */
  public Action chooseAction() {
    // if intention failed there is no point in choosing
    if (this.getRealized().equals(IntentionStates.FAILED)) {
      return null;
    }
    // correct random choice of next action - thanks to seed it should be realy random
    rnd.setSeed(System.currentTimeMillis());
    // nothing chosen yet
    if (this.chosenAction == null) { // pick first by random from possible actions
      // am picking up from notTriedActions
      // if it is null (not yet initialized) I initialize it by all available actions
      if (this.notTriedActions == null) {
        this.notTriedActions = (ArrayList) this.actions.clone();
      }
      if (this.notTriedActions.size() > 0) {
        this.chosenAction = this.notTriedActions.get(rnd.nextInt(this.notTriedActions.size()));
      } else // there are no actions left
      {
        this.chosenAction = null;
      }
    }
    if (chosenAction != null) {
      chosenAction.intention = this;
      chosenAction.activity = this.activity;
    }
    return chosenAction;
  }

  @Override
  public int compareTo(Object o) {
    Intention int2 = (Intention) o;
    if (int2.activity == this.activity) {
      return 0;
    }
    if (int2.activity > this.activity) {
      return -1;
    }
    return 1;
  }

  @Override
  public boolean equals(Object aThat) {
    if (this == aThat) {
      return true;
    }
    if (!(aThat instanceof Intention)) {
      return false;
    }
    Intention that = (Intention) aThat;
    if (this.enabled != that.enabled) {
      return false;
    }
    if (this.activity != that.activity) {
      return false;
    }
    if (!this.realized.equals(that.realized)) {
      return false;
    }

    if ((this.ancestorAction == null) && (that.ancestorAction == null)); else if ((this.ancestorAction == null) || (this.ancestorAction == null)) {
      return false;
    } else if (!this.ancestorAction.equals(that.ancestorAction)) {
      return false;
    }

    if (!this.name.equals(that.name)) {
      return false;
    }
    return true;
  }

  @Override
  public int hashCode() {
    int hash = 7;
    hash = 3 * hash + (this.actions != null ? this.actions.hashCode() : 0);
    hash = 3 * hash + (this.ancestorAction != null ? this.ancestorAction.hashCode() : 0);
    hash = 3 * hash + (this.name != null ? this.name.hashCode() : 0);
    return hash;
  }

  public int getActivity() {
    return activity;
  }

  public void setActivity(int activity) {
    if (activity <= 0) {
      this.activity = 0;
    } else {
      this.activity = activity;
    }
  }

  public boolean isEnabled() {
    return enabled;
  }

  public void setEnabled(boolean enabled) {
    this.enabled = enabled;
  }

  public IntentionStates getRealized() {
    return realized;
  }

  public void setRealized(IntentionStates realized) {
    this.realized = realized;
  }

  public Action getChosenAction() {
    return chosenAction;
  }

  public void setChosenAction(Action chosenAction) {
    this.chosenAction = chosenAction;
  }

  public ArrayList<Action> getActions() {
    return actions;
  }

  public void setActions(ArrayList<Action> actions) {
    this.actions = actions;
  }

  public ArrayList<Action> getNotTriedActions() {
    return notTriedActions;
  }

  public void setNotTriedActions(ArrayList<Action> oldProcesses) {
    this.notTriedActions = oldProcesses;
  }

  public ArrayList<Interval> getActivationIntervals() {
    return activationIntervals;
  }

  public void setActivationIntervals(ArrayList<Interval> activationIntervals) {
    this.activationIntervals = activationIntervals;
  }

  public Action getAncestorAction() {
    return ancestorAction;
  }

  public void setAncestorAction(Action ancestorAction) {
    this.ancestorAction = ancestorAction;
  }

  public String getName() {
    return name;
  }

  public void setName(String name) {
    this.name = name;
  }

  public int getDuration() {
    return duration;
  }

  public void setDuration(int duration) {
    this.duration = duration;
  }

  public void increaseDuration() {
    this.duration += 1;
  }

  public int getMemoryDuration() {
    return memoryDuration;
  }

  public void setMemoryDuration(int memoryDuration) {
    this.memoryDuration = memoryDuration;
  }

  public void setFadeout(int fadeout) {
    this.fadeout = fadeout;
  }

  public int getFadeout() {
    return this.fadeout;
  }

  public void decreaseFadeout() {
    this.fadeout--;
  }

  public int getTimeLimit() {
    if (timeLimit != 0) {
      return timeLimit;
    }
    int max = 0;
    for (Action action : this.actions) {
      if (action.timeLimit > max) {
        max = action.timeLimit;
      }
    }
    return max;
  }

  /**
   * drop out happens when an intention finishes successfuly => so it will not be active in its activation interval
   * -> drops out
   * @param time
   */
  public void dropOut(int time) {
    for (Interval interval : activationIntervals) {
      if (interval.isInInterval(time)) {
        this.dropoutInterval = new Interval(time, interval.getRightSide(), 0);
      }
    }
  }

  /**
   * @return an example instance of Intention - useful when generating first XML entry of some structure of intentions and actions
   * to see how it looks like
   */
  public static Intention exampleOfIntention() {
    Intention result = new Intention();
    HashSet<AffordanceType> affordances = new HashSet<AffordanceType>();
    affordances.add(AffordanceType.TO_CLEAN);
    Interval int1 = new Interval(100, 200, 20);
    Interval int2 = new Interval(500, 700, 50);
    ArrayList<Interval> intervals = new ArrayList<Interval>();
    intervals.add(int2);
    intervals.add(int1);
    result.setActivationIntervals(intervals);
    result.setActivity(10);
    result.setDuration(100);
    result.setMemoryDuration(300);
    result.setRealized(IntentionStates.NO);
    result.setActions(new ArrayList<Action>());
    result.setName("example intention");

    return result;
  }

  /**
   * this procedure tries to restrict the time limits, so the subactions of the intention would fail/end earlier then normaly
   *
   * so far it restrains the interval to 2/3 of the given restrain for each subaction -> though with a bit more effort it can do it exactly
   * like count the time before, make it proportional and set it according to it, but as it is not crucial, I made it this way
   *
   * @param intervalSize - the maximum timeLimit possible -> already in the steps of the logic
   */
  public void restrictTimeLimits(int intervalSize) {
    if (this.getTimeLimit() < intervalSize) {
      return;
    }
    // so I go through the actions and if an action has timeLimit greater than intervalSize
    // I will first restrain the subIntentions, then restrain the timeLimit of the action itself
    // so far I restrain it to intervaSize*2/3, as there is no guarantee how it will look like...
    for (Action action : this.actions) {
      if (action.timeLimit > intervalSize) {
        if (action.intentions != null && !action.intentions.isEmpty()) {
          for (Intention intention : action.intentions) {
            intention.restrictTimeLimits(intervalSize);
          }
        } else {
          action.timeLimit = intervalSize * 2 / 3; // just to play it safe, we give it only 2/3 of the time, so it won't interfere that much
        }
      }
    }
    this.timeLimit = intervalSize;
  }
}


