package decisionMakingSystem;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Vector;
import javax.swing.GroupLayout;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ToolTipManager;
import javax.swing.table.DefaultTableModel;
import utils.Interval;

/**
 * Basically just a copy of DMSModuleViewer.
 * @author Ondrej Burkert
 */
public class DMSModuleViewerPane extends JPanel {
      /** pointer to the DMS */
    private DecisionModuleImpl module = null;

    /** tables, models, scroll panels */
    private JTable intArea = null;
    private DefaultTableModel intAreaModel = null;
    private JScrollPane intAreaSP = null;
    private JTable actArea = null;
    private DefaultTableModel actAreaModel = null;
    private JScrollPane actAreaSP = null;
    private JTable items = null;
    private DefaultTableModel itemsModel = null;
    private JScrollPane itemsSP = null;
    private JTable dayPlan = null;
    private DefaultTableModel dayPlanModel = null;
    private JScrollPane dayPlanSP = null;
    /** names of headers */
    private Vector intAreaColNames = new Vector();
    private Vector actAreaColNames = new Vector();
    private Vector dayPlanColNames = new Vector();
    private Vector itemsColNames = new Vector();
    /** info panel with some additional info from DMS */
    private JPanel infoPanel = null;

    public DMSModuleViewerPane() {}

    public DMSModuleViewerPane(DecisionModuleImpl module) {
        super();
        this.module = module;
        this.setSize(640, 480);
        this.setVisible(true);
        initializeTables();
//        initializeTableModels();
        updateTables();
        initializeInformationPanel();
        initComponents();
    }
    /**
     * This method has to be called from agent for instance, or DMS if you want to display
     * up to date information.
     */
    public void update() {
        updateTables();
        this.repaint();
        infoPanel.repaint();
    }

    /**
     * Layout manager settings.
     */
    private void initComponents() {
        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        // infoPanelLayout
        GroupLayout infoPanelLayout = new GroupLayout(infoPanel);
        infoPanel.setLayout(infoPanelLayout);
        infoPanelLayout.setHorizontalGroup(
                infoPanelLayout.createParallelGroup(GroupLayout.Alignment.LEADING)
            .addGap(0, 300, Short.MAX_VALUE)
        );
        infoPanelLayout.setVerticalGroup(
                infoPanelLayout.createParallelGroup(GroupLayout.Alignment.LEADING)
            .addGap(0, 300, Short.MAX_VALUE)
        );
        // global layout
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(10)
            .addGroup(layout.createSequentialGroup()
                .addComponent(intAreaSP, javax.swing.GroupLayout.PREFERRED_SIZE, 250, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(actAreaSP, javax.swing.GroupLayout.DEFAULT_SIZE, 250, Short.MAX_VALUE)
                .addContainerGap())
            .addGroup(layout.createSequentialGroup()
                .addComponent(dayPlanSP, javax.swing.GroupLayout.PREFERRED_SIZE, 250, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(itemsSP, javax.swing.GroupLayout.DEFAULT_SIZE, 250, Short.MAX_VALUE)
                .addContainerGap())
            .addGroup(layout.createSequentialGroup()
                .addComponent(infoPanel, javax.swing.GroupLayout.PREFERRED_SIZE, 250, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap())
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(10)
            .addGroup(layout.createSequentialGroup()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
                    .addComponent(intAreaSP, javax.swing.GroupLayout.Alignment.LEADING, 0, 0, Short.MAX_VALUE)
                    .addGap(10)
                    .addComponent(actAreaSP, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, 200, Short.MAX_VALUE))
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
                    .addComponent(dayPlanSP, javax.swing.GroupLayout.Alignment.LEADING, 0, 0, Short.MAX_VALUE)
                    .addGap(10)
                    .addComponent(itemsSP, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, 200, Short.MAX_VALUE))
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
                    .addComponent(infoPanel, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, 200, Short.MAX_VALUE))
                )
        );
     }
     /**
      * Initialization of information panel - creation.
      * @see DMSModuleViewerInfo
      */
    private void initializeInformationPanel() {
        infoPanel = new DMSModuleViewerInfo(module);
        this.add(infoPanel);
    }
    /**
     * Initialization of tabels - headers, ScrollPanes, table models.
     */
    private void initializeTables() {
        int firstColumnWidth = 150;
        int secondColumnWidth = 25;
        int thirdColumnWidth = 25;
        // intention area table
        intAreaColNames.add("Name");
        intAreaColNames.add("Act");
        intAreaColNames.add("Parent");
        intAreaModel = new DefaultTableModel(null, intAreaColNames);
        intArea = new JTable(intAreaModel);
        intArea.setAutoCreateRowSorter(true);
        intArea.getColumnModel().getColumn(0).setPreferredWidth(firstColumnWidth);
        intArea.getColumnModel().getColumn(1).setPreferredWidth(secondColumnWidth);
        intArea.getColumnModel().getColumn(2).setPreferredWidth(thirdColumnWidth);
        intAreaSP = new JScrollPane();
        intAreaSP.setViewportView(intArea);
        intAreaSP.setBounds(0, 0, 400, 250);
        ToolTipManager.sharedInstance().unregisterComponent(intArea);
        ToolTipManager.sharedInstance().unregisterComponent(intArea.getTableHeader());
        // process area table
        actAreaColNames.add("Name");
        actAreaColNames.add("Act");
        actAreaColNames.add("Activity");
        actAreaModel = new DefaultTableModel(null, actAreaColNames);
        actArea = new JTable(actAreaModel);
        actArea.setAutoCreateRowSorter(true);
        actArea.getColumnModel().getColumn(0).setPreferredWidth(firstColumnWidth);
        actArea.getColumnModel().getColumn(1).setPreferredWidth(secondColumnWidth);
        actArea.getColumnModel().getColumn(2).setPreferredWidth(thirdColumnWidth);
        actAreaSP = new JScrollPane();
        actAreaSP.setViewportView(actArea);
        actAreaSP.setBounds(0, 0, 400, 250);
        ToolTipManager.sharedInstance().unregisterComponent(actArea);
        ToolTipManager.sharedInstance().unregisterComponent(actArea.getTableHeader());
        // items table
        itemsColNames.add("Name");
        itemsColNames.add("Att");
        itemsColNames.add("Perc");
        itemsModel = new DefaultTableModel(null, itemsColNames);
        items = new JTable(itemsModel);
        items.setAutoCreateRowSorter(true);
        items.getColumnModel().getColumn(0).setPreferredWidth(firstColumnWidth);
        items.getColumnModel().getColumn(1).setPreferredWidth(secondColumnWidth);
        items.getColumnModel().getColumn(2).setPreferredWidth(thirdColumnWidth);
        itemsSP = new JScrollPane();
        itemsSP.setViewportView(items);
        itemsSP.setBounds(0, 0, 400, 250);
        ToolTipManager.sharedInstance().unregisterComponent(items);
        ToolTipManager.sharedInstance().unregisterComponent(items.getTableHeader());
        // day plan table
        dayPlanColNames.add("Name");
        dayPlanColNames.add("Interval");
        dayPlanColNames.add("XY");
        dayPlanModel = new DefaultTableModel(null, dayPlanColNames);
        dayPlan = new JTable(dayPlanModel);
        dayPlan.getColumnModel().getColumn(0).setPreferredWidth(100);
        dayPlan.getColumnModel().getColumn(1).setPreferredWidth(125);
        dayPlan.getColumnModel().getColumn(2).setPreferredWidth(25);
        dayPlanSP = new JScrollPane();
        dayPlanSP.setViewportView(dayPlan);
        dayPlanSP.setBounds(0, 0, 400, 250);
        ToolTipManager.sharedInstance().unregisterComponent(dayPlan);
        ToolTipManager.sharedInstance().unregisterComponent(dayPlan.getTableHeader());

    }
    /**
     * First initialization of table models - initial values of process area, intention area, etc.
     */
    private void initializeTableModels() {
        Vector intAreaValues = transformIntentions(module.intentionArea);
        intAreaModel = new DefaultTableModel(intAreaValues, intAreaColNames);
        Vector actAreaValues = transformActions(module.perceptiveField.processArea);
        actAreaModel = new DefaultTableModel(actAreaValues, actAreaColNames);
        Vector itemsValues = transformItems(module.things.visibleItems);
        itemsModel = new DefaultTableModel(itemsValues, itemsColNames);
        module.log.info("initialized tables 1..3");
        if (module.basicIntentions == null)
            return;
        Vector dayPlanValues = transformDayPlan(module.basicIntentions, module.counter);
        dayPlanModel = new DefaultTableModel(dayPlanValues, dayPlanColNames);
        module.log.info("initialized table 4");
    }

    private boolean isItemPerceived(EItem item) {
        if (module.perceptiveField.perceivedItems.contains(item))
            return true;
        else
            return false;

    }
    /**
     * Transforms a list of intentions into a Vector which is then accepted by a constructor of JTable
     * or processed by an updateTables() method.
     * @param intentions - a list of intentions to be processed
     * @return Vector which stores "Name", "Activity", "AncestorAction" for each intention
     */
    private Vector transformIntentions(ArrayList<Intention> intentions) {
        Vector result = new Vector(3*intentions.size());
        int index = 0;
        for (Intention i : intentions) {
            result.add(index++, new String(i.getName()));
            result.add(index++, new String(Integer.toString(i.getActivity())));
            if (i.getAncestorAction() != null)
                result.add(index++, new String(i.getAncestorAction().name));
            else
                result.add(index++, "None");
        }
        return result;
    }

    /**
     * Transforms a list of actions into a Vector which is then accepted by a constructor of JTable
     * or processed by an updateTables() method.
     * @param actions - a list of actions to be processed
     * @return Vector which stores "Name", "Activity", "Attention" for each action
     */
    private Vector transformActions(ArrayList<Action> actions) {
        Vector result = new Vector(3*actions.size());
        int index = 0;
        for (Action a : actions) {
            result.add(index++, new String(a.name));
            result.add(index++, new String(Integer.toString(a.activity)));
            result.add(index++, new String(Integer.toString(a.attention)));
        }
        return result;
    }

    /**
     * Transforms a list of items into a Vector which is then accepted by a constructor of JTable
     * or processed by an updateTables() method.
     * @param items
     * @return Vector which stores "Name", "Attractivity", "Perceived" for each item
     */
    private Vector transformItems(HashMap<Long, EItem> items) {
        Vector result = new Vector(3*items.values().size());
        int index = 0;
        synchronized(items) {
            for (EItem a : items.values()) {
                result.add(index++, new String(a.name));
                result.add(index++, new String(Integer.toString(a.getAttractivity())));
                result.add(index++, isItemPerceived(a));
            }
        }
        return result;
    }
    /**
     * Transforms a list of intentions from the day plan into a Vector which is
     * then accepted by a constructor of JTable or processed by an updateTables() method.
     * @param intentions
     * @return Vector which stores "Name", "Interval", "" for each intention
     */
    private Vector transformDayPlan (ArrayList<Intention> plan, int counter) {
        Vector result = new Vector(3*plan.size());
        int index = 0, distance = GlobalParameters.LENGHT_OF_A_DAY, tmp;
        Interval toShow = null;
        for (Intention i : plan) {
            result.add(index++, new String(i.getName()));
            toShow = null;
            distance = GlobalParameters.LENGHT_OF_A_DAY;
            for (Interval intr : i.getActivationIntervals()) {
                if (intr.isInInterval(counter)) {
                    toShow = intr;
                    break;
                }
                tmp = (intr.getLeftSide() + intr.getRightSide())/2; // center of the interval
                tmp = Math.abs(tmp - counter); // distance from counter
                if (tmp < distance) {
                    distance = tmp;
                    toShow = intr;
                }
            }
            if (toShow != null)
                result.add(index++, new String(toShow.toString()));
            else
                result.add(index++, "None");
            result.add(index++, new String(Integer.toString(i.getActivity())));
        }
        return result;
    }
    /**
     * Updates values of changed cells - for each table it transforms actions/intentions/items
     * to a Vector and calls actualizeTable() to do the actualization.
     */
    private void updateTables() {
        Vector intAreaValues = transformIntentions(module.intentionArea);
        actualizeTable(intArea, intAreaValues);

        Vector actAreaValues = transformActions(module.perceptiveField.processArea);
        actualizeTable(actArea, actAreaValues);

        Vector itemsValues = transformItems(module.things.visibleItems);
        actualizeTable(items, itemsValues);

        if (module.basicIntentions == null)
            return;
        Vector dayPlanValues = transformDayPlan(module.basicIntentions, module.counter);
        actualizeTable(dayPlan, dayPlanValues);
    }

    /**
     * All tables has an attribute name, so we are now going to actualize each
     * table line by line. This method does it for one table.
     *
     * It first figures out which lines are obsolete, the lines that are still there
     * are just updated if necessary, new lines are added on the place of obsolete
     * records or at new lines. Obsolete records which were not replaced are erased.
     *
     * @param table
     * @param newValues
     */
    private void actualizeTable(JTable table, Vector newValues) {
        int rows = table.getRowCount();
        int [] subIndices = new int[rows];
        boolean substitute = false;
        int i, j;
        // I will precount which of the lines of the table I can replace with new records
        for (i = 0 ; i < rows; i++) {
            if (table.getValueAt(i, 0) == null) {
                subIndices[i] = 1;
                continue;
            }
            substitute = true;
            for (j = 0; j < (newValues.size()/3); j++) {
                if (newValues.get(j*3) == null)
                    continue;
                if (table.getValueAt(i, 0).equals(newValues.get(j*3))) { // it is there already - no need to remove it
                    subIndices[i] = 0;
                    if (!table.getValueAt(i, 1).equals(newValues.get(j*3 + 1)))
                        table.setValueAt(newValues.get(j*3 + 1), i, 1); // actualisation
                    if (!table.getValueAt(i, 1).equals(newValues.get(j*3 + 2)))
                        table.setValueAt(newValues.get(j*3 + 2), i, 2); // actualisation
                    substitute = false;
                    // remove the record which I already used
                    newValues.remove(j*3); // remove first
                    newValues.remove(j*3); // remove second
                    newValues.remove(j*3); // remove third part
                    break;
                }
            }
            if (substitute)
                 subIndices[i] = 1; // line must be removed/replaced
        }
        // now I will replace obsolete records with new ones (nulls if there is not any other)
        j = 0;
        for (i = 0 ; i < table.getRowCount(); i++) {
            if (subIndices[i] == 0) // up to date record - skip
                continue;
            if (newValues.isEmpty()) { // nothing new to add - set to null if necessary
                if (table.getValueAt(i, 0) != null)
                    table.setValueAt(null, i, 0);
                if (table.getValueAt(i, 1) != null)
                    table.setValueAt(null, i, 1);
                if (table.getValueAt(i, 2) != null)
                    table.setValueAt(null, i, 2);
            } else {
                // replace the content of the line by a new value
                table.setValueAt(newValues.get(0), i, 0);
                table.setValueAt(newValues.get(1), i, 1);
                table.setValueAt(newValues.get(2), i, 2);
                // remove the inserted line from newValues
                newValues.remove(0);
                newValues.remove(0);
                newValues.remove(0);
            }
        }
        // put in the rest of records which didn't fit in
        for (j = 0; j < (newValues.size()/3); j++) {
            ((DefaultTableModel)table.getModel()).addRow(newValues.subList(j*3, j*3 + 3).toArray());
        }
    }
}
