package decisionMakingSystem;

import utils.StringListWrapper;
import atomicActions.*;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Queue;
import java.util.logging.Logger;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import utils.TimeUtils;

/**
 * This loader takes care of loading tree plans from xml files.
 * you can obtain an example of XML file with a tree plan by calling createFirstXMLEntry()
 * and you can get a Loader.xml example by calling xmlLoaderEntry().
 * <br> <br>
 * So how it works? You call the method importIntentionForest. It loads a list of files specified in the Loader.xml.
 * Each file should represent one AND-OR-tree plan. So it loads those files one by one reading corresponding intentions from them
 * adding them to the basicIntentions which are then returned. 
 * 
 * One trick!!! as there are atomic actions and xml can't distinguish properly
 * their type (class) and returns only the type of ancestor class, it goes through those 
 * and recreates the objects -> atomic actions thus can't contain some settings. This proceeding
 * is necessary as well because an atomic action needs a link with environment as it 
 * will execute something there so it needs link to agent and decision making system
 * thus we use AtomicActionFactory, which creates AtomicAction object of proper type with all 
 * necessary parameters set (pointer to agent and parent action).
 * 
 * 
 * @author Ondrej
 */
class BasicIntentionLoader {
    /**
     * creates example of root intention to the file Intention.xml.
     * Intention has two actions as a possibilities to satisfy it.
     */
    public static void createFirstXMLEntry() {
        Intention intention = Intention.exampleOfIntention();
        Action action = Action.exampleAction();
        Action action2 = new Action();
        intention.setActions(new ArrayList<Action>());
        intention.getActions().add(action);
        intention.getActions().add(action2);
        // generate XML from that
        try {
            File file = new File("Intention.xml");
            FileOutputStream outputFile = new FileOutputStream(file);
            JAXBContext context = JAXBContext.newInstance(Intention.class);
            Marshaller m = context.createMarshaller();
            m.marshal(intention, outputFile);
        } catch (Exception e) {
            System.err.println("Error in creating intention entry! " + e);
        }
    }
    /** 
     * creates example of Loader.xml with two intentions to load:
     * <ul>
     * <li>Intention.xml
     * <li>Eating.xml
     * </ul>
     */
    public static void xmlLoaderEntry() {
        StringListWrapper list = new StringListWrapper();
        list.list.add("Intention.xml");
        list.list.add("Eating.xml");
        try {
            File file = new File("Loader.xml");
            FileOutputStream outputFile = new FileOutputStream(file);
            JAXBContext context = JAXBContext.newInstance(StringListWrapper.class);
            Marshaller m = context.createMarshaller();
            m.marshal(list, outputFile);
        } catch (Exception e) {
            System.err.println("Error in creating Loader Entry! " + e);
        }
    }
    
    /**
     * imports all root intentions specified in the Loader.xml
     * @param directory - directory, where are located Loader.xml and files specifying the intentions
     * @param module - decision making system module - needed to initialize atomic actions
     */
    public static ArrayList<Intention> importIntentionForest(String directory, DecisionModuleImpl module) {
        float time = System.currentTimeMillis();
        // result
        ArrayList<Intention> basicIntentions = new ArrayList<Intention> ();
        // first I load the names of used behaviors
        File file = new File(directory + "plans" + File.separator + "Loader.xml");
        StringListWrapper wrapper = null; // because of the annotation issues
        ArrayList<String> listOfFiles = new ArrayList<String>();
        try {
            FileInputStream reader = new FileInputStream(file);
            JAXBContext context = JAXBContext.newInstance(StringListWrapper.class);
            Unmarshaller u = context.createUnmarshaller();
            wrapper = (StringListWrapper) u.unmarshal(reader); 

        } catch (Exception e) {
                module.agent.getLog().severe("Error in loading Loader.xml\n" + e);
                System.err.println("Error in loading Loader.xml\n" + e);
        }
        // so now I have a list of files with root intentions
        listOfFiles = wrapper.list;
        module.agent.getLog().info("List of root intentions to load: " + listOfFiles);
        // then I load them one by one
        Intention tempIntention = null;
        for(String name : listOfFiles) {        
            try {
                file = new File(directory + "plans"+ File.separator + name);
                FileInputStream reader = new FileInputStream(file);
                JAXBContext context = JAXBContext.newInstance(Intention.class);
                Unmarshaller u = context.createUnmarshaller();
                tempIntention = (Intention) u.unmarshal(reader); 
                basicIntentions.add(tempIntention);
            } catch (Exception e) {
                module.agent.getLog().severe("Error in loading intention: " + name + "\n" + e);
            }
        }
        // then I iterate through all loaded intentions to get atomic actions and initialize them properly
        Iterator it = null;
        int i;
        for (Intention individual : basicIntentions) {
            // there I will need a list of all actions who has some atomic actions - to retype them
            ArrayList<Action> actionsWithAtomicActions = getActionsWithAtomicActions(individual, module.log);
            // substitute general atomic actions by objects corresponding to their types -> JAXB can't keep that
            retypeAtomicActions(actionsWithAtomicActions, module);
            module.agent.getLog().info("Basic intention: " + individual.getName() + " successfully loaded.");
        }

        expandIntentions(basicIntentions, module);

        time = (System.currentTimeMillis() - time);
        module.agent.getLog().info("Intentions loaded in: " + Math.round(time) + "ms");
        return basicIntentions;
    }

    /**
     * Method used to consolidate intention forest - to enable common subtrees
     * Intentions with extend flag true are not loaded and need to replaced with
     * another intention tree. Intention with extension flag set to true are then
     * deleted because they are not suppossed to be top-level goals.
     */
    private static void expandIntentions(ArrayList<Intention> list, DecisionModuleImpl module) {
        Queue<Intention> qi = new LinkedList<Intention>();
        Queue<Action> qa = new LinkedList<Action>();
        for (Intention i : list) {
            qi.add(i);
        }
        Intention intention;
        Action action;
        while (!qi.isEmpty()) {
            while (!qi.isEmpty()) {
                intention = qi.poll();
                if (intention.getActions() != null) {
                    for (Action act : intention.getActions()) {
                        qa.add(act);
                    }
                }
            }
            while (!qa.isEmpty()) {
                action = qa.poll();
                if (action.intentions != null) {
                    for (int i = 0; i < action.intentions.size(); i++) {
                        intention = action.intentions.get(i);
                        if (intention.extend) {
                            for (Intention sub : list) {
                                if (intention.getName().equals(sub.getName())) {
                                    Intention copy = sub.cloneBySerialize(module.agent, module);
                                    action.intentions.set(i, copy);
                                    intention = copy;
                                }
                            }
                        }
                        qi.add(intention);
                    }
                }
            }
        }

        for (Iterator<Intention> iter = list.iterator(); iter.hasNext();) {
            intention = iter.next();
            if (intention.extension) {
                iter.remove();
            }
        }

    }
    
    public static void main(String []args) {
        BasicIntentionLoader.xmlLoaderEntry();
        // createFirstXMLEntry();
        /*
        DecisionModuleImpl module = new DecisionModuleImpl();
        module.prepareLogic(new Agent(), "C:\\Program Files\\Pogamut 2\\PogamutPlatform\\projects\\EpisodicBot\\");
        for (Intention intention : module.basicIntentions) {
            System.out.println(intention);
            for (Action action: intention.getActions())
                System.out.println(action);
        }
         */
    }
    /**
     * goes through the hierarchy of one intention-action tree and returns a list of actions, which has some atomic actions
     * meanwhile sets as well the timeLimit in the steps of the logic according to the maxMinutes - maybe, let me think:) => yes
     * @param individual - intention to proceed
     * @return list of actions which has some atomic actions
     */
    protected static ArrayList<Action> getActionsWithAtomicActions(Intention individual) {
        ArrayList<Action> result = new ArrayList<Action>();
        if (individual != null && individual.getActions() != null)
            for (Action action : individual.getActions()) {
                action.timeLimit = TimeUtils.minutesToTicksOfLogic(action.realTimeLimit);
                if (action.atomicActions != null && action.atomicActions.size() > 0)
                    result.add(action);
                if (action.intentions != null && action.intentions.size() > 0)
                    for (Intention intention : action.intentions)
                        result.addAll(getActionsWithAtomicActions(intention));
            }
        return result;
    }
    /**
     * goes through the hierarchy of one intention-action tree and returns a list of actions, which has some atomic actions
     * meanwhile sets as well the timeLimit in the steps of the logic according to the maxMinutes - maybe, let me think:) => yes
     * @param individual - intention to proceed
     * @return list of actions which has some atomic actions
     */
    protected static ArrayList<Action> getActionsWithAtomicActions(Intention individual, Logger log) {
        ArrayList<Action> result = new ArrayList<Action>();
        if (individual != null && individual.getActions() != null)
            for (Action action : individual.getActions()) {
                action.timeLimit = TimeUtils.minutesToTicksOfLogic(action.realTimeLimit);
                log.fine("Time limit:" + action.timeLimit + " real time: " + action.realTimeLimit);
                if (action.atomicActions != null && action.atomicActions.size() > 0)
                    result.add(action);
                if (action.intentions != null && action.intentions.size() > 0)
                    for (Intention intention : action.intentions)
                        result.addAll(getActionsWithAtomicActions(intention, log));
            }
        return result;
    }

    private static void retypeAtomicActions(ArrayList<Action> actionsWithAtomicActions, DecisionModuleImpl module) {
            // iterate through actions with atomic actions
            AtomicAction act; // temp atomic action - I need to iterate this way as I will remove the action and add it again
            ArrayList<AtomicAction> actions = null;
            int i;
            for (Action action : actionsWithAtomicActions) {
                actions = action.atomicActions;
                // iterate through atomic actions
                for (i = 0; i < actions.size(); i++) {
                    act = actions.get(i);
                    int attractivity = act.attractivity; //for use in episodic memory only - decide when to forget node
                    actions.remove(i);
                    if (act == null || act.type == null)
                        continue;
                    else {
                        act = AtomicActionFactory.getInstace().getAction(act.type, action, module.agent);
                        act.attractivity = attractivity;
                        actions.add(i, act);
                    }
                }
            }
    }
}