package decisionMakingSystem;

import cz.cuni.amis.pogamut.base3d.worldview.object.Location;

import decisionMakingSystem.Affordance;
import decisionMakingSystem.EItem;
import decisionMakingSystem.GlobalParameters;
import decisionMakingSystem.ItemCathegory;
import java.io.Serializable;
import java.util.ArrayList;
/**
 * Item record serves to store item memory information about an item
 * it does not store anything but the location, affordances and cathegory of the original EItem
 * but it adds other variables - missed, found, seen, time which indicate
 * when was the last time the record was modified and how many times bot
 * found, missed (not found) and saw the thing on the location
 *
 * @author Ondrej
 */
public class ItemRecord implements Comparable, Serializable {
    /** location of the item */
    public Location location = null;
    /** list of affordances of the item - I am searching affordances */
    public ArrayList<Affordance> affordances = null;
    /** cathegory of the item - e.g. pickable */
    public ItemCathegory cathegory = null;
    /** how many times agent went to the place and haven't found it */
    private int missed = 0;
    /** how many times agent went to the place and found it  */
    private int found = 0;
    /** how many times agent saw a thing  */
    private int seen = 0;
    /** when was the last modification of the record - in the steps of the logic (lower = older) */
    private int time = 0;
    /** probability that the item will be on the indicated place */
    public double probability = 0;

    public ItemRecord(EItem eItem, int time) {
        switch (eItem.cathegory) {
            case PICKABLE:
                location = eItem.getLocation();
                break;
            case PLACE:
                location = eItem.getLocation();
                break;
            case PLAYER:
                location = eItem.getLocation();
                break;
            default:
                location = new Location(); // triple of zeros timto si docela koleduju o pruser
        }
        cathegory = eItem.cathegory;
        affordances = eItem.getAffordances();
        this.time = time;
    }

    @Override
    public String toString() {
        this.updateProbability();
        String result = "item cathegory: " + cathegory;
        result += " location: " + location + "\t" + " probability: " + probability;
        result += "\t f: " + found + " s: " + seen + " m: " + missed;
        result += "\t time: " + time;
        return result;
    }
    /**
     * increases seen and updates time
     * @param counter - number of ticks of logic when it is called
     */
    public void seen(int counter) {
        seen++;
        time = counter;
    }

    /**
     * increases found and updates time
     * @param counter - number of ticks of logic when it is called
     */
    public void found(int counter) {
        found++;
        time = counter;
    }

    /**
     * increases missed and updates time
     * @param counter - number of ticks of logic when it is called
     */
    public void missed(int counter) {
        missed++;
        time = counter;
    }

    /**
     * counts probability based on found, missed, seen
     */
    public void updateProbability() {
        double result = (2*found-missed) * 10 + seen;
        if (result > 100)
            result = 100;
        if (result < 0)
            result = 0;
        probability = result/100;
    }

    /**
     * counts probability based on found, missed, seen and the difference between current time and time
     * @param counter
     */
    public void updateProbability(int counter) {
        double result = (2*found-missed) * 10 + seen;
        result -= ((counter - time) / GlobalParameters.LENGHT_OF_A_DAY) * 10; // number of days he spotted it last time
        if (result > 100)
            result = 100;
        if (result < 0)
            result = 0;
        probability = result/100;
    }

    /**
     * counts probability based on found, missed, seen, the difference between current time and time
     * and the distance between the agent and thing
     * @param counter
     */
    public void updateProbability(int counter, Location location) {
        double result = (2*found-missed) * 10 + seen;
        result -= ((counter - time) / GlobalParameters.LENGHT_OF_A_DAY) * 10; // number of days he spotted it last time
        result -= this.location.getDistance(location) / 500; // what do I know;), but closer the better
        if (result > 100)
            result = 100;
        if (result < 0)
            result = 0;
        probability = result/100;

    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ItemRecord other = (ItemRecord) obj;
        if (other.location.equals(location) && this.affordances.equals(other.affordances)
            && other.cathegory.equals(this.cathegory))
            return true;
        return false;
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 61 * hash + (this.location != null ? this.location.hashCode() : 0);
        hash = 61 * hash + (this.affordances != null ? this.affordances.hashCode() : 0);
        hash = 61 * hash + (this.cathegory != null ? this.cathegory.hashCode() : 0);
        return hash;
    }

    /**
     * updates seen, missed, found so it will deteriorate slowly, like the probability won't rise over limits
     */
    public void dailyUpdate() {
        seen -= 10;
        missed -= 1;
        found -= 1;
        seen = (seen < 0 ? 0 : seen);
        found = (found < 0 ? 0 : found);
        missed = (missed < 0 ? 0 : missed);
    }

    @Override
    public int compareTo(Object o) {
        final int BEFORE = -1;
        final int EQUAL = 0;
        final int AFTER = 1;

        //this optimization is usually worthwhile, and can
        //always be added
        if ( this == o ) return EQUAL;

        ItemRecord aThat = (ItemRecord) o;
        //primitive numbers follow this form
        if (this.probability < aThat.probability) return BEFORE;
        if (this.probability > aThat.probability) return AFTER;

        return EQUAL;
    }

  boolean obsolete(int counter) {
    if (probability < 0.05) {
      return true;
    }
    if (counter - time > GlobalParameters.LENGHT_OF_A_DAY) {
      return true;
    }
    return false;
  }

}